char rsh_c[]="$Header: d:/wilhelm/winrsh/RCS/rsh.c%v 1.6 1994/11/07 04:40:07 wcheung Exp wcheung $";
/*************************************************************************
 *
 * rsh - Remote Shell Module
 *
 * Copyright(C) 1994 William K. W. Cheung
 * All Rights Reserved
 *
 * This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * WinRSH/WinRSH32 - Remote Shell for Windows
 * Author: William K. W. Cheung (wcheung@ee.ubc.ca)
 * Date:   June 30, 1994
 *
 ************************************************************************/
#include <winsock.h>
#pragma hdrstop
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "rsh.h"
#include "wprintf.h"
#include "err.h"
#include "msg.h"

#define RSHBUFSIZ		256

extern void LogOutput(char *szBuf);

RSHSTRUCT       rshStruct;
static BOOL     fRshOpenState=FALSE;        // Ensure there is only one WSAStartup() call
static char     cmdbuf[MAXGETHOSTSTRUCT];
static int      cmdbuflen;
static SOCKET   s=INVALID_SOCKET;
static short    sPort, sProto;
static HANDLE   hAsyncGetHost=0;
static SOCKADDR_IN      saddr, saddrLoc;

static void MakeCmdPacket(char *name, char *pass, char *cmd);
static void RshPrintErr(int nErrCode, char *szMsg);

/****************************************************************
 *
 * RshOpen() - Start initializing Winsock and set the protocol
 *
 * This function initializes Winsock interface and sets the
 * remote communication protocol, REXEC or RSH.
 *
 * Parameters:
 *     nProtocol - 0 = REXEC
 *                 1 = RSH
 *
 * Returns:
 *	    TRUE if operation is successful
 *     FALSE if operation fails
 *
 * Side-Effects:
 *     If operation is successful,
 *         fRshOpenState = TRUE
 *         sPort = port number for the socket to be open
 *                 (in network byte order)
 *
 * Notes:
 *     This function does not creat any sockets.
 *
 ***************************************************************/
BOOL RshOpen(int nProtocol)
{
	WSADATA         wsaData;
	int             err;
	LPPROTOENT  lpProto;
	LPSERVENT   lpServ;

	if (fRshOpenState)
	{
		if (nProtocol == 0)
			lpServ = getservbyname("exec", NULL);
		else
			lpServ = getservbyname("shell", NULL);
		if (!lpServ)
		{
			// RshPrintErr(WSAGetLastError(), "Cannot obtain specified service failed.\n");
			sPort = (nProtocol == 0)? htons(IPPORT_EXECSERVER): htons(IPPORT_CMDSERVER);
		}
		else
			sPort = lpServ->s_port;

		return TRUE;
	}
	err = WSAStartup( 0x0101, &wsaData );
	if (err != 0)
	{
		ErrPrintf(ERR_WARN, "Unable to start Window Socket.");
		return FALSE;
	}

	if ( LOBYTE( wsaData.wVersion ) != 1 ||
		 HIBYTE( wsaData.wVersion ) != 1 )
	{
		WSACleanup();
		ErrPrintf(ERR_WARN, "Winsock Socket Version %d.%d", HIBYTE(wsaData.wVersion) & 0xFF,
				  LOBYTE(wsaData.wVersion) & 0xFF );
		return FALSE;
	}

	lpProto = getprotobyname("tcp");
	if (!lpProto)
	{
		// RshPrintErr(WSAGetLastError(), "Cannot obtain the protocol number (Using default).\n");
		sProto = IPPROTO_TCP;
	}
	else
		sProto = lpProto->p_proto;

	if (nProtocol == 0)
		lpServ = getservbyname("exec", NULL);
	else
		lpServ = getservbyname("shell", NULL);
	if (!lpServ)
	{
		// RshPrintErr(WSAGetLastError(), "Cannot obtain specified service (Using default).\n");
		sPort = (nProtocol == 0)? htons(IPPORT_EXECSERVER): htons(IPPORT_CMDSERVER);
	}
	else
		sPort = lpServ->s_port;

	// IFD("Service port number for EXEC is %hd\n", ntohs(serv->s_port));

	fRshOpenState = TRUE;

	return(TRUE);
}

/****************************************************************
 *
 * RshClose() - Close Winsock Interface
 *
 * This function terminates the communication between WinRSH
 * and Winsock.
 *
 ***************************************************************/
void RshClose(void)
{
	if (fRshOpenState)
	{
		WSACleanup();
		fRshOpenState = FALSE;
	}
}

/****************************************************************
 *
 * RshGetHost() - Obtains the mapping between a host name and
 *                its internet address
 *
 * This function tries to convert a host name into its internet
 * address equivalent.  If the supplied address is already an
 * internal address, nothing is done.  However, if the supplied
 * address is a host name, the Windows asynchronous version of
 * gethostbyname() is called.
 *
 * Side Effects:
 *    Upon exit, two things will happen:
 *        1) hAsyncGetHost is set if the input is a host name;
 *           otherwise, hAsyncGetHost = 0.
 *        2) A message "WM_RSHGETHOST" is posted in either cases.
 *           An application should handle this event using
 *           RshConnect().
 *
 ***************************************************************/
BOOL RshGetHost(HWND hWnd, BOOL bConnected)
{
	if (bConnected)
	{
		ErrPrintf(ERR_WARN, "A connection is already in progress.\rPlease try again.");
		return FALSE;
	}

	if (rshStruct.hostname[0] == '\0')
		return FALSE;

	memset(&saddr, 0, sizeof(SOCKADDR_IN));
	MsgPrintf("Connecting to %s...", rshStruct.hostname);
	memset((void *)&saddr, 0, sizeof(saddr));
	saddr.sin_addr.s_addr = inet_addr(rshStruct.hostname);
	hAsyncGetHost = 0;
	if (saddr.sin_addr.s_addr == (u_long)INADDR_NONE)
	{
		hAsyncGetHost = WSAAsyncGetHostByName(hWnd, WM_RSHGETHOST, rshStruct.hostname,
											  cmdbuf, MAXGETHOSTSTRUCT);
		if (hAsyncGetHost == 0)
			ErrPrintf(ERR_WARN, "Unable to translate the host name.");
		else
			return TRUE;
	}
	else
	{
		PostMessage(hWnd, WM_RSHGETHOST, 0, 0L);
		return TRUE;
	}

	return FALSE;
}

/****************************************************************
 *
 * RshConnect() - Creates sockets and tries to connect to remote
 *                host.
 *
 * This function creates a socket using port number from "sPort"
 * and tries to establish a connection to the remote host.
 *
 * Notes:
 * 1) This function should be invoked only upon receiving
 *    a WM_RSHGETHOST message.
 * 2) Since there is no "rresport" in the Winsock specification,
 *    we need to write one ourselves.  The idea is to bind a
 *    socket to one of the reserved ports between 512 and 1023.
 *    This is required in the RSH protocol.  The algorithm is
 *    very simple: try every port from 1023 down to 512 and
 *    check if binding is successful.  If yes, stop; otherwise
 *    repeat the process.
 *
 ***************************************************************/
int RshConnect(HWND hWnd, WPARAM wParam, LPARAM lParam)
{
	int                     nRc;
	LPHOSTENT       lpHost;
	u_short         usResvPort;
	char                    szMsg[256];

	if (hAsyncGetHost)      // inet_addr was successful
	{
		if ((HANDLE)wParam != hAsyncGetHost)
		{
			ErrPrintf(ERR_WARN, "Host '%s' cannot be found.\n", rshStruct.hostname);
			goto cleanup;
		}
		if (WSAGETASYNCERROR(lParam) != 0)
		{
			sprintf(szMsg, "Host '%s' cannot be found.\n", rshStruct.hostname);
			RshPrintErr(WSAGetLastError(), szMsg);
			goto cleanup;
		}
		hAsyncGetHost = 0;

		lpHost = (LPHOSTENT)cmdbuf;
		memcpy(&saddr.sin_addr, lpHost->h_addr, lpHost->h_length);
	}

	s = socket(AF_INET, SOCK_STREAM, (int)sProto);
	if (s == INVALID_SOCKET)
	{
		RshPrintErr(WSAGetLastError(), "Socket creation error:\n");
		goto cleanup;
	}

	saddrLoc.sin_family= AF_INET;
	saddrLoc.sin_addr.s_addr = 0;
	if (sPort == htons(IPPORT_CMDSERVER))
	{
		/*
		 *  Execute "rresport" as in Unix:
		 *  Bind a reserved port to the socket using RSH protocol
		 */
		for(usResvPort=IPPORT_RESERVED-1; usResvPort >= IPPORT_RESERVED/2;
			usResvPort--)
		{
			saddrLoc.sin_port = htons(usResvPort);
			if (bind(s, (SOCKADDR *)&saddrLoc, sizeof(saddrLoc)) == SOCKET_ERROR)
			{
				nRc = WSAGetLastError();
				if (nRc != WSAEADDRINUSE)
				{
					RshPrintErr(nRc, "Socket binding error:\n");
					goto cleanup;
				}
			}
			else
				break;
		}
	}

	saddr.sin_family= AF_INET;
	saddr.sin_port = sPort;

	WSAAsyncSelect(s, hWnd, WM_RSH, FD_READ|FD_CONNECT|FD_CLOSE);

	if (connect(s, (SOCKADDR *)&saddr, sizeof(SOCKADDR)) != 0)
	{
		nRc = WSAGetLastError();
		if (nRc != WSAEWOULDBLOCK)
		{
			RshPrintErr(nRc, "Connection failed:\n");
			goto cleanup;
		}
	}

	return TRUE;

cleanup:
	SendMessage(hWnd, WM_RSHENDSESSION, 0, 0L);
	return FALSE;
}

/****************************************************************
 *
 * RshDisconnect() - Disconnects (closes) a socket.
 *
 * This function disconnects or closes a scoket that is already
 * open.
 *
 * Notes:
 * Some Winsock implemention does not close properly unless
 * the socket "lingering" function is disabled.
 *
 ***************************************************************/
void RshDisconnect(void)
{
	LINGER  linger;

	if (hAsyncGetHost)
	{
		if (WSACancelAsyncRequest(hAsyncGetHost) == SOCKET_ERROR)
			RshPrintErr(WSAGetLastError(), "Cannot cancel querying host request.\n");
		hAsyncGetHost = 0;
	}

	if (s != INVALID_SOCKET)
	{
		shutdown(s, 2);
		// The following will force the remote connect to close
		linger.l_onoff = 1;
		linger.l_linger = 0;
		(void)setsockopt(s, SOL_SOCKET, SO_LINGER, (char FAR *)&linger,
							  sizeof(linger));
		closesocket(s);
		s = INVALID_SOCKET;
	}
}

/****************************************************************
 *
 * RshRecv() - Receive data from the network
 *
 * This function reads data from the network and displays an
 * error message if something is wrong with the connection.
 *
 * Returns:
 *     Number of bytes read - The result is put in lpBuf
 *     -1 if an error occurred
 *
 ***************************************************************/
int RshRecv(LPSTR lpBuf, int nLen)
{
	int     nOLen;
	int     nRc;

	nOLen = recv(s, (char FAR *)lpBuf, nLen, 0);
	if (nOLen < 0)
	{
		nRc = WSAGetLastError();
		if (nRc != WSAEWOULDBLOCK)
			RshPrintErr(nRc, "Receive failed:\n");
	}

	return(nOLen);
}

/****************************************************************
 *
 * RshSend() - Send data to the network
 *
 * This function writes data to the network and displays
 * any error messages
 *
 * Returns:
 *     Number of bytes send
 *     -1 if an error occurred
 *
 ***************************************************************/
int RshSend(LPSTR lpBuf, int nLen)
{
	int     nOLen;

	nOLen = send(s, (char FAR *)lpBuf, nLen, 0);
	if (nOLen < 0)
		RshPrintErr(WSAGetLastError(), "Send failed:\n");

	return(nOLen);
}

/****************************************************************
 *
 * RshRecvAsync() - Receive data from the network asynchronously
 *
 * This function reads data from the network and dumps it to
 * the message window and optionally to a log file.
 *
 ***************************************************************/
BOOL RshRecvAsync(HWND hWnd)
{
	char    	szBuf[RSHBUFSIZ+1];
	int		nLen;

	if (s == INVALID_SOCKET)
		return FALSE;

	nLen = RshRecv(szBuf, RSHBUFSIZ);
	if (nLen > 0)
	{
		szBuf[nLen] = '\0';
		wputs(szBuf);
		LogOutput(szBuf);
		MsgPrintf("Receiving...");
		return TRUE;
	}

cleanup:
	SendMessage(hWnd, WM_RSHENDSESSION, 0, 0L);
	return FALSE;
}

/****************************************************************
 *
 * DoRsh() - Handles all network generated events
 *
 * This function handles all network generated events including
 * FD_READ, FD_CONNECT, and FD_CLOSE.  After a connection is
 * established, an RSH or REXEC initialization packet is
 * transmitted.  FD_READ can be in one of the situations:
 *     1) Waiting for a one-byte response from the network
 *        to indicate whether authenication is successful (0)
 *        or failed (1).
 *     2) If successful, we should enter the data transfer phase.
 *     3) If failed, the received data indicates the cause of the
 *        error, and the result is displayed in a message box.
 *
 * Notes:
 * We should not receive the data immediately upon receiving the
 * FD_READ message in the data transfer phase but set a timer
 * to do so.  The reason is that if the remote side is pumping
 * a huge amount of data to the PC and the receive routine can
 * only receive a fixed amount of data at a time from the receive
 * buffer, Winsock will immediately generates another FD_READ
 * event which is processed immmediately.  Hence, the whole
 * Windows event queue is flooded with FD_READ events, and this
 * will make the application appears to be "hanged".  A 100ms
 * delay seems to be a good value to use.
 *
 ***************************************************************/
#pragma argsused
int DoRsh(HWND hWnd, WPARAM wParam, LPARAM lParam)
{
	static int      nState = 0;
	char     		 szBuf[RSHBUFSIZ+1];
	int             nLen;

	if (s == INVALID_SOCKET)
		return FALSE;

	switch(WSAGETSELECTEVENT(lParam))
	{
		case FD_READ:
			switch (nState)
			{
				case 1:         // Waiting for peer connection message
					nLen = RshRecv((LPSTR)szBuf, 1);
					if ((nLen > 0) && (szBuf[0] == 0))
					{
						MsgPrintf("Connected to %s.  Please wait for incoming data...", rshStruct.hostname);
						nState = 2;
#if 0
						/*
						 * Since we do not transmit anything, we should
						 * perform a "half-close" for the socket (The
						 * send part is closed.)  However, some Winsock
						 * implementation does not handle this properly, so
						 * this part is ignored.
						 */
						shutdown(s, 1);
#endif
					}
					else
						nState = 3;
					break;

				case 2:         // Data Transfer Phase
					KillTimer(hWnd, RSHRECV);
					SetTimer(hWnd, RSHRECV, 100, 0L);
					break;

				case 3:        // Connection failed due to invalid input
					nLen = RshRecv((LPSTR)szBuf, RSHBUFSIZ);
					if (nLen > 0)
					{
						szBuf[nLen] = '\0';
						ErrPrintf(ERR_WARN, "Error from network:\n%s", szBuf);
						goto cleanup;
					}
					break;

				default:
					break;
			}
			break;

		case FD_CONNECT:
			if (WSAGETSELECTERROR(lParam) != 0)
			{
				RshPrintErr(WSAGETSELECTERROR(lParam), "Connection error.\n");
				goto cleanup;
			}
			nState = 1;
			if (sPort == htons(IPPORT_CMDSERVER))
				MakeCmdPacket(rshStruct.user, rshStruct.user, rshStruct.cmd);
			else
				MakeCmdPacket(rshStruct.user, rshStruct.passwd, rshStruct.cmd);
			nLen = RshSend((LPSTR)cmdbuf, cmdbuflen);
			if (nLen < 0)
				goto cleanup;
			break;

		case FD_CLOSE:
			nState = 0;
			if (WSAGETSELECTERROR(lParam) != 0)
				RshPrintErr(WSAGETSELECTERROR(lParam), "Closing error.\n");
			while(RshRecvAsync(hWnd));
			wputs("\n");	// Flush the buffer
			break;

		default:
			break;
	}
	return FALSE;

cleanup:
	SendMessage(hWnd, WM_RSHENDSESSION, 0, 0L);
	return FALSE;
}

/****************************************************************
 *
 * MakeCmdPacket() - Creates a command packet
 *
 * This function creates a command packet according to the RSHd
 * and the REXECd requirements:
 *
 * For RSHd:
 * \0remuser\0locuser\0command\0
 *
 * For REXECd:
 * \0remuser\0password\0command\0
 *
 * The command is also interpreted: If '$i' is found, it is
 * replaced by the local host name.  If '$x' where x <> 'i' is
 * found, it is replaced by x.  Otherwise, the command is
 * copied as is into the transmit buffer.
 *
 ***************************************************************/
static void MakeCmdPacket(char *name, char *pass, char *cmd)
{
	char *p;
	char szHost[40];
	BOOL fEsc;

	if (strstr(cmd, "$i"))
	{
		if (gethostname((char FAR *)szHost, 40) != 0)
		{
			ErrPrintf(ERR_WARN, "Cannot find local host name. Local host name is set to localhost.");
			strcpy(szHost, "localhost");
		}
		else
			wprintf("My host name is %s\n", szHost);
	}
	p = cmdbuf;
	*p++ = 0;
	strcpy( p, name );
	p = strchr( p, 0 );
	strcpy( ++p, pass );
	p = strchr( p, 0 );

	for( fEsc = FALSE, ++p; *cmd != '\0'; cmd++)
	{
		if (fEsc)
		{
			if (*cmd == 'i')
			{
				strcpy(p, szHost);
				p += strlen(szHost);
				continue;
			}
			fEsc = FALSE;
		}
		else if (*cmd == '$')
		{
			fEsc = TRUE;
			continue;
		}
		*p++ = *cmd;
	}
	*p = '\0';

	cmdbuflen = (int)(p - cmdbuf) + 1;
}

/****************************************************************
 *
 * RshPrintErr - Print human understandable error messages
 *
 * This function displays human understandable error messages
 * from WSAGetLastError() with additional comments about the
 * error.
 *
 * Parameters:
 *    nErrCode - should be a value from WSAGetLastError().
 *    szMsg    - Any error message or "" if no additional
 *               comment is needed.
 ***************************************************************/
void RshPrintErr(int nErrCode, char *szMsg)
{
	switch(nErrCode)
	{
		case WSAENETDOWN:
			ErrPrintf(ERR_WARN, "%sThe network subsystem has failed.", szMsg);
			break;
		case WSAEINTR:
			ErrPrintf(ERR_WARN, "%sA blocking call was cancelled.  This can be caused by\n1) a short response time, or\n2) User interrupts the process.", szMsg);
			break;
		case WSAEINPROGRESS:
			ErrPrintf(ERR_WARN, "%sA blocking call is in progress.", szMsg);
			break;
		case WSAENOBUFS:
			ErrPrintf(ERR_WARN, "%sNo buffer space is available.", szMsg);
			break;
		case WSAENOTSOCK:
			ErrPrintf(ERR_WARN, "%sInvalid socket descriptor.", szMsg);
			break;
		case WSAEADDRINUSE:
			ErrPrintf(ERR_WARN, "%sThe specified address is already in use.", szMsg);
			break;
		case WSAEADDRNOTAVAIL:
			ErrPrintf(ERR_WARN, "%sThe specified address is not available\nfrom the local machine.", szMsg);
			break;
		case WSAECONNREFUSED:
			ErrPrintf(ERR_WARN, "%sThe connection attempt was refused.", szMsg);
			break;
		case WSAEINVAL:
			ErrPrintf(ERR_WARN, "%sThe socket is not already bound to an address.", szMsg);
			break;
		case WSAEISCONN:
			ErrPrintf(ERR_WARN, "%sThe socket is already connected.", szMsg);
			break;
		case WSAEMFILE:
			ErrPrintf(ERR_WARN, "%sThe maximum number of sockets has exceeded.", szMsg);
			break;
		case WSAENETUNREACH:
			ErrPrintf(ERR_WARN, "%sNetwork cannot be reached from this host at this time.", szMsg);
			break;
		case WSAETIMEDOUT:
			ErrPrintf(ERR_WARN, "%sAttempt to connect timed out without establishing a connection.", szMsg);
			break;
		case WSAENOTCONN:
			ErrPrintf(ERR_WARN, "%sThe socket is not connected.", szMsg);
			break;
		case WSAESHUTDOWN:
			ErrPrintf(ERR_WARN, "%sThe socket has been shut down.", szMsg);
			break;
		case WSAECONNABORTED:
			ErrPrintf(ERR_WARN, "%sThe virtual circuit was aborted due to timeout or other failure.", szMsg);
			break;
		case WSAECONNRESET:
			ErrPrintf(ERR_WARN, "%sThe virtual circuit was reset by the remote side.", szMsg);
			break;
		case WSAEACCES:
			ErrPrintf(ERR_WARN, "%sThe requested address is a broadcast address.", szMsg);
			break;
		case WSAENETRESET:
			ErrPrintf(ERR_WARN, "%sThe connection must be reset.", szMsg);
			break;
		case WSAHOST_NOT_FOUND:
			ErrPrintf(ERR_WARN, "%sAuthoritative Answer Host is not found.", szMsg);
			break;
		default:
			ErrPrintf(ERR_WARN, "%sError number = %d", szMsg, nErrCode);
			break;
	}
}

