/* IP-related user commands */
#include <stdio.h>
#include "global.h"
#include "mbuf.h"
#include "internet.h"
#include "timer.h"
#include "netuser.h"
#include "iface.h"
#include "ip.h"
#include "cmdparse.h"
#include "commands.h"
#include "socket.h"

/* Dump a routing table entry */
static void near
dumproute(struct route *rp)
{
	tprintf("%-17.17s%-4u",
		rp->target ? inet_ntoa(rp->target) : "default",
		rp->bits);
	tprintf("%-7.7s%-17.17s",
		rp->iface->name,
		rp->gateway ? inet_ntoa(rp->gateway) : "");
	tprintf("%-8lu%c %-7lu%lu\n",
		rp->metric,
		(rp->flags & RTPRIVATE) ? 'P' : ' ',
		read_timer(&rp->timer)/1000L,
		rp->uses);
	return;
}

/* ------------------------ IP subcmds ------------------------ */

static int
doipaddr(int argc,char **argv,void *p)
{
	int32 n;

	if(argc < 2) {
		tprintf("%s\n",inet_ntoa(Ip_addr));
	} else if((n = resolve(argv[1])) == 0){
		tprintf(Badhost,argv[1]);
		return -1;
	} else {
		Ip_addr = n;
	}
	return 0;
}

static int
dortimer(int argc,char **argv,void *p)
{
	return setlong(&ipReasmTimeout,"IP reasm timeout (sec)",argc,argv);
}

static int
doipstat(int argc,char **argv,void *p)
{
	int i;

	for(i = 1; i <= NUMIPMIB; i++){
		tprintf("(%2u)ip%-18s%10lu",
			i,Ip_mib[i].name,Ip_mib[i].value.integer);
		tputs((i % 2) ? "     " : "\n");
	}
	if((i % 2) == 0)
		tputs("\n");

	if(Reasmq != NULLREASM) {
		struct reasm *rp;
		struct frag *fp;

		tputs("Reassembly fragments:\n");
		for(rp = Reasmq; rp != NULLREASM; rp = rp->next){
			tprintf("src %s",inet_ntoa(rp->source));
			tprintf(" dest %s",inet_ntoa(rp->dest));
			tprintf(" id %u pctl %u time %lu len %u\n",
				rp->id,uchar(rp->protocol),read_timer(&rp->timer),rp->length);
			for(fp = rp->fraglist; fp != NULLFRAG; fp = fp->next)
				tprintf(" offset %u last %u\n",fp->offset,fp->last);
		}
	}
	return 0;
}

static int
dottl(int argc,char **argv,void *p)
{
	return setlong(&ipDefaultTTL,"IP TTL",argc,argv);
}

int
doip(int argc,char **argv,void *p)
{
	static struct cmds Ipcmds[] = {
		"address",	doipaddr,	0,	0, NULLCHAR,
		"rtimer",	dortimer,	0,	0, NULLCHAR,
		"status",	doipstat,	0,	0, NULLCHAR,
		"ttl",		dottl,		0,	0, NULLCHAR,
		NULLCHAR,
	};
	return subcmd(Ipcmds,argc,argv,p);
}


/* --------------------------- route subcmds ------------------------ */

/* Add an entry to the routing table
 * E.g., "add 1.2.3.4 ax0 5.6.7.8 3"
 */
static int
doadd(int argc,char **argv,void *p)
{
	struct iface *ifp;
	int32 dest = 0, gateway = 0;
	unsigned bits = 0;

	int32 metric = (argc > 4) ? atol(argv[4]) : 1;
	char private = (strncmp(argv[0],"addp",4) == 0) ? 1 : 0;

	if((ifp = if_lookup(argv[2])) == NULLIF){
		tprintf(Badif,argv[2]);
		return 1;
	}
	if(strcmp(argv[1],"default") != 0) {
		char *bitp;

		/* If IP address is followed by an optional slash and
		 * a length field, (e.g., 128.96/16) get it;
		 * otherwise assume a full 32-bit address
		 */
		if((bitp = strchr(argv[1],'/')) != NULLCHAR){
			/* Terminate address token for resolve() call */
			*bitp++ = '\0';
			bits = atoi(bitp);
		} else {
			bits = 32;
		}
		if((dest = resolve(argv[1])) == 0){
			tprintf(Badhost,argv[1]);
			return 1;
		}
	}
	if(argc > 3) {
		if((gateway = resolve(argv[3])) == 0){
			tprintf(Badhost,argv[3]);
			return 1;
		}
	}
	if(rt_add(dest,bits,gateway,ifp,metric,0,private) == NULLROUTE)
		tputs("Can't add route\n");
	return 0;
}

/* Drop an entry from the routing table
 * E.g., "drop 128.96/16
 */
static int
dodrop(int argc,char **argv,void *p)
{
	unsigned bits = 0;
	int32 dest = 0;

	if(strcmp(argv[1],"default") != 0) {
		char *bitp;

		/* If IP address is followed by an optional slash and length field,
		 * (e.g., 128.96/16) get it; otherwise assume a full 32-bit address
		 */
		if((bitp = strchr(argv[1],'/')) != NULLCHAR){
			*bitp++ = '\0';
			bits = atoi(bitp);
		} else {
			bits = 32;
		}
		if((dest = resolve(argv[1])) == 0) {
			tprintf(Badhost,argv[1]);
			return 1;
		}
	}
	return rt_drop(dest,bits);
}

/* Force a timeout on all temporary routes */
static int
doflush(int argc,char **argv,void *p)
{
	struct route *rp, *rptmp;

	if(R_default.timer.state == TIMER_RUN)
		rt_drop(0,0);	/* Drop default route */

	for(rp = Routes; rp != NULLROUTE; rp = rptmp) {
		rptmp = rp->next;
		if(rp->timer.state == TIMER_RUN)
			rt_drop(rp->target,rp->bits);
	}
	return 0;
}

static int
dolook(int argc,char **argv,void *p)
{
	struct route *rp;
	int32 addr;

	if((addr = resolve(argv[1])) == 0) {
		tprintf(Badhost,argv[1]);
		return 1;
	}
	if((rp = rt_lookup(addr)) == NULLROUTE){
		tprintf("Host %s (%s) unreachable\n",argv[1],inet_ntoa(addr));
		return 1;
	}
	dumproute(rp);
	return 0;
}

/* Display and/or manipulate routing table */
int
doroute(int argc,char **argv,void *p)
{
	struct route *rp;

	struct cmds Rtcmds[] = {
		"add",		doadd,		0,	3,
		"route add <dest addr>[/<bits>] <iface> [gateway] [metric]",

		"addprivate",	doadd,		0,	3,
		"route addprivate <dest addr>[/<bits>] <iface> [gateway] [metric]",

		"drop",		dodrop,		0,	2,
		"route drop <dest addr>[/<bits>]",

		"flush",	doflush,	0,	0,
		NULLCHAR,

		"lookup",	dolook,		0,	2,
		"route lookup <dest addr>",

		NULLCHAR,
	};

	if(argc >= 2)
		return subcmd(Rtcmds,argc,argv,p);

	/* Dump IP routing table
	 * Dest            Len Interface    Gateway          Use
	 * 192.001.002.003 32  sl0          192.002.003.004  0
	 */
	tputs("Dest             Len Iface  Gateway          "
		  "Metric  P Timer  Use\n");

	for(rp = Routes; rp != NULLROUTE; rp = rp->next)
		dumproute(rp);

	if(R_default.iface != NULLIF)
		dumproute(&R_default);

	return 0;
}

