/* Generic serial line interface routines
 * Copyright 1992 Phil Karn, KA9Q
 */
#include <stdio.h>
#include <ctype.h>		/* used for 'tolower' */
#include "global.h"
#include "config.h"
#ifdef ASY
#include "iface.h"
#include "pktdrvr.h"
#include "netuser.h"
#include "asy.h"
#include "n8250.h"
#include "ax25.h"
#include "kiss.h"
#include "slip.h"
#include "nrs.h"
#include "proc.h"
#include "commands.h"

/* Attach a serial interface to the system
 * argv[0]: hardware type, must be "asy"
 * argv[1]: I/O address, e.g., "0x3f8"
 * argv[2]: vector, e.g., "4"
 * argv[3]: mode, may be:
 *	    "slip" (point-to-point SLIP)
 *	    "ax25" (AX.25 frame format in SLIP for raw TNC)
 *	    "nrs" (NET/ROM format serial protocol)
 * argv[4]: interface label, e.g., "sl0"
 * argv[5]: receiver ring buffer size in bytes
 * argv[6]: maximum transmission unit, bytes
 * argv[7]: interface speed, e.g, "9600"
 * argv[8]: optional flags,
 *		"c" for cts flow control
 *		"r" for rlsd (cd) detection
 *
 */
int
asy_attach(int argc,char **argv,void *p)
{
	struct iface *if_asy;
	int cts = 0, dev, rlsd = 0, trigchar = -1, xdev;
	char *ifn;

#ifdef AX25
	if(*Mycall == '\0') {
		tputs(Nomycall);
		return -1;
	}
#endif

	if(if_lookup(argv[4]) != NULLIF) {
		tprintf(Ifexist,argv[4]);
		return -1;
	}

	/* Find unused asy control block */
	for(dev = 0; dev < ASY_MAX; dev++) {
		if(Asy[dev].iface == NULLIF)
			break;
	}
	if(dev >= ASY_MAX){
		tprintf("Max %d async controller\n",ASY_MAX);
		return -1;
	}
	for(xdev = 0; xdev < SLIP_MAX; xdev++) {
		if(Slip[xdev].iface == NULLIF)
			break;
	}
	if(xdev >= SLIP_MAX) {
		tprintf("Max %d devices\n",SLIP_MAX);
		return -1;
	}

	/* Create interface structure and fill in details */
	if_asy = mxallocw(sizeof(struct iface));
	if_asy->addr = Ip_addr;
	if_asy->name = strxdup(argv[4]);
	ifn = if_name(if_asy," rx");
	if_asy->mtu = atoi(argv[6]);
	if_asy->dev = dev;
	if_asy->stop = asy_stop;

	if(argc > 8){
		if(strchr(argv[8],'c') != NULLCHAR)
			cts = 1;
		if(strchr(argv[8],'r') != NULLCHAR)
			rlsd = 1;
	}
    switch(*argv[3]) {
#ifdef	SLIP
	case 's':
		setencap(if_asy,"SLIP");
		if_asy->ioctl = asy_ioctl;
		if_asy->raw = slip_raw;
		if_asy->xdev = xdev;

		Slip[xdev].iface = if_asy;
		Slip[xdev].send = asy_send;
		Slip[xdev].get = get_asy;
		Slip[xdev].type = CL_SERIAL_LINE;
		trigchar = FR_END;
		if_asy->proc = newproc(ifn,256,asy_rx,xdev,NULL,NULL,0);
		break;
#endif
#ifdef	AX25
	case 'a':
		setencap(if_asy,"AX25");
		if_asy->ioctl = kiss_ioctl;
		if_asy->raw = kiss_raw;
		if_asy->xdev = xdev;
		if_asy->hwaddr = strxdup(Mycall);
		init_maxheard(if_asy);
		init_flags(if_asy);

		Slip[xdev].iface = if_asy;
		Slip[xdev].kiss[if_asy->port] = if_asy;		/* G1EMM */
		Slip[xdev].send = asy_send;
		Slip[xdev].get = get_asy;
		Slip[xdev].type = CL_KISS;
		trigchar = FR_END;
		if_asy->proc = newproc(ifn,256,asy_rx,xdev,NULL,NULL,0);
		break;
#endif
#ifdef	NRS
	case 'n':
		setencap(if_asy,"AX25");
		if_asy->ioctl = asy_ioctl;
		if_asy->raw = nrs_raw;
		if_asy->xdev = xdev;

		if_asy->hwaddr = strxdup(Mycall);
		init_maxheard(if_asy);
		init_flags(if_asy);

		Nrs[xdev].iface = if_asy;
		Nrs[xdev].send = asy_send;
		Nrs[xdev].get = get_asy;
		trigchar = ETX;
		if_asy->proc = newproc(ifn,256,nrs_recv,xdev,NULL,NULL,0);
		break;
#endif
	default:
		tprintf("Mode %s unknown for interface %s\n",argv[3],argv[4]);
		xfree(ifn);
		xfree(if_asy->name);
		xfree(if_asy);
		return -1;
	}
	xfree(ifn);
	if_asy->niface = Niface++;
	if_asy->next = Ifaces;
	Ifaces = if_asy;

	asy_init(dev,if_asy,argv[1],argv[2],(unsigned)atoi(argv[5]),trigchar,atol(argv[7]),cts,rlsd);
	return 0;
}

#endif /* ASY */
