#include <stdio.h>
#include <ctype.h>
#include "global.h"
#include "mbuf.h"
#include "timer.h"
#include "enet.h"
#include "ax25.h"
#include "arp.h"
#include "netuser.h"
#include "cmdparse.h"
#include "commands.h"

char *Arptypes[] = {
        "NET/ROM",
        "10 Mb Ethernet",
        "3 Mb Ethernet",
        "AX.25",
        "Pronet",
        "Chaos",
		"",
        "Arcnet",
};

static int near
get_hwtype(char *cp)
{
	/* This is a kludge. It really ought to be table driven */
    switch(*cp) {
	case 'n':	/* Net/Rom pseudo-type */
		return ARP_NETROM;
	case 'e':	/* "ether" */
		return ARP_ETHER;
	case 'a':	/* "ax25" */
        switch(cp[1]) {
		case 'x':
			return ARP_AX25;
		case 'r':
			return ARP_ARCNET;
		}
	}
	tprintf("Unknown hardware type %s\n",cp);
	return -1;
}

static int
doarpadd(int argc,char **argv,void *p)
{
	int32 addr;
	struct arp_tab *ap;
	char hwaddr[AXALEN];

	int pub = (*argv[0] == 'p');   			/* Is this entry published? */
	int hardware = get_hwtype(argv[2]);
	struct arp_type *at = &Arp_type[hardware];

	if((addr = resolve(argv[1])) == 0){
		tprintf(Badhost,argv[1]);
		return 1;
	}
	if(hardware == -1) {
		return -1;
	}
	if(at->scan == NULLFP){
		tputs("Attach device first\n");
		return -1;
	}
	/* If an entry already exists, clear it */
	if((ap = arp_lookup(hardware,addr)) != NULLARP) {
		arp_drop(ap);
	}
	/* Destination address */
	(*at->scan)(hwaddr,argv[3]);

	ap = arp_add(addr,hardware,hwaddr,pub,0);  	/* Put in table */
	stop_timer(&ap->timer);                 	/* Make entry permanent */
	set_timer(&ap->timer,0L);
	return 0;
}

/* Remove an ARP entry */
static int
doarpdrop(int argc,char **argv,void *p)
{
	int32 addr;
	struct arp_tab *ap;

	if((addr = resolve(argv[1])) == 0){
		tprintf(Badhost,argv[1]);
		return 1;
	}
	if((ap = arp_lookup(get_hwtype(argv[2]),addr)) != NULLARP) {
		arp_drop(ap);
	}
	return 0;
}

/* Flush all automatic entries in the arp cache */
static int
doarpflush(int argc,char **argv,void *p)
{
	struct arp_tab *ap;

	for(ap = Arp_tab; ap != NULLARP; ap = ap->next) {
		if(dur_timer(&ap->timer) != 0) {
			arp_drop(ap);
		}
	}
	return 0;
}

/* Dump ARP table */
static void near
dumparp(void)
{
	struct arp_tab *ap;

	tprintf("received %u badtype %u bogus addr %u reqst in %u replies %u reqst out %u\n" \
			"IP addr          Type           Time Q Addr\n",
		Arp_stat.recv,
		Arp_stat.badtype,
		Arp_stat.badaddr,
		Arp_stat.inreq,
		Arp_stat.replies,
		Arp_stat.outreq);

	for(ap = Arp_tab; ap != NULLARP; ap = ap->next) {
		tprintf("%-17s",inet_ntoa(ap->ip_addr));
		if(ap->hardware == ARP_AX25) {
			tputs("AX.25 [");
			switch(ap->flags) {
			case DATAGRAM_MODE:
				tputs("data] ");
				break;
			case CONNECT_MODE:
				tputs("vc]   ");
				break;
			case IPCAM_MODE:
				tputs("ipcam]");
				break;
			default:
				tputs("def]  ");
				break;
			}
		} else {
            tprintf("%-13.13s",Arptypes[ap->hardware]);
		}
		tprintf("  %-5ld",read_timer(&ap->timer)/1000L);

		if(ap->state == ARP_PENDING) {
			tprintf("%-2u",len_q(ap->pending));
		} else {
			tputs("  ");
		}
		if(ap->state == ARP_VALID) {
			if(Arp_type[ap->hardware].format != NULL) {
				char e[MAXPATH];
				(*Arp_type[ap->hardware].format)(e,ap->hw_addr);
				tputs(e);
			}
		} else {
			tputs("[unknown]");
		}
		if(ap->pub) {
			tputs(" (published)");
		}
		tputs("\n");
	}
	return;
}

int
doarp(int argc,char **argv,void *p)
{
	struct cmds Arpcmds[] = {
        "add", doarpadd, 0, 4,
		"arp add <hostid> ether|ax25|netrom <ether addr|callsign>",

        "drop", doarpdrop, 0, 3,
		"arp drop <hostid> ether|ax25|netrom",

        "flush", doarpflush, 0, 0,
		NULLCHAR,

        "publish", doarpadd, 0, 4,
		"arp publish <hostid> ether|ax25|netrom <ether addr|callsign>",

        NULLCHAR,
	};

	if(argc < 2) {
		dumparp();
		return 0;
	}
	return subcmd(Arpcmds,argc,argv,p);
}
