//==================================
// PRELOAD.C - Matt Pietrek 1993
//==================================
#include <windows.h>
#include <windowsx.h>
#include <toolhelp.h>
#include <string.h>
#include "newexe.h"
#include "preload.h"

#define MAX_SEGMENTS 1024   // Max # of segments to track

typedef struct
{
    WORD    segNumber;      // Logical segment number (1 based)
    char    szModule[9];    // Module name of segment's owner
} SEGMENT_INFO;

HWND HWndLoadedLb;          // HWND's of commonly used dialog controls
HWND HWndNeedPreloadLb;
HWND HWndDlg;

BOOL WaitingForLoad = FALSE;
BOOL WaitingForGetMessage = FALSE;
HHOOK hHook=0;                      // WH_GETMESSAGE hook 
HANDLE HInstance;                   // hInstance of this program
HTASK hTask;                        // hTask of WinExec()'ed task
unsigned SegmentsLoadedSoFar;       // Number of segments loaded so far
SEGMENT_INFO far * LoadedSegments;  // Where we keep loaded segment info

FARPROC NotifyCallbackThunk;        // MakeProcInstance() thunks.  If
FARPROC GetMsgProcThunk;            // Borland's _loadds worked, we
FARPROC DialogBoxThunk;             // wouldn't need these

//
// Given a module name and a segment number (e.g., USER segment 4),
// Open the 'NE' file and get the segment's preload flag and size.
// Returns TRUE if the segment is preload
//
BOOL IsSegmentPreload(LPSTR szModule, WORD segNumber, WORD *segSize)
{
    HMODULE hModule;
    HFILE hFile;
    char filename[144];
    DWORD ne_header_offs;
    NEWEXEHEADER neHdr;
    SEGMENT_RECORD segRec;
    
    //
    // Conver module name to HMODULE.  Convert HMODULE to the filename.
    // Open up the file.
    //
    if ( !(hModule = GetModuleHandle(szModule)) )
        return FALSE;   
    if ( !GetModuleFileName(hModule, filename, sizeof(filename)) )
        return FALSE;
    if ( !(hFile = _lopen(filename, OF_SHARE_DENY_NONE)) )
        return FALSE;
    
    //
    // Seek to the 'NE' header and read it in.  Then read in the
    // segment table entry for the specified segment.  Close the file.
    //
    _llseek(hFile, 0x3C, 0);
    _lread(hFile, &ne_header_offs, sizeof(ne_header_offs));
    _llseek(hFile, ne_header_offs, 0);
    _lread(hFile, &neHdr, sizeof(neHdr));
    _llseek(hFile, ne_header_offs + neHdr.segment_offset +
           (segNumber-1) * sizeof(SEGMENT_RECORD), 0);
    _lread(hFile, &segRec, sizeof(segRec));
    _lclose(hFile);

    *segSize = segRec.alloc_size;   // get the segment's in-memory size
    
    return (BOOL)segRec.preload;    // Return preload flag
}

//
// Minimal WH_GETMESSAGE hook procedure.  Once installed, it simply
// looks for the first GetMessage (or PeekMessage) call from the
// application that we WinExec'ed.
//
LRESULT CALLBACK _export GetMsgProc(int code, WPARAM wParam, LPARAM lParam)
{
    if ( hTask == GetCurrentTask() )        // Is it the right task?
        if ( WaitingForGetMessage == TRUE ) // Are we looking for one?
        {
            PostMessage(HWndDlg, WM_COMMAND, IDM_PROGRAM_YIELDED, 0);
            WaitingForGetMessage = FALSE;   // No more segment loads please!
        }

    CallNextHookEx(hHook, code, wParam, lParam);
    
    return 0;
}

//
// TOOLHELP NotifyRegister() callback function.  If we're looking for
// segment loads, record then in the LoadedSegments array.
//
BOOL CALLBACK _export NotifyCallback(WORD wID, DWORD dwData)
{
    //
    // Most notifications get no further than here.
    //
    if ( (WaitingForLoad == FALSE) && (WaitingForGetMessage==FALSE) )
        return FALSE;

    //
    // If we get here, we're somewhere in the process of loading the
    // specified program.
    //
    if ( wID == NFY_LOADSEG )
    {
        NFYLOADSEG far *segPtr = (NFYLOADSEG far *)dwData;
        
        if ( SegmentsLoadedSoFar >= MAX_SEGMENTS )
            return FALSE;
        
        LoadedSegments[SegmentsLoadedSoFar].segNumber = segPtr->wSegNum;
        _fstrncpy(LoadedSegments[SegmentsLoadedSoFar].szModule,
            segPtr->lpstrModuleName, 8);
        LoadedSegments[SegmentsLoadedSoFar].szModule[8] = 0;
            
        SegmentsLoadedSoFar++;
    }
    else if ( wID == NFY_STARTTASK )
    {
        //
        // After we get the NFY_STARTTASK notifications, all the preload
        // segments are in memory.  However, other segments may be
        // loaded before the task enters its message loop.  Therefore
        // we set up a WH_GETMESSAGE hook to look for the first call.
        //
        WaitingForLoad = FALSE;
        WaitingForGetMessage = TRUE;
        hTask = GetCurrentTask();
        hHook = SetWindowsHookEx(WH_GETMESSAGE, (HOOKPROC)GetMsgProcThunk,
                                 HInstance, 0);
    }

    return FALSE;
}

//
// Iterates through all the segments that were loaded as a result of the
// WinExec(). Adds relevevant data to one or both listboxes.
//
void FillInSegmentListboxes(HWND hWndDlg)
{
    HCURSOR oldCursor = 0;
    char buffer[144];
    unsigned i;
    SEGMENT_INFO far *thisSeg;
    WORD segSize;
    BOOL preload;
    DWORD totalSegSize = 0;

    //
    // This may take awhile, so put up the hourglass cursor and
    // disable listbox updating.
    //
    oldCursor = SetCursor( LoadCursor(0, IDC_WAIT) );
    SendMessage(HWndLoadedLb, WM_SETREDRAW, FALSE, NULL);
    SendMessage(HWndNeedPreloadLb, WM_SETREDRAW, FALSE, NULL);
    
    thisSeg = LoadedSegments;
    
    for ( i=0; i < SegmentsLoadedSoFar; i++ )
    {
        segSize = 0;
        preload = IsSegmentPreload(thisSeg->szModule, thisSeg->segNumber,
                                   &segSize);
        totalSegSize += segSize;
        
        wsprintf(buffer, "%s seg %02X (%u bytes)",
                 (LPSTR)(thisSeg->szModule), thisSeg->segNumber, segSize);

        // Always add to the left listbox
        SendMessage(HWndLoadedLb,LB_ADDSTRING,0,(LPARAM)(LPSTR)buffer);
        
        // If segment should be preload, add to the right listbox
        if ( !preload )
            SendMessage(HWndNeedPreloadLb, LB_ADDSTRING,
                        0,(LPARAM)(LPSTR)buffer);
        thisSeg++;
    }

    wsprintf(buffer, "Total: %lu bytes", totalSegSize);
    SetDlgItemText(hWndDlg, IDC_TEXT_TOTAL_SIZE, buffer);
        
    //
    // Restore cursor and listbox updating to normal.
    //
    SendMessage(HWndLoadedLb, WM_SETREDRAW, TRUE, NULL);
    SendMessage(HWndNeedPreloadLb, WM_SETREDRAW, TRUE, NULL);   
    SetCursor(oldCursor);
}

//
// Handle the WM_COMMAND messages
//
void Handle_WM_COMMAND(HWND hWndDlg, WPARAM wParam, LPARAM lParam)
{
    char buffer[144];

    switch ( wParam )
    {   
        case IDC_BUTTON_LOAD:
            
            // Clear the listboxes and set some "state" global vars
            SendMessage(HWndLoadedLb, LB_RESETCONTENT, 0, 0);
            SendMessage(HWndNeedPreloadLb, LB_RESETCONTENT, 0, 0);
            WaitingForLoad = TRUE;
            SegmentsLoadedSoFar = 0;

            // Get the command line from the edit control and WinExec() it
            GetDlgItemText(hWndDlg, IDC_EDIT_PROGRAM, buffer, sizeof(buffer));
            if ( WinExec(buffer, SW_SHOW) < 0x20 )
                MessageBox(hWndDlg, "Couldn't WinExec() program", 0, MB_OK);
            return;
            
        case IDM_PROGRAM_YIELDED:
            //
            // The program has finally called GetMessage (or PeekMessage).
            // undo the WH_GETMESSAGE hook, then fill in the listboxes.
            //
            UnhookWindowsHookEx(hHook);
            hHook = 0;
            FillInSegmentListboxes(hWndDlg);
            return;
    }   
    return; 
}

void Handle_WM_INITDIALOG(HWND hWndDlg)
{
    HWndDlg = hWndDlg;  // Save away in a global variable
                        // We'll need it later to do a PostMessage()
                            
    // Get the HWND's of the commonly modified controls
    HWndLoadedLb = GetDlgItem(hWndDlg, IDC_LISTBOX_LOADED);
    HWndNeedPreloadLb = GetDlgItem(hWndDlg, IDC_LISTBOX_NEED_PRELOAD);

    SetFocus( GetDlgItem(hWndDlg, IDC_EDIT_PROGRAM) );

    // Install the callback to see segment load & task starts
    NotifyRegister(0, (LPFNNOTIFYCALLBACK)NotifyCallbackThunk, NF_NORMAL);
}

//
// Dialog proc for the main dialog
//
BOOL CALLBACK _export PreloadDlgProc(HWND hWndDlg, UINT msg,
                                     WPARAM wParam, LPARAM lParam)
{
    switch ( msg )
    {
        case WM_COMMAND:
            Handle_WM_COMMAND(hWndDlg, wParam, lParam); return TRUE; 
        case WM_INITDIALOG:
            Handle_WM_INITDIALOG(hWndDlg); return TRUE;
        case WM_CLOSE:
            EndDialog(hWndDlg, 0); return FALSE;
    }    
    return FALSE;
}

int PASCAL WinMain( HANDLE hInstance, HANDLE hPrevInstance,
                    LPSTR lpszCmdLine, int nCmdShow )
{
    if ( hPrevInstance )
        return 0;

    HInstance = hInstance;

    // Allocate memory for the loaded segment info array
    LoadedSegments = (SEGMENT_INFO far *)
        GlobalAllocPtr(GMEM_MOVEABLE, sizeof(SEGMENT_INFO) * MAX_SEGMENTS);     
    if ( !LoadedSegments )
    {
        MessageBox(0, "Couldn't allocate memory", 0, MB_OK);
        return 1;
    }

    //
    // Get the required MakeProcInstance() thunks.  We wouldn't need
    // these if we could rely on _loadds working properly.
    //
    NotifyCallbackThunk = MakeProcInstance(NotifyCallback, hInstance);
    GetMsgProcThunk = MakeProcInstance((FARPROC)GetMsgProc, hInstance);
    DialogBoxThunk = MakeProcInstance(PreloadDlgProc, hInstance);
    
    DialogBox(hInstance, "PreloadDlg", 0, (DLGPROC)DialogBoxThunk);
    
    //
    // Free the allocated memory, undo the TOOLHELP and hook callbacks,
    // and free the MakeProcInstance() thunks.
    //
    (void)GlobalFreePtr( LoadedSegments );
    NotifyUnRegister(0);
    if ( hHook )
        UnhookWindowsHookEx(hHook);
    FreeProcInstance(NotifyCallbackThunk);
    FreeProcInstance(GetMsgProcThunk);
    FreeProcInstance(DialogBoxThunk);

    return 0;
}
