/****************************************************************************

        PROGRAM: useralgo.c

        HISTORY: 12/8/95 - original
                 12/9/95 - merge DLLMAIN.C and SATSTRUC.H
                 11/7/96 - separate SATSTRUC.H again for compatibility w
                              3.0 and later

        PURPOSE: DLL module with User Algorithm for WinOrbit 2.9 or later


        FUNCTIONS:
                LibMain()
                    Initialize the DLL, etc.

                WEP()
                    Clean up on exit from DLL (when WinOrbit exits).

                UserSatAlgorithm (lpWorkingDescriptor)
                    This is the exported function WinOrbit expects to find
                    Note the structure definition.  This must be rigorously
                    adhered to in order to avoid a GPF.

        EXTERNAL FUNCTIONS:
                CelestialToGeocentric(lpdouble, lpVector, lpVector)
                    This rotates a position vector from inertial
                    coordinates to earth-centered coordinates, at the
                    specified time.

        COMMENTS:
                Note: You must build this with the Large memory model!

                You cannot access the String variables unless you have
                    VBAPI.LIB


****************************************************************************/

/****************************************************************************

   Library files to include

****************************************************************************/

#include <windows.h>
#include <math.h>
#include "satstruc.h"

#define Pi 3.14159265358979
#define Pi_times_2 (Pi * 2.0)
#define Pi_over_180 (Pi / 180.0)
#define R_Equatorial 6378.165    // Earth Radius, km (Equatorial)
#define G0 75369793000000.0      // G *M(earth), (Orbit/d)^2/km^3
#define G1 1.0027379093          // Sidereal/solar time ratio



/****************************************************************************

   Function prototypes (external)
   Note: These must be declared in the .DEF file as IMPORTS from VBRUN300
         using their numerical values (which I don't know... 3/31/96)
         unless there is an "import library" for it (VBAPI.LIB)

****************************************************************************/

void FAR PASCAL CelestialToGeocentric (lpdouble, lpVector, lpVector);

/****************************************************************************

   Function prototypes (local)
   Note: Exported functions do not need to be declared or
         to appear in the module definition table

****************************************************************************/

// none

/****************************************************************************

    Global variables

****************************************************************************/

// none

/****************************************************************************

        FUNCTION:
                 LibMain (HANDLE, HANDLE, WORD)

        PURPOSE:
                Initialize library.
                This routine is called from the DLL entry point in LIBINIT.ASM
                which is called when WinOrbit first loads the DLL
                Use LIBINIT.OBJ for linking this..

****************************************************************************/

BOOL FAR PASCAL LibMain
(
    HANDLE hmod,
    HANDLE segDS,
    WORD cbHeapSize
)
{
    // Avoid compiler warnings on unused (but required) formal parameters
    cbHeapSize = cbHeapSize;
    segDS = segDS;

    // Leave our DS unlocked when we're not running
    UnlockData( 0 );

    return TRUE;
}


/****************************************************************************

        FUNCTION:
                 WEP (BOOL flag)

        PURPOSE:
                Handle exit notification from Windows.
                This routine is called by Windows when the library is freed
                by WinOrbit.  Be sure to deallocate any memory or handles
                you have allocated!

****************************************************************************/

VOID FAR PASCAL _export WEP
(
    BOOL fSystemExit
)
{
    // Avoid compiler warnings on unused (but required) formal parameters
    fSystemExit = fSystemExit;
}



/****************************************************************************

        FUNCTION:
                 UserSatAlgorithm (workingdescriptor)

        PURPOSE:
                Compute the satellite position for WinOrbit's use.  The time
                for the calculation is in the Current.Epoch element. The
                algorithm here is the "ideal" keplerian orbit, assuming
                no drag and a point-like earth.  Compare with the
                corresponding results from the builtin algorithms in
                WinOrbit (be sure to turn off the drag flag in Print or
                Map windows!)

        NOTES:
                My apologies to Tom Clark for hacking his code from Orbit
                magazine!  The orientation calculation follows the logic
                in Christie Harper's "STP"

****************************************************************************/

void FAR PASCAL _export UserSatAlgorithm
(
    lpTrackingData Sat
)
{
    Vector Ant;      // Christie Harper's ax/ay/az
    double VX0, VY0, ED;
    double E2 ;
    double S0, S1, S2 ;
    double C0, C1, C2 ;
    double Q ;          // Q = orbit number with fractional part
    double X0, Y0, R3     ;
    double M1, Residual_Error;
    double Anomaly_Sine, Anomaly_Cosine, Eccentric_Anomaly ;
    double DeltaTime, DragFactor ;
    double K2 ;


// Extract some parameters from the satellite data structures
    DeltaTime = Sat->Current.Epoch - Sat->Ref.Epoch ;
    Sat->Current = Sat->Ref ;
    Sat->Current.Epoch = Sat->Ref.Epoch + DeltaTime ;
    DragFactor = Sat->Ref.MeanMotionDeriv * Sat->DragMult ;

// Tom Clark's FNM(CalcTime) ************
    Q = Sat->Ref.MeanAnomaly / Pi_times_2
            + Sat->Ref.OrbitNum
            + Sat->Ref.MeanMotion * DeltaTime
            + DragFactor * pow(DeltaTime , 2.0) ;
    Sat->Current.OrbitNum =  (long) floor(Q + .000001) ;
    Sat->Current.MeanAnomaly = (Q - Sat->Current.OrbitNum) * Pi_times_2 ;

// Tom Clark's FNC **********************
// Update mean motion, semi-major axis
    Sat->Current.MeanMotion = Sat->Ref.MeanMotion
            + 2 * DeltaTime * DragFactor ;
    Sat->Current.SemiMajorAxis = pow(
            (G0 / (Sat->Current.MeanMotion * Sat->Current.MeanMotion)),
            (1. / 3.)) ;
    E2 = 1 - Sat->Ref.Eccentricity * Sat->Ref.Eccentricity ;

//  NODAL CORRECTION FOR NON-SPHERICAL G FIELD
    K2 = 9.95
            * pow((R_Equatorial / Sat->Current.SemiMajorAxis) , 3.5)
            / (E2 * E2) ;

// UPDATE OTHER ELEMENTS TO CURRENT EPOCH
    S1 = sin(Sat->Ref.Inclination) ;
    C1 = cos(Sat->Ref.Inclination) ;
    Sat->Current.RAAN = Sat->Ref.RAAN
            - DeltaTime * K2 * C1 * Pi_over_180 ;
    Sat->Current.ArgPerigee = Sat->Ref.ArgPerigee
            + DeltaTime * K2 * (2.5 * (C1 * C1) - .5) * Pi_over_180 ;

// Tom Clark's FNK **********************
// Solve Kepler's equation iteratively for Eccentric_Anomaly and its sin and cos

    Eccentric_Anomaly = Sat->Current.MeanAnomaly
            + Sat->Ref.Eccentricity * sin(Sat->Current.MeanAnomaly)
            + .5 * Sat->Ref.Eccentricity * Sat->Ref.Eccentricity
                    * sin(2 * Sat->Current.MeanAnomaly);
    do {
        Anomaly_Sine = sin(Eccentric_Anomaly) ;
        Anomaly_Cosine = cos(Eccentric_Anomaly) ;
        R3 = 1 - Sat->Ref.Eccentricity * Anomaly_Cosine ;
        M1 = Eccentric_Anomaly - Sat->Ref.Eccentricity * Anomaly_Sine ;
        Residual_Error = M1 - Sat->Current.MeanAnomaly ;
        Eccentric_Anomaly = Eccentric_Anomaly - Residual_Error / R3 ;
    } while (fabs(Residual_Error) > .000001) ;

// Compute position in orbital plane (X0-->perigee)
    X0 = Sat->Current.SemiMajorAxis * (Anomaly_Cosine - Sat->Ref.Eccentricity);
    Y0 = Sat->Current.SemiMajorAxis * sqrt(E2) * Anomaly_Sine ;
    Sat->Pos.R = Sat->Current.SemiMajorAxis * R3;

// Compute Velocity in orbital plane (VZ0=0) in km/sec
    ED = (R_Equatorial / 1000) * (Sat->Current.MeanMotion / R3) / 86400;
    VX0 = -Sat->Current.SemiMajorAxis * Anomaly_Sine * ED;
    VY0 = Sat->Current.SemiMajorAxis * E2 * Anomaly_Cosine * ED ;

// Rotate from orbit plane to celestial coordinates

// COORDINATE ROTATION MATRIX - from Clark
    S0 = sin(Sat->Current.RAAN) ;
    C0 = cos(Sat->Current.RAAN) ;
    S2 = sin(Sat->Current.ArgPerigee);
    C2 = cos(Sat->Current.ArgPerigee) ;

// Note subscript transposition relative to BASIC !!!!
    Sat->Pos.C[1][1] = C2 * C0 - S2 * S0 * C1 ;
    Sat->Pos.C[2][1] = -S2 * C0 - C2 * S0 * C1 ;
    Sat->Pos.C[1][2] = C2 * S0 + S2 * C0 * C1;
    Sat->Pos.C[2][2] = -S2 * S0 + C2 * C0 * C1 ;
    Sat->Pos.C[1][3] = S2 * S1 ;
    Sat->Pos.C[2][3] = C2 * S1 ;

    // new elements for squint calc - from Harper
    Sat->Pos.C[3][1] = S1 * S0 ;
    Sat->Pos.C[3][2] = -S1 * C0 ;
    Sat->Pos.C[3][3] = C1 ;

// ROTATE
    Sat->Pos.ECI.X = X0 * Sat->Pos.C[1][1] + Y0 * Sat->Pos.C[2][1] ;
    Sat->Pos.ECI.Y = X0 * Sat->Pos.C[1][2] + Y0 * Sat->Pos.C[2][2] ;
    Sat->Pos.ECI.Z = X0 * Sat->Pos.C[1][3] + Y0 * Sat->Pos.C[2][3] ;

    Sat->Pos.ECIVelocity.X = VX0 * Sat->Pos.C[1][1] + VY0 * Sat->Pos.C[2][1];
    Sat->Pos.ECIVelocity.Y = VX0 * Sat->Pos.C[1][2] + VY0 * Sat->Pos.C[2][2];
    Sat->Pos.ECIVelocity.Z = VX0 * Sat->Pos.C[1][3] + VY0 * Sat->Pos.C[2][3];

// Vector rotated to Geocentric coords
    CelestialToGeocentric (&Sat->Current.Epoch, &Sat->Pos.ECI, &Sat->Pos.Geocentric );

}

