/*
**	$id: ssvcid bmfile.c 1.2 08/03/92 10:00 am$
**		This file contains the functions needed to support reading and writing
**	bitmaps from and to Windows 3.0 bitmap files, as well as for reading in
**	icon files.
**
**	(C) 1991-3 Larry Widing
*/
#include	<windows.h>
#include	<malloc.h>
#include	<dos.h>
#include	<stdlib.h>
#include	"bitmaps.h"
#include	"bmmanip.h"

/*
** HDIB												handle of created DI bitmap
** ReadBitmapFile(const char *filename);	name of file to load
**
**    This function will read the passed file in, and create a device
**	dependant bitmap, returning the handle to the created bitmap to the
**	caller.
**
** Modification History:
** 09/06/91  LCW  Created
*/
HDIB
ReadBitmapFile(const char *filename)
{
	HFILE					file;
	int					rc;
	int					block;
	long					size;
	HDIB					hdata = (HANDLE)NULL;
	char HUGE			*pdata;
	char HUGE			*ptr;
	BITMAPFILEHEADER	bfHdr;
	OFSTRUCT				ofs;

	/*
	**	1. Open file
	*/
	file = OpenFile((LPSTR)filename, (LPOFSTRUCT)&ofs, OF_READ | OF_SHARE_DENY_WRITE);
	if (file != -1)
	{
		/*
		**	2. Read in BITMAPFILEHEADER and verify that this is a bitmap file
		*/
		rc = _lread(file, (LPSTR)&bfHdr, sizeof(BITMAPFILEHEADER));
		if (rc == sizeof(BITMAPFILEHEADER)
			|| bfHdr.bfType == ('B' + ('M' << 8)))
		{
			/*
			**	2.1. Verify that the bfSize field is correct
			*/
		{
			LONG	_offset = _llseek(file, 0L, 1);
			LONG	_size = _llseek(file, 0L, 2);
			_llseek(file, _offset, 0);
			if (bfHdr.bfSize != _size)
				bfHdr.bfSize = _size;
		}

			/*
			**	3. Allocate storage for packed DIB
			*/
			size = bfHdr.bfSize - sizeof(BITMAPFILEHEADER);
			hdata = GlobalAlloc(GMEM_MOVEABLE | GMEM_NODISCARD, size);
			if (hdata != (HANDLE)NULL)
			{
				rc = -1;
				pdata = (char HUGE *)GlobalLock(hdata);
				if (pdata != NULL)
				{
					/*
					**	4. Read in DIB header and bits into packed-DIB buffer
					*/
					block = 16 * 1024;	/* size of chunks to read in */
					ptr = pdata;

					while (size > 0)
					{
						if (size < (long)block)
						{
							block = (int)size;
						}

						if (_lread(file, (LPSTR)ptr, block) != block)
						{
							ErrorBox("ReadBitmapFile(): Error reading BMP file");
							break;
						}

						size -= (long)block;
#if	defined(__TSC__)
						if ((FP_OFF(ptr) + block) == 0)
						{
							ptr = MK_FP(FP_SEG(ptr) + 8, 0);
						}
						else
						{
							ptr += block;
						}
#else
						ptr += block;
#endif
					}

					if (size == 0)
					{
						rc = 0;
					}
					GlobalUnlock(hdata);
				}
				else
				{
					ErrorBox("ReadBitmapFile(): Error locking packed DIB memory");
				}

				if (rc < 0)
				{
					GlobalFree(hdata);
					hdata = (HANDLE)NULL;
				}
			}
			else
			{
				ErrorBox("ReadBitmapFile(): Unable to allocate memory for packed DIB");
			}
		}
		else
		{
			ErrorBox("ReadBitmapFile(): Error reading BITMAPFILEHEADER");
		}
		_lclose(file);
	}
	else
	{
		ErrorBox("ReadBitmapFile(): Unable to open bitmap file");
	}

	return hdata;
}

/*
** int
** WriteBitmapFile(
**		const char *filename,	name of file to load
**		const HANDLE hbm);		handle to packed DIB
**
**    This function will write the passed packed Device Independant Bitmap
**	to the specified file.
**
** Modification History:
** 09/06/91  LCW  Created
*/
int
WriteBitmapFile(const char *filename, const HANDLE hbm)
{
	int					file;
	int					rc = -1;
	int					block;
	long					size;
	char HUGE			*ptr;
	BITMAPFILEHEADER	bfHdr;
	LPBITMAPINFO		bmi;
	OFSTRUCT				ofs;

	/*
	**	1. Open output file
	*/
	file = OpenFile((LPSTR)filename, (LPOFSTRUCT)&ofs, OF_CREATE | OF_WRITE);
	if (file != -1)
	{
		/*
		**	2. Lock memory resource
		*/
		bmi = (LPBITMAPINFO)GlobalLock(hbm);
		if (bmi != NULL)
		{
			/*
			**	3. Create BITMAPFILEHEADER and write to file
			*/
			bfHdr.bfType = ('B' + ('M' << 8));
			bfHdr.bfSize = sizeof(BITMAPFILEHEADER) + GlobalSize(hbm);
			bfHdr.bfReserved1 = 0;
			bfHdr.bfReserved2 = 0;
			bfHdr.bfOffBits = sizeof(BITMAPFILEHEADER)
				+ (DWORD)(DIBitmapBits(bmi) - (LPSTR)bmi);
			if (_lwrite(file, (LPSTR)&bfHdr, sizeof(bfHdr)) == sizeof(bfHdr))
			{
				/*
				**	4. Write out DIB header and packed bits to file
				*/
				size = GlobalSize(hbm);
				ptr = (char HUGE *)bmi;
				block = 16 * 1024;	/* size of chunks to write out */

				while (size > 0)
				{
					if (size < (long)block)
					{
						block = (int)size;
					}

					if (_lwrite(file, (LPSTR)ptr, block) != block)
					{
						ErrorBox("WriteBitmapFiile(): Error writing DIB");
						break;
					}

					size -= (long)block;
#if	defined(__TSC__)
					if ((FP_OFF(ptr) + block) == 0)
					{
						ptr = MK_FP(FP_SEG(ptr) + 8, 0);
					}
					else
					{
						ptr += block;
					}
#else
					ptr += block;
#endif
				}

				if (size == 0)
				{
					rc = 0;
				}
			}
			else
			{
				ErrorBox("WriteBitmapFile(): Error writing BITMAPFILEHEADER");
			}

			GlobalUnlock(hbm);
		}
		else
		{
			ErrorBox("WriteBitmapFile(): Error locking bitmap into memory");
		}
		_lclose(file);
	}
	else
	{
		ErrorBox("WriteBitmapFile(): Error opening output file");
	}

	if (rc != 0)
	{
		unlink(filename);
	}
	
	return rc;
}

/*
**	Modification History
**	--------------------
**	$lgb$
** 10/15/91     Larry Widing   Initial version for Win Tech Journal Article.
** 02/10/92     Larry Widing   Added HDIB type.
** 08/03/92     Larry Widing   
**	$lge$
*/
