#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#include <limits.h>
#include "iconedit.h"
#include "iconedit.rh"

#ifdef  __BORLANDC__
#   define  min(a,b) (((a)<=(b)) ? (a) : (b))
#   define  max(a,b) (((a)>=(b)) ? (a) : (b))
#endif

HINSTANCE g_hinst = 0;
HACCEL    g_hacc  = 0;
HHOOK     g_hhook = 0;
HWND      g_hwnd  = 0;

void * __cdecl operator new(unsigned int n)
{
    return (void *) GlobalAlloc(GPTR, n);
}

void __cdecl operator delete(void *p)
{
    GlobalFree(p);
}

int Width (const RECT& rc) { return rc.right  - rc.left; }
int Height(const RECT& rc) { return rc.bottom - rc.top ; }

struct Tool { // Base class for all tools.
    enum MouseButtons { left = 0, right = 1, num_buttons = 2 };
    int nBtn; // Which button came down?
    virtual void ButtonDown(HWND, int, int, int) {}
    virtual void MouseMove (HWND, int, int     ) {}
    virtual void ButtonUp  (HWND, int, int     ) {}
};

struct BmpDC {
    BITMAP  bmi;
    HBITMAP hbmSave;
    HDC     hdc;

    void CreateDC(HBITMAP hbm) {
        GetObject(hbm, sizeof bmi, &bmi);
        hdc = CreateCompatibleDC(0);
        hbmSave = SelectBitmap(hdc, hbm);
    };
    
    BmpDC(HBITMAP hbm = 0) { // Constructor.
        if (hbm)
            CreateDC(hbm);
    }

    void DeleteDC(BOOL bDelete = FALSE) {
        HBITMAP hbm = SelectBitmap(hdc, hbmSave);
        ::DeleteDC(hdc);
        if (bDelete)
            DeleteBitmap(hbm);
    };
    
    void Blt(const BmpDC& bmpSrc) {
        BitBlt(hdc, 0, 0, min(bmi.bmWidth, bmpSrc.bmi.bmWidth),
            min(bmi.bmHeight, bmpSrc.bmi.bmHeight), 
            bmpSrc.hdc, 0, 0, SRCCOPY);
    }
    
    HBITMAP Copy(void) const {
        HBITMAP hbmNew = CreateBitmap(bmi.bmWidth, bmi.bmHeight, 
            bmi.bmPlanes, bmi.bmBitsPixel, 0);
        BmpDC bmpNew(hbmNew);
        bmpNew.Blt(*this);
        bmpNew.DeleteDC();
        return hbmNew;
    }
    
    void Clear(COLORREF cr = RGB(0, 0, 0), 
        int w = INT_MAX, int h = INT_MAX) {
        HBRUSH hbr = CreateSolidBrush(cr);
        const RECT rc = { 0, 0, 
            min(w, bmi.bmWidth), min(h, bmi.bmHeight) };
        FillRect(hdc, &rc, hbr);
        DeleteBrush(hbr);
    }
    
    void Flip(void) {
        StretchBlt(hdc, 0, 0, bmi.bmWidth, bmi.bmHeight, 
            hdc, bmi.bmWidth - 1, 0, -bmi.bmWidth, bmi.bmHeight, 
            SRCCOPY);
    }
};

struct Editor {
    ICON_DESCR *pid;
    Tool       *pTool;
    ICONINFO   ii;
    BmpDC      xor, and;
    HBITMAP    hbmSaveXor, hbmSaveAnd; // For undo.
    int        nCell, nIdx[Tool::num_buttons], nPalOffs;
    HWND       hwndZoom, hwndImg, hwndPal, hwndTB, hwndNext;
    COLORREF   acrPal[2 + 16];
    BOOL       bDirty;
    
    void ScaleDown(int& x, int& y) const { x /= nCell; y /= nCell; }
    BOOL IsIcon() const { return IMAGE_ICON == pid->uiType; }
    
    Editor(ICON_DESCR *p, Tool *pFirstTool) { // Constructor.
        pid = p;
        if (IsIcon()) 
            GetIconInfo((HICON) pid->horg, &ii);
        else {
            BmpDC bmp((HBITMAP) pid->horg);
            ii.hbmColor = bmp.Copy();
            bmp.DeleteDC();
            GetObject(ii.hbmColor, sizeof xor.bmi, &xor.bmi);
            ii.hbmMask = CreateBitmap(xor.bmi.bmWidth, 
                xor.bmi.bmHeight, 1, 1, 0);
            and.CreateDC(ii.hbmMask);
            and.Clear();
            and.DeleteDC();
        }
        pid->hnew = (HANDLE) CreateIconIndirect(&ii);
        and.CreateDC(ii.hbmMask );
        xor.CreateDC(ii.hbmColor);
        pTool = pFirstTool;
        const int nMax = max(xor.bmi.bmWidth, xor.bmi.bmHeight);
        nCell = max(min(8, 430 / nMax), 225 / xor.bmi.bmWidth);
        const COLORREF *pcr = (const COLORREF *) 
            LockResource(LoadResource(g_hinst,
            FindResource(g_hinst, "Palette", RT_RCDATA)));
        for (int i = 0; i < 2 + 16; ++i)
            acrPal[i] = pcr[i];
        nPalOffs = IsIcon() ? 0 : 2;
        nIdx[Tool::left ] = 2; // Black.
        nIdx[Tool::right] = 3; // White.
    }
    
    void SetTool(Tool *pNewTool, int id) {
        delete pTool;
        pTool = pNewTool;
        SendMessage(hwndTB, TB_CHECKBUTTON, id, TRUE);
    }
    
    void Paint(HDC hdc, int w, int h) const {
        HBRUSH hbr = CreateSolidBrush(acrPal[0]);
        DrawIconEx(hdc, 0, 0, 
            (HICON) pid->hnew, w, h, 0, hbr, DI_NORMAL);
        DeleteBrush(hbr);
    }
    
    void Refresh(int x = -1, int y = -1) {
        and.DeleteDC();
        xor.DeleteDC();
        DestroyIcon((HICON) pid->hnew);
        pid->hnew = (HANDLE) CreateIconIndirect(&ii);
        and.CreateDC(ii.hbmMask );
        xor.CreateDC(ii.hbmColor);
        if (-1 == x) {
            InvalidateRect(hwndZoom, 0, FALSE);
        } else {
            RECT rc = { x      * nCell,  y      * nCell, 
                       (x + 1) * nCell, (y + 1) * nCell };
            InvalidateRect(hwndZoom, &rc, FALSE);
        }
        UpdateWindow(hwndZoom);
        InvalidateRect(hwndImg, 0, FALSE);
        UpdateWindow(hwndImg);
        bDirty = TRUE;
    }

    void DeleteUndoInfo(void) {
        if (hbmSaveAnd)
            DeleteBitmap(hbmSaveAnd);
        if (hbmSaveXor)
            DeleteBitmap(hbmSaveXor);
    }

    void Remember(void) {
        DeleteUndoInfo();
        hbmSaveAnd = and.Copy();
        hbmSaveXor = xor.Copy();
        SendMessage(hwndTB, TB_ENABLEBUTTON, ID_UNDO, TRUE);
    }
    
    void DrawPalColor(HDC hdc, int left, int top, int nIdx) const {
        HBRUSH hbr = 
            SelectBrush( hdc, CreateSolidBrush( acrPal[ nIdx ] ) );
        Rectangle( hdc, left + 1, top + 1, left + 15, top + 15 );
        DeleteBrush( SelectBrush( hdc, hbr ) );
        if ( nIdx < 2 ) { // It's a screen color.
            RECT rc = { left + 2, top + 2, left + 14, top + 14 };
            FrameRect( hdc, &rc, GetStockBrush( WHITE_BRUSH ) );
            InsetRect( &rc, 1, 1 );
            FrameRect( hdc, &rc, GetStockBrush( BLACK_BRUSH ) );
        }
    }
    
    void SetScreenColor(int nIdx, COLORREF crBk) {
        acrPal[nIdx] = crBk;
        const COLORREF crSave = GetPixel(xor.hdc, 0, 0); // Save...
        SetPixel(xor.hdc, 0, 0, acrPal[nIdx]);
        PatBlt(xor.hdc, 0, 0, 1, 1, DSTINVERT);
        acrPal[1 - nIdx] = GetPixel(xor.hdc, 0, 0);
        SetPixel(xor.hdc, 0, 0, crSave);          // ...and restore.
        InvalidateRect(hwndPal , 0, FALSE);
        InvalidateRect(hwndZoom, 0, FALSE);
        InvalidateRect(hwndImg , 0, FALSE);
    }
    
    void OnUndo(HWND, int) {
        if (SendMessage(hwndTB, TB_ISBUTTONENABLED, ID_UNDO, 0)) {
            and.DeleteDC(/* bDelete = */ TRUE);
            xor.DeleteDC(/* bDelete = */ TRUE);
            DestroyIcon((HICON) pid->hnew);
            ii.hbmMask  = hbmSaveAnd;
            ii.hbmColor = hbmSaveXor;
            hbmSaveAnd = hbmSaveXor = 0;
            pid->hnew = (HANDLE) CreateIconIndirect(&ii);
            and.CreateDC(ii.hbmMask );
            xor.CreateDC(ii.hbmColor);
            InvalidateRect(hwndZoom, 0, FALSE);
            InvalidateRect(hwndImg , 0, FALSE);
            SendMessage(hwndTB, TB_ENABLEBUTTON, ID_UNDO, FALSE);
        }
    }
    
    void OnDelete(HWND, int) {
        Remember();
        if (nIdx[Tool::right] < 2) {     // It's a palette color.
            and.Clear(RGB(255, 255, 255));
            xor.Clear(acrPal[2 + nIdx[Tool::right]]); // B or W.
        } else {                         // It's a screen color,
            and.Clear();                 // set to opaque.
            xor.Clear(acrPal[nIdx[Tool::right]]);
        }
        Refresh();
    }
    
    void OnCopyCut(HWND hwndDlg, int id) {
        if (OpenClipboard(hwndDlg)) {
            EmptyClipboard();
            HBITMAP hbmClp = xor.Copy();
            BmpDC bmpClp(hbmClp);
            Paint(bmpClp.hdc, 
                bmpClp.bmi.bmWidth, bmpClp.bmi.bmHeight);
            bmpClp.DeleteDC();
            SetClipboardData(CF_BITMAP, hbmClp);
            CloseClipboard();
            if (ID_CUT == id)
                OnDelete(hwndDlg, id);
        }
    }
    
    void OnPaste(HWND hwndDlg, int) {
        if (IsClipboardFormatAvailable(CF_BITMAP) && 
            OpenClipboard(hwndDlg)) 
        {
            Remember();
            and.Clear();
            BmpDC bmpClp((HBITMAP) GetClipboardData(CF_BITMAP));
            xor.Blt(bmpClp);
            bmpClp.DeleteDC();
            CloseClipboard();
            Refresh();
        }
    }
    
    void OnFlip(HWND, int) {
        Remember();
        and.Flip();
        xor.Flip();
        Refresh();
    }
    
    void OnOKCancel(HWND hwndDlg, int id) {
        if (IDCANCEL == id && bDirty) {
            char szTitle[80];
            GetWindowText(hwndDlg, szTitle, sizeof szTitle);
            switch (MessageBox(hwndDlg, 
                "Do you want to save your changes?", 
                szTitle, MB_YESNOCANCEL | MB_ICONQUESTION)) {
            case IDCANCEL:
                return;
            case IDYES:
                id = IDOK;
                break;
            }
        }
        
        ChangeClipboardChain(hwndDlg, hwndNext);
        DeleteUndoInfo();
        if (IDOK != id || !IsIcon())
            DestroyIcon((HICON) pid->hnew);
        if (IDOK == id && !IsIcon())
            pid->hnew = xor.Copy();
        
        and.DeleteDC(/* bDelete = */ TRUE);
        xor.DeleteDC(/* bDelete = */ TRUE);
        EndDialog(hwndDlg, id);
        delete pTool;
        delete this;
    }
};

Editor *GetEditor(HWND hwnd) 
{
    Editor *p = (Editor *) GetWindowLong(hwnd, GWL_USERDATA);
    if (0 == p)
        p = (Editor *) GetWindowLong(GetParent(hwnd), GWL_USERDATA);
    return p;
}

void ChangePalColor(HWND hwnd, BOOL fDbl, int x, int y, int n)
{
    if (0 <= (y -= 32)) {
        Editor *p = GetEditor(hwnd);
        p->nIdx[n] = (y / 16) * 2 + (x / 16) % 2 + p->nPalOffs;
        InvalidateRect(hwnd, 0, FALSE);
        static COLORREF acrCustom[16] = { RGB(0, 0, 0) };
        CHOOSECOLOR cc = { sizeof(CHOOSECOLOR), hwnd, 0, 
            p->acrPal[p->nIdx[n]], acrCustom, CC_RGBINIT };
        if (fDbl && p->nIdx[n] < 2 && ChooseColor(&cc))
            p->SetScreenColor(p->nIdx[n], cc.rgbResult); 
    }
}

void Pal_OnPaint(HWND hwnd)
{
    PAINTSTRUCT ps;
    BeginPaint(hwnd, &ps);
    
    const Editor *p = GetEditor(hwnd);
    p->DrawPalColor(ps.hdc, 5, 5, p->nIdx[Tool::left]);
    ExcludeClipRect(ps.hdc, 5, 5, 5 + 15, 5 + 15);
    p->DrawPalColor(ps.hdc, 11, 11, p->nIdx[Tool::right]);
    
    for (int nIdx = p->nPalOffs; nIdx < 18; ++nIdx) {
        const int left = ((nIdx - p->nPalOffs) % 2) * 16;
        const int top  = ((nIdx - p->nPalOffs) / 2) * 16 + 32;
        p->DrawPalColor(ps.hdc, left, top, nIdx);
    }
    
    EndPaint(hwnd, &ps);
}

void Pal_OnLButtonDown(HWND hwnd, BOOL fDbl, int x, int y, UINT)
{
    ChangePalColor(hwnd, fDbl, x, y, 0);
}

void Pal_OnRButtonDown(HWND hwnd, BOOL fDbl, int x, int y, UINT)
{
    ChangePalColor(hwnd, fDbl, x, y, 1);
}

LRESULT CALLBACK PalWndProc(
    HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    switch (msg) {
    HANDLE_MSG(hwnd, WM_PAINT        , Pal_OnPaint      );
    HANDLE_MSG(hwnd, WM_LBUTTONDOWN  , Pal_OnLButtonDown);
    HANDLE_MSG(hwnd, WM_RBUTTONDOWN  , Pal_OnRButtonDown);
    HANDLE_MSG(hwnd, WM_LBUTTONDBLCLK, Pal_OnLButtonDown);
    HANDLE_MSG(hwnd, WM_RBUTTONDBLCLK, Pal_OnRButtonDown);
    }
    
    return DefWindowProc(hwnd, msg, wParam, lParam);
}

struct PaintTool : public Tool {
    virtual void ButtonDown(HWND hwnd, int x, int y, int nButton) {
        GetEditor(hwnd)->Remember();
        nBtn = nButton;
        MouseMove(hwnd, x, y);
    }
    
    virtual void MouseMove(HWND hwnd, int x, int y) {
        Editor *p = GetEditor(hwnd);
        if (IDC_ZOOM == GetWindowID(hwnd))
            p->ScaleDown(x, y);
        if (p->nIdx[nBtn] < 2) { // Screen color.
            SetPixelV(p->and.hdc, x, y, RGB(255, 255, 255));
            SetPixelV(p->xor.hdc, x, y, p->acrPal[2+p->nIdx[nBtn]]);
        } else {
            SetPixelV(p->and.hdc, x, y, RGB(0, 0, 0));
            SetPixelV(p->xor.hdc, x, y, p->acrPal[p->nIdx[nBtn]]);
        }
        p->Refresh(x, y);
    }
};

struct FloodTool : public Tool {
    virtual void ButtonDown(HWND hwnd, int x, int y, int nButton) {
        Editor *p = GetEditor(hwnd);
        p->Remember();
        if (IDC_ZOOM == GetWindowID(hwnd))
            p->ScaleDown(x, y);
        HBRUSH hbrSave = SelectBrush(p->xor.hdc,
            CreateSolidBrush(p->acrPal[p->nIdx[nButton]]));
        ExtFloodFill(p->xor.hdc, x, y,
            GetPixel(p->xor.hdc, x, y), FLOODFILLSURFACE);
        DeleteBrush(SelectBrush(p->xor.hdc, hbrSave));
        p->Refresh();
    }
};

BOOL Img_OnEraseBkgnd(HWND hwnd, HDC hdc)
{
    const Editor *p = GetEditor(hwnd);
    if (4 < p->nCell && IDC_ZOOM == GetWindowID(hwnd)) {
        RECT rc;
        GetClientRect(hwnd, &rc);
        HBITMAP hbm = LoadBitmap(g_hinst, "Grid");
        HBRUSH hbr = CreatePatternBrush(hbm);
        FillRect(hdc, &rc, hbr);
        DeleteBrush(hbr);
        DeleteBitmap(hbm);
    }
    return TRUE;
}

void Img_OnPaint(HWND hwnd)
{
    PAINTSTRUCT ps;
    BeginPaint(hwnd, &ps);
    
    RECT rc;
    GetClientRect(hwnd, &rc);
    const Editor *p = GetEditor(hwnd);
    BOOL bGrid = 4 < p->nCell && IDC_ZOOM == GetWindowID(hwnd);
    if (bGrid) {
        for (int y = rc.top; y < rc.bottom; y += p->nCell)
            ExcludeClipRect(ps.hdc, 0, y, rc.right, y + 1);
        for (int x = rc.left; x < rc.right; x += p->nCell)
            ExcludeClipRect(ps.hdc, x, 0, x + 1, rc.bottom);
    }
    p->Paint(ps.hdc, rc.right - bGrid, rc.bottom - bGrid);
    EndPaint(hwnd, &ps);
}

void Img_OnLButtonDown(HWND hwnd, BOOL, int x, int y, UINT)
{
    if (0 == GetCapture()) {
        SetCapture(hwnd);
        GetEditor(hwnd)->pTool->ButtonDown(hwnd, x, y, 0);
    }
}

void Img_OnRButtonDown(HWND hwnd, BOOL, int x, int y, UINT)
{
    if (0 == GetCapture()) {
        SetCapture(hwnd);
        GetEditor(hwnd)->pTool->ButtonDown(hwnd, x, y, 1);
    }
}

void Img_OnMouseMove(HWND hwnd, int x, int y, UINT)
{
    if (GetCapture() == hwnd)
        GetEditor(hwnd)->pTool->MouseMove(hwnd, x, y);
}

void Img_OnButtonUp(HWND hwnd, int x, int y, UINT)
{
    if (GetCapture() == hwnd) {
        ReleaseCapture();
        GetEditor(hwnd)->pTool->ButtonUp(hwnd, x, y);
    }
}

LRESULT CALLBACK ImgWndProc(
    HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    switch (msg) {
    HANDLE_MSG(hwnd, WM_ERASEBKGND , Img_OnEraseBkgnd );
    HANDLE_MSG(hwnd, WM_PAINT      , Img_OnPaint      );
    HANDLE_MSG(hwnd, WM_LBUTTONDOWN, Img_OnLButtonDown);
    HANDLE_MSG(hwnd, WM_RBUTTONDOWN, Img_OnRButtonDown);
    HANDLE_MSG(hwnd, WM_MOUSEMOVE  , Img_OnMouseMove  );
    HANDLE_MSG(hwnd, WM_LBUTTONUP  , Img_OnButtonUp   );
    HANDLE_MSG(hwnd, WM_RBUTTONUP  , Img_OnButtonUp   );
    }
    
    return DefWindowProc(hwnd, msg, wParam, lParam);
}

HWND CreateToolbar(HWND hwndParent)
{
    #define WS_TOOLBAR (TBSTYLE_TOOLTIPS | CCS_TOP | CCS_NODIVIDER)
    HWND hwnd = CreateWindowEx(
        0, TOOLBARCLASSNAME, 0, WS_CHILD | WS_VISIBLE | WS_TOOLBAR,
        0, 0, 0, 0, hwndParent, (HMENU) ID_BAR, g_hinst, 0);
    SendMessage(hwnd, TB_BUTTONSTRUCTSIZE, sizeof(TBBUTTON), 0);
    
    TBADDBITMAP tbabmpStd = { HINST_COMMCTRL, IDB_STD_SMALL_COLOR };
    SendMessage(hwnd, TB_ADDBITMAP, 15, (LPARAM) &tbabmpStd);
    
    TBADDBITMAP tbabmp = { g_hinst, ID_BAR };
    SendMessage(hwnd, TB_ADDBITMAP, 5, (LPARAM) &tbabmp);
    
    static const TBBUTTON tb[] = {
        {0         , 0        , 0              , TBSTYLE_SEP       },
        {15 + 0    , IDOK     , TBSTATE_ENABLED, TBSTYLE_BUTTON    },
        {15 + 1    , IDCANCEL , TBSTATE_ENABLED, TBSTYLE_BUTTON    },
        {0         , 0        , 0              , TBSTYLE_SEP       },
        {STD_CUT   , ID_CUT   , TBSTATE_ENABLED, TBSTYLE_BUTTON    },
        {STD_COPY  , ID_COPY  , TBSTATE_ENABLED, TBSTYLE_BUTTON    },
        {STD_PASTE , ID_PASTE , TBSTATE_ENABLED, TBSTYLE_BUTTON    },
        {STD_UNDO  , ID_UNDO  , 0                                  },
        {0         , 0        , 0              , TBSTYLE_SEP       },
        {STD_DELETE, ID_DELETE, TBSTATE_ENABLED                    },
        {15 + 2    , ID_FLIP  , TBSTATE_ENABLED                    },
        {0         , 0        , 0              , TBSTYLE_SEP       },
        {15 + 3    , ID_PAINT , 
              TBSTATE_ENABLED | TBSTATE_CHECKED, TBSTYLE_CHECKGROUP},
        {15 + 4    , ID_FILL  , TBSTATE_ENABLED, TBSTYLE_CHECKGROUP},
    };
    
    SendMessage(hwnd, TB_ADDBUTTONS, 
        sizeof(tb) / sizeof(tb[0]), (LPARAM)(TBBUTTON *) tb);
    return hwnd;
}

BOOL Dlg_OnInitDialog(HWND hwnd, HWND, LPARAM lParam)
{
    g_hwnd = hwnd;
    Editor *p = new Editor((ICON_DESCR *) lParam, new PaintTool);
    SetWindowLong(hwnd, GWL_USERDATA, (long) p);
    p->hwndZoom = GetDlgItem(hwnd, IDC_ZOOM);
    p->hwndImg  = GetDlgItem(hwnd, IDC_IMG);
    p->hwndPal  = GetDlgItem(hwnd, IDC_PAL);
    p->hwndTB   = CreateToolbar(hwnd);
    p->SetScreenColor(0, GetSysColor(COLOR_3DFACE));
    
    RECT rcTB, rcDlg, rcZoom, rcImg, rcPal;
    GetWindowRect(p->hwndTB, &rcTB);

    SendMessage(p->hwndTB, TB_HIDEBUTTON, ID_FILL, p->IsIcon());
    
    int nWid = p->nCell * p->xor.bmi.bmWidth  + (4 < p->nCell);
    int nHei = p->nCell * p->xor.bmi.bmHeight + (4 < p->nCell);
    SetWindowPos(p->hwndZoom, 0, 4, Height(rcTB) + 2, 
        nWid + 2, nHei + 2, SWP_NOZORDER);
    SetWindowPos(p->hwndImg, 0, nWid + 10, Height(rcTB) + 2, 
        p->and.bmi.bmWidth + 2, p->and.bmi.bmHeight + 2, 
        SWP_NOZORDER);
    SetWindowPos(p->hwndPal, 0, nWid + 10,
        Height(rcTB) + p->and.bmi.bmHeight + 6 + 2, 
        32 + 2, 162 + (p->IsIcon() ? 16 : 0), SWP_NOZORDER);
    
    GetWindowRect(p->hwndZoom, &rcZoom);
    MapWindowPoints(HWND_DESKTOP, hwnd, (POINT *) &rcZoom, 2);
    GetWindowRect(p->hwndImg, &rcImg);
    MapWindowPoints(HWND_DESKTOP, hwnd, (POINT *) &rcImg, 2);
    GetWindowRect(p->hwndPal, &rcPal);
    MapWindowPoints(HWND_DESKTOP, hwnd, (POINT *) &rcPal, 2);
    RECT rc = { 0, 0, max(rcImg.right, rcPal.right) + 4,
        max(rcZoom.bottom, rcPal.bottom) + 4 };
    AdjustWindowRectEx(&rc, GetWindowStyle(hwnd), FALSE,
        GetWindowExStyle(hwnd));
    GetWindowRect(hwnd, &rcDlg);
    SetWindowPos(hwnd, 0, 
        rcDlg.left - (Width (rc) - Width (rcDlg)) / 2,
        rcDlg.top  - (Height(rc) - Height(rcDlg)) / 2,
        Width(rc), Height(rc), SWP_NOZORDER);
    p->hwndNext = SetClipboardViewer(hwnd);
    if (0 != p->pid->pszDlgTitle)
        SetWindowText(hwnd, p->pid->pszDlgTitle);
    SetFocus(hwnd);
    return FALSE;
}

void Dlg_OnCommand(HWND hwnd, int id, HWND, UINT)
{
    Editor *p = GetEditor(hwnd);
    
    switch (id) {
    case IDCANCEL : case IDOK   : p->OnOKCancel(hwnd, id); break;
    case ID_DELETE:               p->OnDelete  (hwnd, id); break;
    case ID_CUT   : case ID_COPY: p->OnCopyCut (hwnd, id); break;
    case ID_PASTE :               p->OnPaste   (hwnd, id); break;
    case ID_FLIP  :               p->OnFlip    (hwnd, id); break;
    case ID_UNDO  :               p->OnUndo    (hwnd, id); break;
        
    case ID_PAINT: 
        p->SetTool(new PaintTool, id);
        break;
    case ID_FILL:
        if (!p->IsIcon())
            p->SetTool(new FloodTool, id);
        break;
    }
}

void Dlg_OnDrawClipboard(HWND hwnd)
{
    const Editor *p = GetEditor(hwnd);
    if (IsWindow(p->hwndNext))
        FORWARD_WM_DRAWCLIPBOARD(p->hwndNext, SendMessage);
    SendMessage(p->hwndTB, TB_ENABLEBUTTON, ID_PASTE, 
        IsClipboardFormatAvailable(CF_BITMAP));
}

void Dlg_OnChangeCBChain(HWND hwnd, HWND hwndRemove, HWND hwndNext)
{
    if (GetEditor(hwnd)->hwndNext == hwndRemove)
        GetEditor(hwnd)->hwndNext = hwndNext;
    else
        FORWARD_WM_CHANGECBCHAIN(GetEditor(hwnd)->hwndNext, 
            hwndRemove, hwndNext, SendMessage);
}

BOOL CALLBACK DlgProc(
    HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    switch (msg) {
    HANDLE_MSG(hwnd, WM_INITDIALOG   , Dlg_OnInitDialog   );
    HANDLE_MSG(hwnd, WM_COMMAND      , Dlg_OnCommand      );
    HANDLE_MSG(hwnd, WM_DRAWCLIPBOARD, Dlg_OnDrawClipboard);
    HANDLE_MSG(hwnd, WM_CHANGECBCHAIN, Dlg_OnChangeCBChain);
        
    case WM_NOTIFY:
        TOOLTIPTEXT *pTTT = (TOOLTIPTEXT *) lParam;
        if (TTN_NEEDTEXT == pTTT->hdr.code) {
            pTTT->lpszText = MAKEINTRESOURCE(pTTT->hdr.idFrom);
            pTTT->hinst = g_hinst;
            return TRUE;
        }
    }
    
    return FALSE;
}

LRESULT CALLBACK HookProc(int nCode, WPARAM wParam, LPARAM lParam)
{
    if (MSGF_DIALOGBOX == nCode && 
        TranslateAccelerator(g_hwnd, g_hacc, (MSG *) lParam))
        return TRUE;
    return CallNextHookEx(g_hhook, nCode, wParam, lParam);
}

extern "C" int IEAPI EditIcon(ICON_DESCR *pid)
{
    const WNDCLASS wcImg = { 0, ImgWndProc, 0, 0, g_hinst, 0,
        LoadCursor(0, IDC_ARROW), 0, 0, "Image" };
    static const ATOM atomImg = RegisterClass(&wcImg);
    
    const WNDCLASS wcPal = { CS_DBLCLKS, PalWndProc, 0, 0, 
        g_hinst, 0, LoadCursor(0, IDC_ARROW),
        (HBRUSH)(COLOR_3DFACE + 1), 0, "Palette" };
    static const ATOM atomPal = RegisterClass(&wcPal);
    
    InitCommonControls();
    g_hhook = SetWindowsHookEx(
        WH_MSGFILTER, HookProc, g_hinst, GetCurrentThreadId());
    const int nRet = DialogBoxParam(g_hinst, "IconEdit",
        pid->hwndParent, DlgProc, (LPARAM) pid);
    UnhookWindowsHookEx(g_hhook);
    return nRet;
}

extern "C" int APIENTRY _DllMainCRTStartup(
    HINSTANCE hinst, DWORD dwReason, LPVOID)
{
    if (DLL_PROCESS_ATTACH == dwReason) {
        g_hinst = hinst;
        g_hacc = LoadAccelerators(g_hinst, "AccTable");
    }
    return TRUE;
}
