// compile with STRICT defined and large memory model.
#include <windows.h>
#include <string.h>
#include <malloc.h>
#include <io.h>
#include <direct.h>
#include "module.h"

int     GetImportFileName(LPSTR, LPSTR, LPSTR, int);
BOOL    InModuleList(LPMODULE, LPSTR);
BOOL    AppendModuleList(LPMODULE *, LPSTR, LPSTR, BOOL);

#define EMAGIC      0x5A4D

typedef struct tagEXEHDR {
    WORD wSignature;  // executable signature
    WORD wExtraBytes; // number of bytes in last page
    WORD wPages;      // number of pages in file
    WORD wRelocItems; // number of entries in relocation table
    WORD wHeaderSize; // size of header, in paragraphs
    WORD wMinAlloc;   // minimum allocation
    WORD wMaxAlloc;   // maximum allocation
    WORD wInitSS;     // initial SS value
    WORD wInitSP;     // initial SP value
    WORD wCheckSum;   // checksum
    WORD wInitIP;     // initial IP value
    WORD wInitCS;     // initial CS value
    WORD wRelocTable; // offset to relocation table
    WORD wOverlay;    // overlay number
} EXEHDR;

#define NEMAGIC     0x454E
#define NEHDROFF    0x3C

typedef struct tagNEWEXEHDR {
    WORD wSignature;    // new executable signature
    BYTE bVersion;      // linker version number
    BYTE bRevision;     // linker revision number
    WORD wEntryTable;   // relative offset to entry table
    WORD wEntryTableSize;  // size of entry table
    DWORD dwCheckSum;   // CRC checksum of file
    WORD wFlags;        // flags
    WORD wAutoDS;       // automatic data segment
    WORD wHeapSize;     // size of local heap
    WORD wStackSize;    // size of stack
    WORD wInitIP;       // initial IP value
    WORD wInitCS;       // initial CS value
    WORD wInitSP;       // initial SP value
    WORD wInitSS;       // initial SS value
    WORD wSegmentTableSize; // segments in segment table
    WORD wModuleTableSize; // entries in module table
    WORD wNonResNameTableSize; // non-resident name table size
    WORD wSegmentTable; // offset to segment table
    WORD wResourceTable; // offset to resource table
    WORD wResNameTable; // offset to resident name table
    WORD wModuleTable;  // offset to module ref. table
    WORD wImportTable;  // offset to import name table
    WORD wNonResNameTable; // offset to non-resident name table
    WORD wMovable;      // number of movable entry points
    WORD wAlignment;    // alignment shift for segment data
    WORD wResourceSeg;  // resource segment count
    BYTE bExeType;      // target operating system
    BYTE bOtherFlags;   // other EXE flags
    WORD wGangLoad;     // relative offset to gangload area
    WORD wGangLoadSize; // size of gangload area
    WORD wMinSwapSize;  // minimum code swap size
    WORD wExpVersion;   // expected version number for Windows
} NEWEXEHDR;

#if defined(__BORLANDC__)
    #include <dir.h>
    #define _getcwd getcwd
    #define _access access
#elif defined(__WATCOMC__)
    #define _getcwd getcwd
#endif

BOOL GetImportModules(LPMODULE *ppModuleList,
                    LPSTR pszFileName, MODERRPROC ShowError)
{
    HFILE hf;
    EXEHDR oldhdr;
    NEWEXEHDR newhdr;
    WORD wNewExeOffset, wModOffset, wModules;
    BYTE cbNameLen;
    LONG lTablePos, lNamePos;
    char szModuleName[9], szModulePath[128];
    char szModuleFile[128];
    char *p;

    if ((hf = _lopen(pszFileName, READ)) == HFILE_ERROR) {
        ShowError(E_NFILE, pszFileName);
        return FALSE;
    }

    lstrcpyn(szModulePath, pszFileName,
        sizeof(szModulePath));
    if ((p = strrchr(szModulePath, '\\')) != NULL)
        *(p+1) = '\0';
    else {
        _getcwd(szModulePath, sizeof(szModulePath)-1);
        if (szModulePath[lstrlen(szModulePath)-1] != '\\')
            lstrcat(szModulePath, "\\");
    }
        
    // Read in the old-style executable header block;
    if (_lread(hf, &oldhdr, sizeof(EXEHDR)) !=
        sizeof(EXEHDR)) {
        _lclose(hf);
        ShowError(E_NMAGIC, pszFileName);
        return FALSE;
    }

    // Verify that this is an executable file
    if (oldhdr.wSignature != EMAGIC) {
        _lclose(hf);
        ShowError(E_NMAGIC, pszFileName);
        return FALSE;
    }
        
    // Verify that this is a Windows executable
    if (oldhdr.wRelocTable < 0x40) {
        _lclose(hf);
        ShowError(E_OLDAPP, pszFileName);
        return FALSE;
    }

    // Read the offset to the NE header block
    _llseek(hf, (LONG)NEHDROFF, 0);
    _lread(hf, &wNewExeOffset, sizeof(WORD));
    
    // Read in the NE header block
    _llseek(hf, (LONG)wNewExeOffset, 0);
    if (_lread(hf, &newhdr, sizeof(NEWEXEHDR)) !=
        sizeof(NEWEXEHDR)) {
        _lclose(hf);
        ShowError(E_CORRUPT, pszFileName);
        return FALSE;
    }

    // Check the NE signature
    if (newhdr.wSignature != NEMAGIC) {
        _lclose(hf);
        ShowError(E_CORRUPT, pszFileName);
        return FALSE;
    }

    // Get the module name for this executable
    _llseek(hf, (LONG)(wNewExeOffset+newhdr.wResNameTable), 0);
    _lread(hf, &cbNameLen, sizeof(BYTE));
    _lread(hf, szModuleName, (int)cbNameLen);
    szModuleName[cbNameLen] = '\0';
    AppendModuleList(ppModuleList, szModuleName,
                     pszFileName, FALSE);

    // Calculate the absolute offset of the module table
    lTablePos = (LONG)(wNewExeOffset + newhdr.wModuleTable);

    wModules = newhdr.wModuleTableSize;
    while (wModules > 0) {    
        // Seek to the next entry in the module table
        _llseek(hf, lTablePos, 0);
        _lread(hf, &wModOffset, sizeof(WORD));

        // Calculate offset to the module name
        lNamePos = (LONG)(wNewExeOffset +
            newhdr.wImportTable + wModOffset);
        _llseek(hf, lNamePos, 0);

        // Read module name
        _lread(hf, &cbNameLen, sizeof(BYTE));
        _lread(hf, szModuleName, (int)cbNameLen);
        szModuleName[cbNameLen] = '\0';

        // Check if module is in table and add if needed
        if (!InModuleList(*ppModuleList, szModuleName)) {
            if (!GetImportFileName(szModuleName, szModulePath,
                szModuleFile, sizeof(szModuleFile)))
                ShowError(E_NMODULE, szModuleName);
            else {
                if (AppendModuleList(ppModuleList,
                    szModuleName, szModuleFile, TRUE)) {
                    if (!GetImportModules(ppModuleList,
                                  szModuleFile, ShowError)) {
                        _lclose(hf);
                        return FALSE;
                    }
                }
            }
        }
        wModules--, lTablePos += sizeof(WORD);
    }
    _lclose(hf);
    return TRUE;
}

int GetImportFileName(LPSTR pszModuleName,
        LPSTR pszModulePath, LPSTR pszModuleFile,
        int cbNameLen)
{
    HMODULE hModule;
    OFSTRUCT of;
    char szBaseName[15];
    
    // Get module name from in-memory module table 
    if ((hModule = GetModuleHandle(pszModuleName)) != 0)
        return GetModuleFileName(hModule, pszModuleFile,
            cbNameLen);

    // Search for the module filename
    lstrcpy(szBaseName, pszModuleName);
    lstrcat(szBaseName, ".DLL");
    of.cBytes = sizeof(OFSTRUCT);

    if (OpenFile(szBaseName, &of, OF_EXIST|OF_SEARCH) !=
        HFILE_ERROR) {
        lstrcpyn(pszModuleFile, of.szPathName, cbNameLen);
        return lstrlen(pszModuleFile);
    } else {
        char szPathName[128];
        lstrcpy(szPathName, pszModulePath);
        lstrcat(szPathName, szBaseName);
        if (_access(szPathName, 0) != -1) {
            lstrcpyn(pszModuleFile, szPathName, cbNameLen);
            return lstrlen(pszModuleFile);
        }
    }

    return 0;
}

BOOL InModuleList(LPMODULE pModuleList, LPSTR pszModuleName)
{
    while (pModuleList) {
        if (!lstrcmp(pszModuleName, pModuleList->szName))
            return TRUE;
        pModuleList = pModuleList->pNext;
    }
    return FALSE;
}

BOOL AppendModuleList(LPMODULE *ppModuleList, LPSTR
    pszModuleName, LPSTR pszModuleFile, BOOL fFlag)
{
    LPMODULE pNewModule;
            
    pNewModule = (LPMODULE)malloc(sizeof(MODULE));
    if (!pNewModule)
        return FALSE;
    
    lstrcpyn(pNewModule->szName, pszModuleName, 9);
    lstrcpyn(pNewModule->szFile, pszModuleFile, 128);
    pNewModule->fImported = fFlag;
    pNewModule->pNext = (LPMODULE)NULL;

    if (!*ppModuleList)
        *ppModuleList = pNewModule;
    else {
        LPMODULE pLastModule = *ppModuleList;
        while (pLastModule->pNext)
            pLastModule = pLastModule->pNext;
        pLastModule->pNext = pNewModule;
        pLastModule = pNewModule;
    }
    
    return TRUE;
}

VOID FreeModuleList(LPMODULE pModuleList)
{
    LPMODULE pFree;
        
    while (pModuleList) {
        pFree = pModuleList;
        pModuleList = pModuleList->pNext;
        free(pFree);
    }
}
