//----------------------------------------------------------------------------
//
//	Sysload.c - Ton Plooy
//
//----------------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <malloc.h>
#define  STRICT
#include <windows.h>
#include <toolhelp.h>
#include "sysload.h"
#include "callback.h"

#define ID_LISTBOX		1
#define MODNAMELEN		8
#define VMM_CS			0x28
#define MAXSELECTORS	1000

typedef struct _SELCOUNT {
	WORD    wSel;
	DWORD   dwCount;
	char    szModule[MODNAMELEN + 1];
	HMODULE hMod;
	WORD    wSeg;
} SELCOUNT;


HINSTANCE hInst;
SENDBUF AddressInfo;
char szTitle[256];
SELCOUNT FAR *SelBuf;
int nNoOfSels;
int nMaxSels;


//---- Local prototypes ------------------------------------------------------
BOOL RegWndClass(HINSTANCE hInstance, char *szAppName);
InitMain(HWND hWnd);
EndMain(HWND hWnd);
int AddExecInfo(HWND hWnd, LPSENDBUF lpInfo);
int DispExecInfo(HWND hWnd, HWND hList, clock_t cTicks, DWORD dwCount);
int IncSelector(WORD wCS);
int FindSel(WORD wSelector);
int AddSel(WORD wSelector);
int WinMsg(HWND hWnd, const char *szFmt, ...);
BOOL CALLBACK About(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam);
HWND CreateListbox(HWND hWnd, int x, int y, int nWidth, int nHeight, int nID);
int SearchGlobalHeap(WORD wSel, LPSTR szModule, WORD *pwSeg, HMODULE *phMod);

//----------------------------------------------------------------------------

int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
					LPSTR lpCmdLine, int nCmdShow)
{
	static char szAppName[] = "SysLoad";
    HWND hWnd, hWndPrev;
	MSG msg;

	// Make sure only one copy of the program runs
	if ((hWndPrev = FindWindow((LPSTR)szAppName, NULL)) != NULL) {
		// Attempt to start another copy, bring first copy up
		BringWindowToTop(hWndPrev);
		return 0;
	}

    hInst = hInstance;

	InitMain(hWnd);

	if (RegWndClass(hInstance, szAppName) < 0)
		return 0;

    hWnd = CreateWindow(szAppName,
        "SysLoad",
        WS_OVERLAPPEDWINDOW,
        CW_USEDEFAULT,
        CW_USEDEFAULT,
        CW_USEDEFAULT,
        CW_USEDEFAULT,
        NULL,
        NULL,
        hInstance,
        NULL);

    if (hWnd == NULL)
        return 0;

    ShowWindow(hWnd, nCmdShow);
    UpdateWindow(hWnd);

    while (GetMessage(&msg, NULL, 0, 0)) {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }

	EndMain(hWnd);

    return ((int)msg.wParam);
}

//----------------------------------------------------------------------------

LRESULT CALLBACK MainWndProc(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam)
{
    FARPROC lpDlgProc;
	PAINTSTRUCT ps;
	HDC hDC;
	static int nTabs[] = { 10, 36, 80, 114, 156, 172 };
	static HWND hList;
	static DWORD dwCount = 0L;
	clock_t cEnd;
	static clock_t cStart;

    switch (wMsg) {

		// -------------------------------------------------------------------
        case WM_CREATE:                          
			hList = CreateListbox(hWnd, 0, 0, 0, 0, ID_LISTBOX);
			SendMessage(hList, LB_SETTABSTOPS, 6, (DWORD)(LPINT)nTabs);
			if (RegisterWnd(hWnd) != 0) {
				WinMsg(hWnd, "IRQ0.386 VxD not registered");
				PostQuitMessage(0);
			}
			cStart = clock();
			SendMessage(hList, LB_ADDSTRING, 0,
				(DWORD)(LPSTR)"Sampling data...");
            break;

		// -------------------------------------------------------------------
        case WM_COMMAND:
            switch (wParam) {
				
			case IDM_CLEAR:
				SendMessage(hList, LB_RESETCONTENT, 0, 0L);
				break;

			case IDM_ABOUT:
                lpDlgProc = MakeProcInstance((FARPROC)About, hInst);
                DialogBox(hInst, "AboutBox", hWnd, (DLGPROC)lpDlgProc);
                FreeProcInstance(lpDlgProc);
				break;

			default:
                return DefWindowProc(hWnd, wMsg, wParam, lParam);

			}
			break;

		// -------------------------------------------------------------------
        case WM_ADDRESS_INFO:
			cEnd = clock();
			while (GetAddressInfo(hWnd, &AddressInfo) == 0) {
				AddExecInfo(hWnd, &AddressInfo);
				if ((++dwCount % 50) == 0)
					DispExecInfo(hWnd, hList, cEnd - cStart, dwCount);
			}
			break;

		// -------------------------------------------------------------------
        case WM_SIZE:
			MoveWindow(hList, 10, 30, LOWORD(lParam) - 20, HIWORD(lParam) - 40, FALSE);
			break;

		// -------------------------------------------------------------------
		case WM_PAINT:
		    hDC = BeginPaint (hWnd, (LPPAINTSTRUCT)&ps);
			TextOut(hDC, 10, 10, szTitle, strlen(szTitle));
		    EndPaint(hWnd, (LPPAINTSTRUCT)&ps);
			break;

		// -------------------------------------------------------------------
        case WM_DESTROY:
            PostQuitMessage(0);
            break;

		// -------------------------------------------------------------------
        default:
            return DefWindowProc(hWnd, wMsg, wParam, lParam);
    }

    return 0L;
}

//----------------------------------------------------------------------------

AddExecInfo(HWND hWnd, LPSENDBUF lpInfo)
{
	int i;

	if (lpInfo->wStat == ENTRY_OVERFLOW)
		return -1;

	for (i = 0; i < NO_OF_SAMPLES; i++) {
		IncSelector((WORD)lpInfo->Call[i].dwCS);
	}
 
	return 0;
}

//----------------------------------------------------------------------------

IncSelector(WORD wCS)
{
	// Selector wCS was referenced, so increment its usage count. If
	// we had no previous entry for this selector a new one will be
	// created.
	int i;

	if ((i = FindSel(wCS)) < 0) {
		if ((i = AddSel(wCS)) < 0) {
			WinMsg(NULL, "Error: AddSel failed for i = %d", i);
			return -1;
		}
	}
	SelBuf[i].dwCount++;

	return 0;
}

//----------------------------------------------------------------------------

DispExecInfo(HWND hWnd, HWND hList, clock_t cTicks, DWORD dwCount)
{
	int i, j, nLen;
	char szLine[256], szFile[64];
	HDC hDC;

	hDC = GetDC(hWnd);
	i = wsprintf(szTitle, "Seconds: %ld  Interrupts: %ld",
			(DWORD)cTicks / CLOCKS_PER_SEC, dwCount * NO_OF_SAMPLES);
	TextOut(hDC, 10, 10, szTitle, i);
	ReleaseDC(hWnd, hDC);

	SendMessage(hList, WM_SETREDRAW, FALSE, 0L);
	SendMessage(hList, LB_RESETCONTENT, 0, 0L);
	for (i = 0; i < nNoOfSels; i++) {
		// Find place to insert string in listbox (count high to low)
		j = 0;
		while ((SendMessage(hList, LB_GETTEXT, j,
							(DWORD)(LPSTR)szLine) != LB_ERR) &&
				(SelBuf[i].dwCount < (DWORD)atol(szLine + 5))) {
			j++;
		}
		// Contruct listbox entry
		nLen = wsprintf(szLine, "\t%02d%%\t%08ld\t%04Xh\t%s",
				(int)((SelBuf[i].dwCount * 100) / (dwCount * NO_OF_SAMPLES)),
				SelBuf[i].dwCount, SelBuf[i].wSel,
				(LPSTR)SelBuf[i].szModule);
		if (GetModuleFileName(SelBuf[i].hMod, szFile, sizeof(szFile)) != 0) {
			wsprintf(szLine + nLen, "\t%02Xh\t%s",
					SelBuf[i].wSeg, (LPSTR)szFile);
		}
		SendMessage(hList, LB_INSERTSTRING, j, (DWORD)(LPSTR)szLine);
	}
	SendMessage(hList, WM_SETREDRAW, TRUE, 0L);
	InvalidateRect(hList, NULL, FALSE);
	UpdateWindow(hList);

	return 0;
}

//----------------------------------------------------------------------------

BOOL CALLBACK About(HWND hDlg, UINT wMsg, WPARAM wParam, LPARAM lParam)
{
    switch (wMsg) {

        case WM_COMMAND:
			switch (wParam) {

			case IDOK:
	            EndDialog(hDlg, TRUE);
	            break;

			}
			break;

		default:
			return FALSE;
    }
    return TRUE;
}

//----------------------------------------------------------------------------

int WinMsg(HWND hWnd, const char *szFmt, ...)
{
	char szBuffer[256];
	char *pArguments;

	pArguments = (char *)&szFmt + sizeof szFmt;
    wvsprintf(szBuffer, szFmt, pArguments);

    MessageBeep(MB_OK);
    MessageBox(hWnd, (LPSTR)szBuffer, (LPSTR) "Info",
				  MB_OK | MB_APPLMODAL);

    return 0;
}

//----------------------------------------------------------------------------

BOOL RegWndClass(HINSTANCE hInstance, char *szAppName)
{
	int rc;
	PWNDCLASS pWndClass;
	HANDLE hMem;
	
	hMem = LocalAlloc(LPTR, sizeof (WNDCLASS));
	if ((pWndClass = (PWNDCLASS) LocalLock(hMem)) == NULL) {
		WinMsg(NULL, "Error: alloc failed in init");
		return -1;
	}

	pWndClass->style         = CS_HREDRAW | CS_VREDRAW;
	pWndClass->lpfnWndProc   = MainWndProc;
	pWndClass->cbClsExtra    = 0;
	pWndClass->cbWndExtra    = 0;
	pWndClass->hInstance     = hInstance;
	pWndClass->hIcon         = LoadIcon(hInstance, (LPSTR)szAppName);
	pWndClass->hCursor       = LoadCursor(NULL, IDC_ARROW);
	pWndClass->hbrBackground = GetStockObject(WHITE_BRUSH);
	pWndClass->lpszMenuName  = (LPSTR)szAppName;
	pWndClass->lpszClassName = (LPSTR)szAppName;
		
	rc = RegisterClass((LPWNDCLASS) pWndClass);

	LocalUnlock(hMem);	
	LocalFree(hMem);

	return rc ? 0 : -1;
}

//----------------------------------------------------------------------------

InitMain(HWND hWnd)
{
	nNoOfSels = 0;
	nMaxSels = MAXSELECTORS;
	if ((SelBuf = (SELCOUNT FAR *)_fmalloc(sizeof(SELCOUNT) * nMaxSels)) == NULL)
		WinMsg(hWnd, "Error: can't allocate memory");

	return 0;
}

//----------------------------------------------------------------------------

EndMain(HWND hWnd)
{
	_ffree(SelBuf);

	return 0;
}

//----------------------------------------------------------------------------

HWND CreateListbox(HWND hWnd, int x, int y, int nWidth, int nHeight, int nID)
{
	// Create a listbox child window with the given dimensions.
	HWND hListbox;

    hListbox = CreateWindow("ListBox",    // Predefined control class
        NULL,                             // Caption
		WS_CHILD | LBS_NOTIFY |           // Window style
		WS_CLIPSIBLINGS | WS_BORDER |
		WS_OVERLAPPED | WS_VISIBLE | 
		WS_VSCROLL | WS_HSCROLL | 
		LBS_USETABSTOPS,
		x,                                // x location
		y,                                // y location
		nWidth,                           // width
		nHeight,                          // height
		hWnd,                             // Parent Window
		(HMENU)nID,                       // Child window ID | menu handle
		hInst,                            // instance
		NULL);                            // create args

    if (hListbox == NULL)
        return NULL;

    ShowWindow(hListbox, SW_SHOWNORMAL);
    UpdateWindow(hListbox);

	return hListbox;
}

//----------------------------------------------------------------------------

SearchGlobalHeap(WORD wSel, LPSTR szModule, WORD *pwSeg, HMODULE *phMod)
{
	MODULEENTRY ModuleEntry;
	GLOBALENTRY GlobalEntry;
	BOOL bRc;

	ModuleEntry.dwSize = sizeof(ModuleEntry);
	GlobalEntry.dwSize = sizeof(GlobalEntry);

	bRc = GlobalFirst(&GlobalEntry, GLOBAL_ALL);
	while (bRc) {
			
		if ((GlobalEntry.wType == GT_CODE) &&
				(((WORD)GlobalEntry.hBlock & 0xFFFC) == (wSel & 0xFFFC))) {
			if (ModuleFindHandle(&ModuleEntry, GlobalEntry.hOwner) != NULL) {
				_fstrcpy(szModule, ModuleEntry.szModule);
				*pwSeg = GlobalEntry.wData;
				*phMod = ModuleEntry.hModule;
				return 0;
			}
			else {
				_fstrcpy(szModule, "");
				return -1;
			}
		}
		bRc = GlobalNext(&GlobalEntry, GLOBAL_ALL);
	}

	_fstrcpy(szModule, "");
	return -1;
}

//----------------------------------------------------------------------------

int FindSel(WORD wSelector)
{
	int i = 0;

	while ((i < nNoOfSels) && (wSelector > SelBuf[i].wSel))
		i++;

	return ((i < nNoOfSels) && (wSelector == SelBuf[i].wSel)) ? i : -1;
}

//----------------------------------------------------------------------------

int AddSel(WORD wSelector)
{
	// Find place to insert
	HMODULE hMod;
	WORD wSeg;
	char szModule[MODNAMELEN + 1];
	int i = 0;

	while ((i < nNoOfSels) && (wSelector > SelBuf[i].wSel))
		i++;

	if ((i < nNoOfSels) && (wSelector == SelBuf[i].wSel)) {
		if (wSelector != 0)
			return -1; // Already exists
		else
			return i;  // 
	}

	if (nNoOfSels == nMaxSels)
		return -2;

	// Move Sel[i] -> Sel[i + 1]
	_fmemmove(SelBuf + i + 1, SelBuf + i, (nNoOfSels - i) * sizeof(SELCOUNT));

	SelBuf[i].wSel = wSelector;
	SelBuf[i].dwCount = 0;
	SelBuf[i].szModule[0] = '\0';
	SelBuf[i].wSeg = 0;
	SelBuf[i].hMod = NULL;
	nNoOfSels++;

	// Try to find a module name for the segment
	if (wSelector == VMM_CS) {
		_fstrcpy(SelBuf[i].szModule, "VMM");
	}
	else if (SearchGlobalHeap(wSelector, szModule, &wSeg, &hMod) < 0) {
		_fstrcpy(SelBuf[i].szModule, wSelector >= 0xF000 ? "BIOS" : "DOS");
	}
	else {
		_fstrcpy(SelBuf[i].szModule, szModule);
		SelBuf[i].hMod = hMod;
		SelBuf[i].wSeg = wSeg;
	}

	return i;
}
