;-----------------------------------------------------------------------------
;
; Irq0.asm -- Virtual Irq0 Device
;
; Ton Plooy   sep 94
;
;-----------------------------------------------------------------------------

.386p

INCLUDE vmm.inc
INCLUDE vpicd.inc

;----------------------------------------------------------------------------
;	Equates
;----------------------------------------------------------------------------

Irq0_VxD_ID         EQU 3006h
VERS_MAJ            EQU 1
VERS_MIN            EQU 0
VERSION             EQU ((VERS_MAJ SHL 8) OR VERS_MIN)
CFlag               EQU 1

;----------------------------------------------------------------------------
;	VxD declaration
;----------------------------------------------------------------------------

Declare_Virtual_Device IRQ0, VERS_MAJ, VERS_MIN, Irq0_VxD_Control, \
    Irq0_VxD_ID, Undefined_Init_Order, \
    Irq0_API_Handler, Irq0_API_Handler,

;----------------------------------------------------------------------------
;	Data segment
;----------------------------------------------------------------------------

VxD_DATA_SEG

ADDRESS STRUC
	dwIP	dd	?
	dwCS	dd	?
ADDRESS ENDS

SendBuf	ADDRESS 10h DUP (<>)		; Size must match NO_OF_SAMPLES

IDT STRUC
	wLimit	dw	0
	dwLBA	dd	0
IDT ENDS

IGATE STRUC
	wOff2	dw	0
	wSel	dw	0
	bRes	db	0
	bFlag	db	0
	wOff1	dw	0
IGATE ENDS

IdtReg	IDT	<>

; API jump table for the supported functions
Irq0_API_Call    label dword
	dd  offset32 Get_Version        ; AX=0
	dd  offset32 Set_CallBack       ; AX=1
Irq0_API_MaxCall EQU ( $ - Irq0_API_Call ) / 4

IntNo		dd 0	; Interrupt number for IRQ 0
DllProc_Off	dw 0	; Offset for DLL call procedure
DllProc_Seg	dw 0	; Segment
DllBuf_Off32	dd 0	; Offset for the DLL data buffer
DllBuf_Seg	dw 0	; Buffer segment
ListHandle	dd 0	; Handle for our linked list
ListNode	dd 0	; Linked list node
OurDS		dw 0
pSendBuf	dd 0
pEndBuf		dd 0
Irq0_Seg	dw 0	; 
Irq0_Off	dd 0	; 

VxD_DATA_ENDS

;----------------------------------------------------------------------------
;	Locked code segment
;----------------------------------------------------------------------------

VxD_LOCKED_CODE_SEG

; Control Procedure.
; We process the Sys_VM_Init message to initialize our driver.

BeginProc Irq0_VxD_Control

	cmp	eax, Sys_VM_Init
	je	short SetChain
	jmp	Continue

SetChain:
	; Save our DS for use in the interrupt handler
	mov	ax, ds
	mov	WORD PTR OurDS, ax

	; Get interrupt number
	mov	eax, 0			; IRQ 0. EBX has SysVM
	VxDcall	VPICD_Convert_IRQ_To_Int
	jc	Continue

	mov	IntNo, eax		; EAX has interrupt number
	mov	IntNo, 50h		; VPICD is wrong, hardcode the number

	sidt	FWORD PTR [IdtReg]	; Store IDT
	mov	esi, IdtReg.dwLBA	; Linear base address

	mov	eax, SIZE IGATE		; Calculate IDT offset
	mul	IntNo			; eax = eax * IntNo
	add	eax, IdtReg.dwLBA	; eax = IDT + offset

	; Save original address
	movzx	ebx, WORD PTR [eax].IGATE.wOff1
	shl	ebx, 16
	mov	bx,  WORD PTR [eax].IGATE.wOff2
	mov	Irq0_Off, ebx
	mov	bx, [eax].IGATE.wSel
	mov	Irq0_Seg, bx

	; Now set our entry address in the IntGate
	cli				; Clear Interrupt Flag
	mov	ebx, OFFSET32 Irq0Hook
	shr	ebx, 16
	mov	WORD PTR [eax].IGATE.wOff1, bx
	mov	ebx, OFFSET32 Irq0Hook
	mov	WORD PTR [eax].IGATE.wOff2, bx
	mov	WORD PTR [eax].IGATE.wSel, cs
	sti				; Set Interrupt Flag

	; Allocate a linked list structure
	xor	eax, eax
	mov	ecx, SIZE SendBuf	; Node size
	VMMcall	List_Create
	mov	[ListHandle], esi

	VMMcall	List_Allocate
	mov	[ListNode], eax
	mov	[pSendBuf], eax
	mov	[pEndBuf], eax
	add	[pEndBuf], SIZE SendBuf
		

Continue:
	clc
	ret

EndProc Irq0_VxD_Control

;-----------------------------------------------------------------------------
;
; Irq0_hook - Entry point for the IRQ 0 ints
;
;-----------------------------------------------------------------------------

BeginProc Irq0Hook

	push	eax
	push	ebx
	push	ecx
	push	esi
	push	ds

	; Get our DS
	mov	ax, WORD PTR cs:[OurDS]
	mov	ds, ax

	cmp	[DllProc_Seg], 0
	je	short Return			; No callback installed

	; Save cs:eip info into our SendBuf
	mov	esi, [pSendBuf]
	mov	eax, DWORD PTR ss:[esp + 14h]	; We use 14h bytes
	mov	DWORD PTR [esi].ADDRESS.dwIP, eax
	movzx	eax, WORD PTR ss:[esp + 14h + 04h]
	mov	DWORD PTR [esi].ADDRESS.dwCS, eax

	; Adjust pointer and check if our buffer is full
	add	DWORD PTR [pSendBuf], SIZE Address
	mov	eax, [pEndBuf]
	cmp	eax, [pSendBuf]
	jnz	short Return

	; Yes, buffer is full. Allocate a new buffer and send the
	; full buffer to the DLL.

	; Add the node to the linked list.
	mov	esi, [ListHandle]
	mov	eax, [ListNode]
	VMMcall	List_Attach

	; Alocate a new list entry
	mov	esi, [ListHandle]
	VMMcall	List_Allocate
	mov	[ListNode], eax
	mov	DWORD PTR [pSendBuf], eax
	mov	DWORD PTR [pEndBuf], eax
	add	[pEndBuf], SIZE SendBuf

	; Schedule the call. Use The PEF_Always_Sched flag, because
	; the VMM is not aware of our handling at interrupt time.
	VMMcall	Get_Sys_VM_Handle
	mov	eax, Low_Pri_Device_Boost
	mov	ecx, PEF_Wait_For_STI or PEF_Wait_Not_Crit or PEF_Always_Sched
	mov	esi, offset32 CallDllProc
	VMMcall	Call_Priority_VM_Event
		
Return:
	pop	ds
	pop	esi
	pop	ecx
	pop	ebx
	pop	eax

	jmp	DWORD PTR cs:Irq0_Off	; Jump to original irq handler

EndProc	Irq0Hook


VxD_LOCKED_CODE_ENDS

;----------------------------------------------------------------------------
;	Moveable code segment
;----------------------------------------------------------------------------

VxD_CODE_SEG

;-----------------------------------------------------------------------------
;
;	CallDllProc - Activated through Call_Priority_VM_Event or 
;	Call_When_VM_Ints_Enabled. We're only allowed to modify
;	EAX, EBX, ECX, EDX, ESI, EDI registers.
;
;-----------------------------------------------------------------------------

BeginProc CallDllProc

	Push_Client_State		; Save Client reg struct on the stack

	VMMcall	Begin_Nest_Exec

	; Copy info to dll memory
	mov	esi, [ListHandle]
	VMMcall	List_Remove_First

	; CX has len, ds:esi has src. Set dest(es:edi)
	push	es
	mov	ecx, SIZE SendBuf
	mov	esi, eax	; ds:esi -> Src
	mov	es, word ptr [DllBuf_Seg]
	mov	edi, dword ptr [DllBuf_Off32]
	rep	movsb
	pop	es

	; Now deallocate the list node pointed to by EAX
	mov	esi, [ListHandle]
	VMMcall	List_Deallocate

	; call DllProc()
	mov	cx, [DllProc_Seg]
	movzx	edx, [DllProc_Off]
	VMMcall	Simulate_Far_Call

	VMMcall	Resume_Exec

	VMMcall	End_Nest_Exec

	Pop_Client_State		; Restore Client Register Structure

	ret

EndProc CallDllProc

;----------------------------------------------------------------------------
;	API functions
;----------------------------------------------------------------------------

BeginProc	Irq0_API_Handler

	movzx	eax, [ebp.Client_AX]            ; Get caller's AX register
	cmp	eax, Irq0_API_MaxCall           ; Valid function number?
	jae	short fail
	and	[ebp.Client_EFlags], NOT CFlag  ; Clear carry for success
	call	Irq0_API_Call[eax * 4]          ; Call through table
	ret

fail:	or	[ebp.Client_EFlags], CFlag      ; Set carry
	ret

EndProc	Irq0_API_Handler

;----------------------------------------------------------------------------
;
;	Function AX = 0.  Irq0_Get_Version.
;
;----------------------------------------------------------------------------

BeginProc	Get_Version
	mov	[ebp.Client_AX], VERSION
	ret

EndProc	Get_Version

;----------------------------------------------------------------------------
;
;	Function AX = 1.  Irq0_Set_Callback.
;
;----------------------------------------------------------------------------

BeginProc   Set_Callback

	; ES:DI has callback address, BX:CX has Buf
	mov	ax, [ebp.Client_ES]
	mov	[DllProc_Seg], ax
	mov	ax, [ebp.Client_DI]
	mov	[DllProc_Off], ax
	mov	ax, [ebp.Client_BX]
	mov	[DllBuf_Seg], ax
	movzx	eax, [ebp.Client_CX]
	mov	dword ptr [DllBuf_Off32], eax

	mov	[ebp.Client_AX], 0
	ret

EndProc	Set_Callback

;-----------------------------------------------------------------------------
;	API end
;-----------------------------------------------------------------------------

VxD_CODE_ENDS

    END

