#define STRICT
#if !defined(WIN32) && (defined(__WIN32__) || defined(_WIN32))
#   define WIN32    1
#endif
#if defined(WIN32)
#   define DLLEXP   __declspec(dllexport)
#   define DLLCALL  __stdcall
#else
#   define DLLEXP   
#   define DLLCALL  __export FAR PASCAL
#endif

#include "barcode.h"
#include <ctype.h>
#include <string.h>
#include <stdlib.h>

struct         StandardBarcodeRecord
{
char                    encode_char;
char                    coding[21];
};

/* Fill the encodation tables with data.  The encodation table
 * should contain the character to be encoded, and it's respective
 * bar-code string.  In some instances, one table may hold three
 * different "flavors" of the same symbology (i.e. UPC-A holds an
 * "A" code, a "B" code, and a "C" code).
 */
struct StandardBarcodeRecord     two_of_five[10] =
{
{'0',"00110"},  {'1',"10001"},  {'2',"01001"},  {'3',"11000"},
{'4',"00101"},  {'5',"10100"},  {'6',"01100"},  {'7',"00011"},
{'8',"10010"},  {'9',"01010"}
};

struct StandardBarcodeRecord     upc_code[30]=
{        /* Code A */
{'0',"2100"},   {'1',"1110"},   {'2',"1011"},   {'3',"0300"},
{'4',"0021"},   {'5',"0120"},   {'6',"0003"},   {'7',"0201"},
{'8',"0102"},   {'9',"2001"},
         /*  Code B */
{'0',"0012"},   {'1',"0111"},   {'2',"1101"},   {'3',"0030"},
{'4',"1200"},   {'5',"0210"},   {'6',"3000"},   {'7',"1020"},
{'8',"2010"},   {'9',"1002"},
         /*  Code C */
{'0',"2100"},   {'1',"1110"},   {'2',"1011"},   {'3',"0300"},
{'4',"0021"},   {'5',"0120"},   {'6',"0003"},   {'7',"0201"},
{'8',"0102"},   {'9',"2001"}
};

int CheckStrictNumeric(const char *str);
int FindDigitString(char *str,char fndchar,
                    struct StandardBarcodeRecord *bcd,
                    int max);
int CreateBarcodeBitmap(HDC hdc,HBITMAP *hbmp,int *bheight,
                        int *bwidth,char *str,int ratio,
                        int density,int height);
int CalcMOD10Checksum(const char *str);
void DrawALine(HDC shdc,HDC dhdc,int xpos,int ypos,
              int size);

void charcat(char* str, char c)
    {
    char    addon[2] = {'\0', '\0'};
    addon[0]    = c;
    strcat(str, addon);
    }

DLLEXP int DLLCALL Barcode2of5(HDC hdc, char *str, HBITMAP *hbmp,
    int *bheight, int *bwidth, int checksum, int ratio, int density,
    int height)
{
char    prefix_code[] = "101000";
char    suffix_code[] = "10001";
int     chksum;
char    *digit_string;
char    return_string[21];
int     ctr1, ctr2;

/*  The mode is 0 no checksum, 1 for checksum */

    if(CheckStrictNumeric(str))
        return(-1);

/*  The sum of all the digits, including the checksum,  */
/*  should be a multiple of 10.                         */
    if(checksum)
        chksum = CalcMOD10Checksum(str);
    digit_string = (char*)calloc((strlen(str)*10)+20, sizeof(char));
    if(digit_string == 0)
        return(-1);

/*  Create a standardized code string from the barcode structure */
    strcpy(digit_string,prefix_code);
    for(ctr1=0;ctr1<strlen(str);ctr1++)
    {
        FindDigitString(return_string,str[ctr1],two_of_five,10);
        for(ctr2 = 0; return_string[ctr2] != 0; ++ctr2)
        {
            charcat(digit_string, return_string[ctr2]);
            charcat(digit_string, '0');
        }
    }
    if(checksum)
    {
        FindDigitString(return_string,(char)(chksum+'0'),two_of_five,10);
        for(ctr2 = 0; return_string[ctr2] != 0; ++ctr2)
        {
            charcat(digit_string, return_string[ctr2]);
            charcat(digit_string, '0');
        }
    }
    strcat(digit_string,suffix_code);

/*  Create the bitmap. */
    if(CreateBarcodeBitmap(hdc,hbmp,bheight,bwidth,
             digit_string,ratio,density, height))
    {
        free(digit_string);
        return(-1);
    }
    free(digit_string);
    return(0);
}

DLLEXP int DLLCALL Barcode2of5I(HDC hdc, char *str, HBITMAP *hbmp,
    int *bheight, int *bwidth, int checksum, int ratio, int density,
    int height)
{
char    prefix_code[] = "0000";
char    suffix_code[] = "100";
char    *digit_string, *newstr;
char    return_string[21], return_string1[21];
int     chksum, ctr1, ctr2;

/*  The mode is 0 no checksum, 1 for checksum.
//
//  Since Interleaved 2 of 5 requires an even amount of
//  digits, then a "0" character must be added to the
//  beginning of the string if the initial string is filled
//  with an odd number of characters.  Note:  in the case
//  of the checksum digit, the zero is prefixed to the
//  string when the initial string has an even number of digits.
*/
    if(CheckStrictNumeric(str))
        return(-1);
    newstr = (char*)malloc(strlen(str)+3);
    if(newstr == 0)
        return(-1);
    switch(checksum)
    {
    case 0: /* no checksum, make string even length if needed */
        wsprintf(newstr, "%s%s", (strlen(str)%2) ? "0" : "", str);
        break;
    case 1: /* checksum desired */
        wsprintf(newstr, "%s%s", (strlen(str)%2) ? "" : "0", str);
        chksum = CalcMOD10Checksum(newstr);
        charcat(newstr, (char)('0' + chksum));
        break;
    default:
        return(-1);
    }

    digit_string = (char*)malloc((strlen(str)*10)+20);
    if(digit_string == 0)
    {
        free(newstr);
        return(-1);
    }

/*  Create standardized code string from the barcode structure */
    strcpy(digit_string,prefix_code);
    for(ctr1=0;ctr1<strlen(newstr);ctr1+=2)
    {
        FindDigitString(return_string,newstr[ctr1],two_of_five,10);
        FindDigitString(return_string1,newstr[ctr1+1],two_of_five,10);
        ctr2 = 0;
        while(return_string[ctr2] && return_string1[ctr2])
        {
            if(return_string[ctr2] != 0)
                charcat(digit_string, return_string[ctr2]);
            if(return_string1[ctr2] != 0)
                charcat(digit_string, return_string1[ctr2]);
            ctr2++;
        }
    }
    strcat(digit_string,suffix_code);

/*  Create the bitmap.  */

    if(CreateBarcodeBitmap(hdc,hbmp,bheight,bwidth,
                         digit_string,ratio,density,height))
    {
        free(digit_string);
        free(newstr);
        return(-1);
    }

    free(digit_string);
    free(newstr);
    return(0);
}

#ifdef __BORLANDC__
#   pragma argsused
#endif
DLLEXP int DLLCALL BarcodeUPCA(HDC hdc, char *str, HBITMAP *hbmp,
    int *bheight, int *bwidth, int checksum, int ratio, int density,
    int height)
{
char        prefix_code[] = "000";
char        guard_code[] = "00000";
char        suffix_code[] = "000";
int         chksum;
char        *digit_string;
char        return_string[21];
int         ctr1;

/*  Note: the checksum parameter is unused.     */
    if(strlen(str) != 11)
        return(-1);
    if(CheckStrictNumeric(str))
        return(-1);
    chksum = CalcMOD10Checksum(str);
    if(chksum < 0)
        return(-1);

    digit_string = (char*)malloc(((strlen(str)+1)*4)+20);
    if(digit_string == 0)
        return(-1);

/*  Create a standardized code string from the barcode structure */

/*  The first 6 digits of UPC A use code A      */
    strcpy(digit_string,prefix_code);
    for(ctr1=0;ctr1<6;ctr1++)
    {
        FindDigitString(return_string,*(str+ctr1),upc_code,10);
        strcat(digit_string,return_string);
    }

    strcat(digit_string,guard_code);

/*  The last 5 digits, and checksum, use code C     */
    for(ctr1=0;ctr1<5;ctr1++)
    {
        FindDigitString(return_string,*(str+ctr1+6),&upc_code[20],10);
        strcat(digit_string,return_string);
    }
    FindDigitString(return_string,(char)(chksum+'0'),&upc_code[20],10);
    strcat(digit_string,return_string);
    strcat(digit_string,suffix_code);

/*  Create the bitmap.      */

    if(CreateBarcodeBitmap(hdc,hbmp,bheight,bwidth,
                         digit_string,ratio,density,height))
    {
        free(digit_string);
        return(-1);
    }
    free(digit_string);
    return(0);
}

/*  This routine checks to see if the string passed to it
//  contains only numeric sequences         */
int CheckStrictNumeric(const char *str)
{
    while(*str)
        if(!isdigit(*str++))
            return(-1);
    return(0);
}


/*  Find correct bar encodation string within a table. */
int FindDigitString(char *str,char fndchar,
                    struct StandardBarcodeRecord *bcd, int max)
{
int         ctr1;

    for(ctr1=0;ctr1<max;ctr1++)
    {
        if((bcd+ctr1)->encode_char == fndchar)
        {
        strcpy(str,(bcd+ctr1)->coding);
        return(0);
        }
    }
    return(-1);
}

/*  This routine creates a device context-specific bitmap
//  from a string of bar patterns.  The parameters used for
//  density and height are specified for mils (see below),
//  and the ratio is specified as 0 for 2:1, 1 for 3:1, and
//  2 for 4:1.  It returns an error code (0 for ok, or -1
//  for error), the handle to the bitmap, and the
//  dimentions of the bitmap (redundant, but it eliminates
//  a few extra steps).
//
//  This could be identified as the barcoding "engine"
//  of this system.
*/
int CreateBarcodeBitmap(HDC hdc,HBITMAP *hbmp,int *bheight,
                        int *bwidth,char *str,int ratio,
                        int density,int height)
{
long        logicy, logicx, bitmap_width, line_width, bitmap_height;
HDC         thdc;
HBITMAP     thbmp, oldbmp;
int         pixel_pos, ctr1, tmpint, bar_width, color_flag;
HBRUSH      bkgrnd;
RECT        rect;

    if((GetDeviceCaps(hdc,RASTERCAPS) & RC_BITBLT) == 0)
        return(-1);
    if(strlen(str) < 1 || density < 1 || height < 1)
        return(-1);
    ratio+=2;
/*  Calculate critical measurement factors for bitmap
//  dimentions and bar dimentions.  Everything in a bar-code
//  is measured in mils, which is one thousands of an inch
//  (0.001").  Therefore, this code will scale the bitmap
//  so that the physical representation will match it's
//  requested dimentions.  Note: since this is integer
//  math, some rounding errors will occur.
*/
    logicy = (long int)GetDeviceCaps(hdc,LOGPIXELSY);
    logicx = (long int)GetDeviceCaps(hdc,LOGPIXELSX);
    bitmap_height = (long int)(logicy * height)/1000;
    if(bitmap_height < 1)
    bitmap_height = 1;
    line_width = (long int)((logicx * density) / 1000);
    if(line_width < 1)
        line_width = 1;

/*  Loop through the barcode bas and calculate the width of
//  the bitmap.  There is a formula to do this, but it
//  tends to vary from symbol to symbol.  This was the
//  most "generic" method
*/
    bitmap_width = 0;
    for(ctr1=0;ctr1<strlen(str);ctr1++)
    {
        tmpint =  (int)((*(str+ctr1)-'0'));
        if(tmpint != 0)
            tmpint = tmpint * ratio;
        else
            tmpint = 1;
        bitmap_width += tmpint;
    }
    bitmap_width = bitmap_width * line_width;

/*  Create a matching workspace (Device Context) and
//  bitmap.  Then initialize the bitmap to an all white
//  background (this step may be redundant for a printer
//  device context).
*/
    thdc = CreateCompatibleDC(hdc);
    if(thdc == NULL)
        return(-1);
    thbmp = CreateCompatibleBitmap(thdc,(int)bitmap_width,
                                 (int)bitmap_height);
    if(thbmp == NULL)
    {
        DeleteDC(thdc);
        return(-1);
    }
    *bheight = (int)bitmap_height;
    *bwidth = (int)bitmap_width;
    bkgrnd = GetStockObject(WHITE_BRUSH);
    oldbmp = SelectObject(thdc,thbmp);
    SetRect(&rect,0,0,GetDeviceCaps(thdc,HORZRES),
          GetDeviceCaps(thdc,VERTRES));
    FillRect(thdc,&rect,bkgrnd);

/*  Loop through the digits of the bitmap and draw the
//  vertical lines.  See the comments on the "cheaters"
//  line at DrawALine().
*/
    pixel_pos = 0;
    color_flag = 1;
    while(*str != 0)
    {
        tmpint  = (int)(*str-'0');
        if(tmpint < 1)
            bar_width = (int)line_width;
        else
            bar_width = (int)(line_width *
                  (long int)(ratio * tmpint));
        if(color_flag)
            DrawALine(hdc, thdc, pixel_pos, (int)bitmap_height, bar_width);
        pixel_pos+=(int)bar_width;
        color_flag = !color_flag;
        str++;
    }

    SelectObject(thdc,oldbmp);
    DeleteDC(thdc);
    *hbmp = thbmp;
    return(0);
}

/*  This function draws a vertical line at the specified
//  points with the specified size.  I call this type of
//  line a "cheaters line" because it does not use the GDI
//  line-drawing functions, but instead uses a obscure
//  opcode in BitBlt() that copies nothing but black pixels
//  to the destination HDC.
*/
void DrawALine(HDC shdc,HDC dhdc,int xpos,int ypos,int size)
{
    BitBlt(dhdc,xpos,0,size,ypos,shdc,0,0,BLACKNESS);
}


/*  This routine calculates a weighted Mod 10 checksum
//  that's used by both 2 of 5 symbols and UPC symbols.
//  The formula being:
//  c = 3e(1) + e(2) + 3e(3) + e(4) + ... 3e(n) + e(n+1)
//      MOD 10
//  where e(n) is a barcode segment, and c is the check
//  sum.  Note: the formula does not have to end on "n+1",
//  it can end on "n".
//
//  In English, the check digit is remainder of the
//  summation of all the weighted digits in the bar-code,
//  divided by 10.
*/
int CalcMOD10Checksum(const char *str)
{
int chksum, weight, ctr1;

    chksum = 0;
    weight = 3;
    for(ctr1=0;ctr1<strlen(str);ctr1++)
    {
        if(!isdigit(*(str+ctr1)))
            return(-1);
        chksum+=(int)(((*(str+ctr1))-'0')*weight);
        chksum = chksum % 10;
        weight = (weight == 3) ? 1 : 3;
    }
    if(chksum > 0)
        chksum = 10-chksum;
    return(chksum);
}
