// security.c

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include "security.h"


VOID DumpSD(HANDLE Handle)
{
    PSECURITY_DESCRIPTOR sd = NULL;
    PSID sid = NULL;
    PACL dacl = NULL;
    SECURITY_DESCRIPTOR_CONTROL sdc;
    ULONG size = 0, revision;
    BOOL bDefault, bPresent;

    Dbg("________Dumping Security Descriptor________\n"); 

    //
    // retrieve the Security Descriptor for this object (all info)
    //
    if (!GetKernelObjectSecurity(Handle, 
             OWNER_SECURITY_INFORMATION | GROUP_SECURITY_INFORMATION |
             DACL_SECURITY_INFORMATION, sd, size, &size)) {
        
        if (GetLastError() == ERROR_INSUFFICIENT_BUFFER) {
            sd = malloc(size);
            if (!GetKernelObjectSecurity(Handle, 
                     OWNER_SECURITY_INFORMATION | GROUP_SECURITY_INFORMATION |
                     DACL_SECURITY_INFORMATION, sd, size, &size)) {
                free(sd);
                return;
            }
        }
    }

    //
    // validate the security descriptor
    //
    if (!IsValidSecurityDescriptor(sd)) {
        Dbg("   This security descriptor is invalid!\n");
        free(sd);
        return;
    }

    // 
    // first, retrieve control information for the security descriptor
    //
    if (GetSecurityDescriptorControl(sd, &sdc, &revision)) {
        Dbg("-> Control Information:\n");
        Dbg1("   Revision = %d\n", revision);
        if (sdc & SE_OWNER_DEFAULTED) Dbg("   Owner Defaulted\n");
        if (sdc & SE_GROUP_DEFAULTED) Dbg("   Group Defaulted\n");
        if (sdc & SE_DACL_PRESENT)    Dbg("   DACL Present\n");
        if (sdc & SE_DACL_DEFAULTED)  Dbg("   DACL Defaulted\n");
        if (sdc & SE_SACL_PRESENT)    Dbg("   SACL Present\n");
        if (sdc & SE_SACL_DEFAULTED)  Dbg("   SACL Defaulted\n");
        if (sdc & SE_SELF_RELATIVE)   Dbg("   Self Relative\n");
    }

    //
    // retrieve the owner SID
    //
    if (GetSecurityDescriptorOwner(sd, &sid, &bDefault)) {
        Dbg("-> Owner SID Information:\n");
        if (bDefault) {
           Dbg("   SE_OWNER_DEFAULTED flag is set\n");
        }
        if (sid != NULL) { 
            DumpSid(sid, 0);
        } else {
            Dbg("   Security Descriptor has no owner\n");
        }
    }

    //
    // retrieve the primary group SID
    //
    if (GetSecurityDescriptorGroup(sd, &sid, &bDefault)) {
        Dbg("-> Primary Group SID Information:\n");
        if (bDefault) {
           Dbg("   SE_OWNER_DEFAULTED flag is set\n");
        }
        if (sid != NULL) { 
            DumpSid(sid, 0);
        } else {
            Dbg("   Security Descriptor has no primary group\n");
        }
    }

    //
    // retrieve the descretionary access control list
    //
    if (GetSecurityDescriptorDacl(sd, &bPresent, &dacl, &bDefault)) {

        Dbg("-> Discretionary Access Control List:\n");
        if (bPresent) {
            if (bDefault) {
                Dbg("   SE_OWNER_DEFAULTED flag is set\n");
            }
            DumpAcl(dacl);
        } else {
            Dbg("   Security Descriptor has no DACL\n");
        }
    }

    //
    // NOTE: I didn't bother dumping SACL info, you can retrieve this
    // in a similar manner by calling GetSecurityDescriptorSacl
    //
    free(sd);
    return;

} // DumpSD

VOID DumpAcl(PACL Acl)
{
    ACL_REVISION_INFORMATION rev;
    ACL_SIZE_INFORMATION size;
    PACCESS_ALLOWED_ACE ace;
    ULONG ulSize, i;

    Dbg("-> Access Control List\n");

    ulSize = sizeof(ACL_REVISION_INFORMATION);
    if (GetAclInformation(Acl, &rev, ulSize, AclRevisionInformation)) {
        Dbg1("   Revision = %d\n", rev.AclRevision);
    }

    ulSize = sizeof(ACL_SIZE_INFORMATION);
    if (GetAclInformation(Acl, &size, ulSize, AclSizeInformation)) {
        Dbg1("   AceCount = %d\n", size.AceCount);
        Dbg1("   Bytes In Use = %d\n", size.AclBytesInUse);
        Dbg1("   Bytes Free = %d\n", size.AclBytesFree);
    }

    //
    // Dump info for each ace in the acl
    //
    for (i=0; i < Acl->AceCount; i++) {
        if (GetAce(Acl, i, (LPVOID *)&ace)) {
            if (ace->Header.AceType == ACCESS_ALLOWED_ACE_TYPE) {
                Dbg("   Access ALLOWED: ");
            } else if (ace->Header.AceType == ACCESS_DENIED_ACE_TYPE){
                Dbg("   Access DENIED: ");
            } else {
                Dbg("   Audit ace: ");
            }
            DumpSid((PSID)&(ace->SidStart), 0);
            DumpAccessMask(ace->Mask);
        }
    }

    return;

} // DumpAcl

VOID DumpAccessMask(ACCESS_MASK Mask)
{
    // Bits 0 - 15: Specific Rights
    Dbg1("   Specific(%x)", (Mask & SPECIFIC_RIGHTS_ALL));
    
    // Bits 16 - 23 (16-20 currently defined): Standard rights
    if (Mask & DELETE) Dbg(",DELETE");
    if (Mask & READ_CONTROL) Dbg(",READ_CONTROL");
    if (Mask & WRITE_DAC) Dbg(",WRITE_DAC");
    if (Mask & WRITE_OWNER) Dbg(",WRITE_OWNER");
    if (Mask & SYNCHRONIZE) Dbg(",SYNCHRONIZE");

    // Bit 24: Access system security
    if (Mask & ACCESS_SYSTEM_SECURITY) Dbg(",ACCESS_SYSTEM_SECURITY");

    // Bit 25: Maximum allowed
    if (Mask & MAXIMUM_ALLOWED) Dbg(",MAXIMUM_ALLOWED");
    // Bits 26 - 27: Reserved

    // Bits 28 - 31: Generic Rights
    if (Mask & GENERIC_ALL) Dbg(",GENERIC_ALL");
    if (Mask & GENERIC_EXECUTE) Dbg(",GENERIC_EXECUTE");
    if (Mask & GENERIC_WRITE) Dbg(",GENERIC_WRITE");
    if (Mask & GENERIC_READ) Dbg(",GENERIC_READ");

    Dbg("\n");
} // DumpAccessMask

VOID DumpSid(PSID Sid, DWORD Attributes)
{
    SID_NAME_USE Use;
    char szAccount[MAX_PATH], szDomain[MAX_PATH];
    ULONG size1 = MAX_PATH, size2 = MAX_PATH;

    if (LookupAccountSid(NULL, Sid, szAccount, &size1, szDomain, &size2, &Use)) {        
        Dbg1("   Account = %s", szAccount);
        Dbg1(", Domain = %s\n", szDomain);

        if ((Attributes & SE_GROUP_MANDATORY) == SE_GROUP_MANDATORY) {
            Dbg("   Attribute: SE_GROUP_MANDATORY\n");
        }
        if ((Attributes & SE_GROUP_ENABLED_BY_DEFAULT) == SE_GROUP_ENABLED_BY_DEFAULT) {
            Dbg("   Attribute: SE_GROUP_ENABLED_BY_DEFAULT\n");
        }
        if ((Attributes & SE_GROUP_ENABLED) == SE_GROUP_ENABLED) {
            Dbg("   Attribute: SE_GROUP_ENABLED\n");
        }
        if ((Attributes & SE_GROUP_OWNER) == SE_GROUP_OWNER) { 
            Dbg("   Attribute: SE_GROUP_OWNER\n");
        }
        if ((Attributes & SE_GROUP_LOGON_ID) == SE_GROUP_LOGON_ID) {
            Dbg("   Attribute: SE_GROUP_LOGON_ID\n");
        }

    } else {
        if (GetLastError() == ERROR_NONE_MAPPED) {
            Dbg("   No mapping between account name and SID IDs\n");
        } else {
            Dbg("   LookupAccountSid returned an error\n");
        }
    }
        
} // DumpSid

BOOL SetHandleSecurity(HANDLE Handle, LPTSTR AccountName, DWORD  Access,
                       BOOL Allowed)
{
    BOOL status = FALSE, bPresent, bDefault;
    PSID sid = NULL;
    SID_NAME_USE use;
    PSECURITY_DESCRIPTOR sd = NULL, newSd = NULL;
    PACL dacl = NULL, newDacl = NULL;
    PACCESS_ALLOWED_ACE ace;
    DWORD sidSize = 0, sdSize = 0, daclSize = 0, size = MAX_PATH, i;
    char szDomain[MAX_PATH];

    // retrieve the SID that corresponds to this user/group
    if (!LookupAccountName(NULL, AccountName, sid, &sidSize,
                           szDomain, &size, &use) &&
        (GetLastError() == ERROR_INSUFFICIENT_BUFFER)) {

        sid = malloc(sidSize);
        if (!LookupAccountName(NULL, AccountName, sid, &sidSize,
                               szDomain, &size, &use)) {
            goto CleanUp;
        }
    }

    // retrieve the SD for this kernel object handle
    if (!GetKernelObjectSecurity(Handle, DACL_SECURITY_INFORMATION,
                                 sd, sdSize, &sdSize) &&
        (GetLastError() == ERROR_INSUFFICIENT_BUFFER)) {

        sd = malloc(sdSize);
        if (!GetKernelObjectSecurity(Handle, DACL_SECURITY_INFORMATION,
                                     sd, sdSize, &sdSize)) {
            goto CleanUp;
        }
    }

    if (GetSecurityDescriptorDacl(sd, &bPresent, &dacl, &bDefault) &&
        bPresent) {

        daclSize = dacl->AclSize;
        size = GetLengthSid(sid) + sizeof(ACCESS_ALLOWED_ACE)
               - sizeof(DWORD);
    } else {
        // no sd for this kernel object handle
        size = GetLengthSid(sid) + sizeof(ACL) +
               sizeof(ACCESS_ALLOWED_ACE) - sizeof(DWORD);
    }

    newDacl = malloc(daclSize + size);
    if (newDacl == NULL) {
        goto CleanUp;
    }

    newSd = malloc(GetSecurityDescriptorLength(sd) + size);
    if (newSd == NULL) {
        goto CleanUp;
    }

    if (!InitializeSecurityDescriptor(newSd, 
            SECURITY_DESCRIPTOR_REVISION)) {

        goto CleanUp;
    }

    if (!InitializeAcl(newDacl, daclSize + size, ACL_REVISION)) {
        goto CleanUp;
    }

    // if the new ace is "access denied", add it first to the new dacl
    if (!Allowed) {
        if (!AddAccessDeniedAce(newDacl, ACL_REVISION, Access, sid)) {
            goto CleanUp;            
        }
    }

    // add the original ACEs to the new DACL
    if (dacl) {
        for (i=0; i < dacl->AceCount; i++) {
            if (GetAce(dacl, i, (LPVOID *)&ace)) {
                if (ace->Header.AceType == ACCESS_DENIED_ACE_TYPE) {
                    AddAccessDeniedAce(newDacl, ACL_REVISION, ace->Mask,
                                       (PSID)&(ace->SidStart));
                } else {               
                    AddAccessAllowedAce(newDacl, ACL_REVISION, ace->Mask,
                                        (PSID)&(ace->SidStart));
                }
            }
        }
    }

    // if the new ace is "access allowed", add it last to the new dacl
    if (Allowed) {
        if (!AddAccessAllowedAce(newDacl, ACL_REVISION, Access, sid)) {
            goto CleanUp;
        }
    }

    // finally, apply the new dacl to the object        
    if (SetSecurityDescriptorDacl(newSd, TRUE, newDacl, FALSE)) {
        if (IsValidSecurityDescriptor(newSd)) {
            if (SetKernelObjectSecurity(Handle,
                    DACL_SECURITY_INFORMATION, newSd)) {
                status = TRUE;
            }
        }
    }

    CleanUp:
    if (sid) free(sid);
    if (sd) free(sd);
    if (newDacl) free(newDacl);
    if (newSd) free(newSd);

    return status;
} // SetHandleSecurity
