// sendkeys.cpp
//
#define STRICT
#include <windows.h>
#include <process.h>
#include <string.h>
#pragma hdrstop

#include <vector>  // for STL vector class
using std::vector;

#include "sendkeys.h"

class KeyList {
   // manages a list of keys strokes
   public:
      KeyList( const WORD *pkeys );
      KeyList( const char *keystr );
      // used during playback hook
      BOOL Next();
      int GetCurrent( EVENTMSG& );
      void Add( UINT key ) {
         keys.insert(keys.end(),key);
      }
      void ModKeysUp();
      void DoToggles();
   private:
      int idx;
      vector<UINT> keys;
      DWORD time;
};

void KeyList::ModKeysUp()
{
   WORD modkeys[] = { VK_SHIFT, VK_CONTROL, VK_MENU,
                        VK_LWIN, VK_RWIN, 0 };
   // send keyup for all keys that are down
   for(int k=0; modkeys[k]; k++ )
      if( GetKeyState(modkeys[k]) < 0 )
         keys.insert(keys.begin(),modkeys[k]|SK_UP);
}

void KeyList::DoToggles()
{
   WORD togkeys[] = { VK_CAPITAL, VK_NUMLOCK, VK_SCROLL, 0 };
   // toggle keys that are enabled
   for(int k=0; togkeys[k]; k++ )
      if( GetKeyState(togkeys[k]) & 1 ) {
         keys.insert(keys.begin(),togkeys[k]);
         keys.insert(keys.end(),togkeys[k]);
      }
}

KeyList::KeyList( const WORD *pkeys )
   : idx(-1), time(GetCurrentTime())
{
   for(int i=0;pkeys[i];i++)
      Add(pkeys[i]);
}

KeyList::KeyList( const char *keystr )
   : idx(-1), time(GetCurrentTime())
{
   BOOL shift = FALSE, ctrl = FALSE, alt = FALSE;
   for(;*keystr;keystr++) {
      WORD vk = VkKeyScan(*keystr);
      if(vk != 0xffff ) {
         BOOL new_shift = !!(vk&0x100);
         BOOL new_ctrl  = !!(vk&0x200);
         BOOL new_alt   = !!(vk&0x400);
         if( new_shift != shift ) {
            Add(VK_SHIFT|(shift?SK_UP:SK_DOWN));
            shift = new_shift;
         }
         if( new_ctrl != ctrl ) {
            Add(VK_CONTROL|(ctrl?SK_UP:SK_DOWN));
            ctrl = new_ctrl;
         }
         if( new_alt != alt ) {
            Add(VK_MENU|(alt?SK_UP:SK_DOWN));
            alt = new_alt;
         }
         keys.insert(keys.end(),LOBYTE(vk));
      } else {
         if(shift)
            Add(VK_SHIFT|SK_UP);
         shift=FALSE;
         if(ctrl)
            Add(VK_SHIFT|SK_UP);
         ctrl=FALSE;
         if(alt)
            Add(VK_MENU|SK_UP);
         Add(VK_MENU|SK_DOWN);
         Add(VK_NUMPAD0);
         BYTE k = *keystr;
         Add(VK_NUMPAD0+k/100);
         Add(VK_NUMPAD0+(k/10)%10);
         Add(VK_NUMPAD0+k%10);
         Add(VK_MENU|SK_UP);
         alt = FALSE;
      }
   }
   // force all modifiers up
   if(alt)
      Add(VK_MENU|SK_UP);
   if(ctrl)
      Add(VK_CONTROL|SK_UP);
   if(shift)
      Add(VK_SHIFT|SK_UP);
}

BOOL KeyList::Next()
{
   time = GetCurrentTime();
   if( idx == -1 )
      idx++;
   else {
      if( keys[idx] & (SK_UP|SK_DOWN) )
         idx++;  // go to next key
      else
         keys[idx] |= SK_UP; // did the down, now do the up
   }
   return idx==keys.size(); // no more keys
}

int KeyList::GetCurrent( EVENTMSG& msg )
{
   if( keys[idx] & SK_UP )
      msg.message = (keys[idx]&SK_SYS)?WM_SYSKEYUP:WM_KEYUP;
   else
      msg.message = (keys[idx]&SK_SYS)?WM_SYSKEYDOWN:WM_KEYDOWN;
   msg.paramL = LOBYTE(keys[idx]);
   msg.paramH = MapVirtualKey(LOBYTE(keys[idx]),0);
   msg.time = time;
   msg.hwnd = NULL;
   return 0;
}

class SendKeysHook {
// class to manage data that for a journal playback hook
   public:
      static BOOL SendKeys( const WORD *keys );
      static BOOL SendKeys( const char *keystr );
   private:
      SendKeysHook( const WORD *keys )
         : hHook(0), keylist(keys)
      {}
      SendKeysHook( const char *keystr )
        : hHook(0), keylist(keystr)
      {}
      ~SendKeysHook() {
         Unhook();
         theHook = NULL;
      }
      void Unhook();    // remove hook
      BOOL Start();     // install hook
      HHOOK hHook;      // hook handle
      KeyList keylist;
      static LRESULT CALLBACK HookProc( int, WPARAM, LPARAM );
      LRESULT PlayBack( int, WPARAM, LPARAM );
      static void Thread( LPVOID p );
      void Thread();
      static SendKeysHook* theHook;
};

SendKeysHook* SendKeysHook::theHook;

void SendKeysHook::Unhook()
{
   if(hHook)
      UnhookWindowsHookEx(hHook);
   hHook = 0;
}

BOOL SendKeysHook::Start() {
   if(keylist.Next())
      return FALSE; // no keys to send
   theHook = this;
   keylist.ModKeysUp(); // force modifiers up
   keylist.DoToggles(); // toggle lock key off then restore
   // install the hook
   hHook = SetWindowsHookEx(WH_JOURNALPLAYBACK,HookProc,
                              GetModuleHandle(0),0);
   return hHook != NULL;
}

LRESULT CALLBACK SendKeysHook::HookProc( int nCode,
  WPARAM wParam, LPARAM lParam )
// function called by Windows for the playback hook
{
   if(theHook) // make sure data is there
      return theHook->PlayBack(nCode,wParam,lParam);
   return 0;
}

LRESULT SendKeysHook::PlayBack( int nCode, WPARAM wParam, LPARAM lParam )
{
   switch(nCode) {
      case HC_SKIP:
         if( keylist.Next() ) {
            // out of keys
            Unhook();
            PostQuitMessage(0); // tell thread to quit
         }
         return 0;

      case HC_GETNEXT: {
         EVENTMSG& msg = *LPEVENTMSG(lParam);
         return keylist.GetCurrent(msg); // get the current message
      }

      case HC_SYSMODALON:
      case HC_SYSMODALOFF:
         return 0;

      default:
         return CallNextHookEx(hHook,nCode,wParam,lParam);
   }
}


void SendKeysHook::Thread( LPVOID p )
{
   ((SendKeysHook *)p)->Thread();
}


class CriticalSectionObject {
   // class to encapsulate a OS critical section
   public:
      CriticalSectionObject() {
         InitializeCriticalSection(&cs);
      }
      ~CriticalSectionObject() {
         DeleteCriticalSection(&cs);
      }
      void Enter() {
         EnterCriticalSection(&cs);
      }
      void Leave() {
         LeaveCriticalSection(&cs);
      }
   private:
      CRITICAL_SECTION cs;
};

class CriticalSection {
   // class to encapsulate a critical section
   public:
      CriticalSection( CriticalSectionObject& CSO ) : CSO(CSO) {
         CSO.Enter();
      }
      ~CriticalSection() {
         CSO.Leave();
      }
   private:
      CriticalSectionObject& CSO;
};

void SendKeysHook::Thread()
// function for thread that exists during the hook
{
   static CriticalSectionObject CSO;

   CriticalSection CS(CSO); // enter/leave critical section

   try {
      if(Start()) {
         // start a message loop
         MSG msg;
         while(GetMessage(&msg,0,0,0)) {
            if( msg.message == WM_CANCELJOURNAL )
               break; // exit loop
            TranslateMessage(&msg);
            DispatchMessage(&msg);
         }
      }
   } catch(...) {
      // don't allow exception to propagate from thread
   }
   delete this; // delete hook object
}

BOOL SendKeysHook::SendKeys( const WORD *pkeys )
{
   try {
      SendKeysHook *hook = new SendKeysHook(pkeys);
      if( hook && _beginthread(Thread,0,hook) )
         return TRUE;
      else {
         delete hook;
         return FALSE;
      }
   } catch(...) {
      return FALSE;
   }
}

BOOL SendKeysHook::SendKeys( LPCSTR keystr )
{
   try {
      SendKeysHook *hook = new SendKeysHook(keystr);
      if(hook && _beginthread(Thread,0,hook))
         return TRUE;
      else {
         delete hook;
         return FALSE;
      }
   } catch(...) {
      return FALSE;
   }
}

extern "C"
BOOL WINAPI SendKeys( const WORD *pkeys )
{
   return SendKeysHook::SendKeys(pkeys);
}

extern "C"
BOOL WINAPI SendKeysStr( LPCSTR keystr )
{
   return SendKeysHook::SendKeys(keystr);
}

