// STATPROX.cpp - main module for VxD STATPROX
//
//  11/20/96    VV      Original
//
// This VxD acts as a Ring0 proxy for Ring3 stats making them
// visible at the Win-95 SYSMON application. The VxD serves as
// a gateway between a Win32 app and Microsoft's PERF.VXD.
//

#define DEVICE_MAIN
#include "statprox.h"
Declare_Virtual_Device(STATPROX)
#undef DEVICE_MAIN

#include    <Vsvctab.h>

#include    "classes.h"

BOOL StatproxDevice::OnSysDynamicDeviceInit()
{
    if (::PERF_Get_Version() <= 0)
        return FALSE;   // refuse to load unless PERF is avail
    m_pPerfHeap = NULL; m_Usage = 0;
    return TRUE;
}

BOOL StatproxDevice::OnSysDynamicDeviceExit()
{
    if (m_Usage==0) {
        if (m_pPerfHeap) 
             delete m_pPerfHeap; 
        return TRUE;
    }
    else
        return FALSE;
}

BOOL StatproxDevice::OnDeviceInit(VMHANDLE, PCHAR)
{
    // called when loaded statically (system.ini/[386Enh])
    return OnSysDynamicDeviceInit();
}

DWORD StatproxDevice::OnW32DeviceIoControl
                            (PIOCTLPARAMS pDIOCParams)
{
    // define names for convenience:
    #define Func        pDIOCParams->dioc_IOCtlCode
    #define pStruct1    pDIOCParams->dioc_InBuf
    #define pStruct2    pDIOCParams->dioc_OutBuf
    #define pdwResult   pDIOCParams->dioc_bytesret
    #define pProcess    pDIOCParams->dioc_ppdb

    switch (Func) {
     // OS-provided control codes:
     case DIOC_OPEN:                    
        // CreateFile() was called: create the list on 1st usage
        if (m_Usage++==0)
            CStatServer::CreateList();  
        break;              // ok
     case DIOC_CLOSEHANDLE: 
        // CloseHandle() was called: destroy if it was the last
        if (--m_Usage==0) {
            CStatServer::DestroyList(); // last gone, destroy everything
            if (m_pPerfHeap) delete m_pPerfHeap; m_pPerfHeap=NULL;
        }
        break;
     case STATDIOC_INIT:                    
        // pass a shared area to be used for all monitorees
        // pStruct1 -- area
        if (m_pPerfHeap) delete m_pPerfHeap;
        m_pPerfHeap = new CPerfHeap(PDWORD(pStruct1));
        break;
     case STATDIOC_DEINIT:
        // inform that the shared are is about to disappear
        if (m_pPerfHeap) 
            { delete m_pPerfHeap; m_pPerfHeap=NULL; }
        break;
     case STATDIOC_REGISTER_STAT:
        // register a stat:
        // pStruct1 -- STATDIOC_req_t;
        // pStruct2 -- STATDIOC_resp_t;
        PSTATDIOC_resp_t(pStruct2)->index = -1; // assume failure
        { CStatServer* pServer = 
            GetServerNode(PSTATDIOC_req_t(pStruct1)->pszSubsystem);
          if (pServer) {
           PSTATDIOC_resp_t(pStruct2)->hServerCookie =
                     DWORD(pServer); 
           PSTATDIOC_resp_t(pStruct2)->hStatCookie =
                DWORD(AddStat(
                        pServer,
                        PSTATDIOC_req_t(pStruct1)->pszName,
                        PSTATDIOC_req_t(pStruct1)->pszDescription,
                        PSTATDIOC_req_t(pStruct1)->bValueNotRate,
                        &PSTATDIOC_resp_t(pStruct2)->index));
          }
        }
        break;
     case STATDIOC_UNREGISTER_STAT:
        // unregister a stat:
        // pStruct1 -- STATDIOC_resp_t;
        // pdwResult -- BOOL if any servers left
        *pdwResult = FALSE;
        { CStatServer* pServer = (CStatServer*)
                (PSTATDIOC_resp_t(pStruct1)->hServerCookie);
          if (pServer && pServer->IsGood()) 
            RemoveStat(pServer, 
                     PSTATDIOC_resp_t(pStruct1)->hStatCookie,
                     PSTATDIOC_resp_t(pStruct1)->index
                     );
          else
            break; // bad descriptor?
        }
        // now, indicate if any servers left:
        if (!CStatServer::First()) *pdwResult = TRUE;
        break;
     default:
        return 1;   // err
    } /* switch */
    return 0;
}

CStatServer* StatproxDevice::GetServerNode(const char* szServer) 
{
     // Find existing or create a new one
    CStatServer *pnode=CStatServer::First();
    while (pnode) {
        if (strcmp(pnode->Name(), szServer)==0)
            break;
        else
            pnode = CStatServer::Next(pnode);
    }
    if (!pnode) {  
        pnode = new CStatServer(szServer);
        pnode->Append();
    }
    return pnode;
}

STATHANDLE StatproxDevice::AddStat(CStatServer* pServer,
     const char* szName, 
     const char* szDescription,
     BOOL bValueNotRate, int* pindex)
{
     int ix = m_pPerfHeap->Allocate();  // alloc a dword
     if (ix>=0) {
         STATHANDLE hStat =
             pServer->AddStat(
                szName,
                szDescription,
                m_pPerfHeap->GetAddr(ix),
                bValueNotRate);
         if (hStat) {   // OK!
            *pindex = ix;
            return hStat;
         }
      }
      return NULL;
}

void StatproxDevice::RemoveStat(CStatServer* pServer,
                        const STATHANDLE hStat, const int index)
{
    m_pPerfHeap->Free(index);           // free the dword
    pServer->RemoveStat(hStat);

    // now, check if this server deserves to live any longer
    if (!pServer->IsReferenced()) {
        pServer->Remove();          // out of the list
        delete pServer;             // kill     
    }       
}

// end of file
