/////////////////////////////////////////////////////////////
//
// AppStats.c   -   Main module of APPSTATS.DLL: implements 
//                  an API to SYSMON for Win32 applications.
//
// 12/16/96     VV  Original
//
////////////////////////////////////////////////////////////
#define     STRICT
#include    <windows.h>
#include    <malloc.h>

#define     _APPSTATS_IMPLEMENTATION_

#include    "AppStats.h"
#include    "StatDioc.h"

// Internal Statistics item descriptor:
typedef struct HSTAT_Tag_t {
    struct HSTAT_Tag_t* pNext;  // for linked list 
    PDWORD  pMonitoree;         // ptr to the monitored dword
    STATDIOC_resp_t idStat;     // stats identification
} HSTAT_t, *PHSTAT_t;

///////////////////////////////////////////////////////
// Internal implementation
///////////////////////////////////////////////////////

// Static data:
static char     s_szDevice[] = "STATPROX";      // proxy vxd name
static HANDLE   s_hProxyDev = INVALID_HANDLE_VALUE; // vxd's handle
static HANDLE   s_hMappedFile = INVALID_HANDLE_VALUE; // memmap file 
static PDWORD   s_pArea = NULL; // shared area (stats "heap") 
static PHSTAT_t s_pStatsList=NULL; // active statistics list
static HSTAT_t  s_Dummy={0, (PDWORD)&s_Dummy.idStat}; // a dummy 
static char     s_szDefaultCategory[MAX_STATS_NAME]="?"; 

//////////////////////////////////////////////////////
// Initialize():    Attach to the driver and create
// ============     shared stats "heap"
void Initialize(void)
{
    char szDevPath[2][MAX_PATH]; 
    int i;
    BOOL bPassMapToDev;

    // Try to open the device: either as a pre-loaded or 
    // or a dynamic one:
    wsprintf(szDevPath[0], "\\\\.\\%s", s_szDevice);
    wsprintf(szDevPath[1], "\\\\.\\%s.VXD", s_szDevice);

    for (i=0; i<2; i++)
      if ((s_hProxyDev = CreateFile((LPCSTR)szDevPath[i],
                GENERIC_READ | GENERIC_WRITE,
                FILE_SHARE_READ | FILE_SHARE_WRITE,
                NULL, OPEN_ALWAYS, FILE_FLAG_DELETE_ON_CLOSE, 0))
                    !=INVALID_HANDLE_VALUE) break;
    if (s_hProxyDev==INVALID_HANDLE_VALUE)
            return;             

    // The device is available, create named memory-mapped "file" -
    // shared area between the VxD and the app.
    s_hMappedFile = CreateFileMapping((HANDLE)-1, 
                NULL, PAGE_READWRITE, 0, 4096, "APPSTATS_HEAP");
    if (s_hMappedFile==NULL) {              // failed, return
            CloseHandle(s_hProxyDev);
            s_hMappedFile=s_hProxyDev=INVALID_HANDLE_VALUE;
            return;
        }
    
    // Ok, now mark if it was the first file with this name;
    // we might use this fact a few lines below
    bPassMapToDev = GetLastError() != ERROR_ALREADY_EXISTS;
    
    // map the area into to the current process's space
    s_pArea = (PDWORD) MapViewOfFile(s_hMappedFile, 
                            FILE_MAP_ALL_ACCESS, 0,0,0);    

    // pass the "heap" area to the device on the very creation only:
    if (bPassMapToDev) 
        DRIVER_REQUEST(s_hProxyDev, STATDIOC_INIT, 
            s_pArea, s_pArea, s_pArea);

    // Retrieve the process file name to use as a default 
    // "Category" on statCreate():
    GetModuleFileName(NULL, szDevPath[0], MAX_PATH);
    GetFileTitle(szDevPath[0], 
        s_szDefaultCategory, sizeof(s_szDefaultCategory));
}

//////////////////////////////////////////////////////
// Deinitialize():  Detach from the driver and destroy
// ==============   the stats "heap"
void Deinitialize(void)
{
    UnmapViewOfFile(s_pArea); s_pArea=NULL;
    CloseHandle(s_hMappedFile); 
    CloseHandle(s_hProxyDev);
    s_hMappedFile=s_hProxyDev=INVALID_HANDLE_VALUE;
}

/////////////////////////////////////////////////////
// DllMain():   Standard Win32 DLL entry/exit point
// ==========
#if defined(__BORLANDC__)
#   pragma argsused
#endif
BOOL WINAPI DllMain(HINSTANCE hinstDll, DWORD fdwReason, LPVOID p)
{
    switch (fdwReason) {
        case DLL_PROCESS_ATTACH:    
            // Initialize resources: 
            Initialize();
            break;

        case DLL_PROCESS_DETACH: 
            // Destroy all existing statistics, release resources:
            while (s_pStatsList)
                statDestroy((HSTAT)s_pStatsList);
            Deinitialize();
            break;
    }
    return TRUE;
}


// User API implementation

///////////////////////////////////////////////////////////
// statCreate():    Create a SYSMON statistics item
// ============
// Parameters:
//      szName:     Statistics Name
//      stType:     Monitoring mode: value or change rate
//      szDescription: Optional description string (or NULL)        
//      szCategory: Optional Category string, (or NULL)
// Returns:
//      Stats Handle. NOTE: it could be a dummy one, never NULL.
APPSTATS_API HSTAT __stdcall statCreate(const char* szName,
                              statsmonitor_t stType,
                              const char* szDescription,
                              const char* szCategory)
{
    STATDIOC_req_t req;
    PHSTAT_t pStat = malloc(sizeof(HSTAT_t));

    req.pszName         = szName ? szName : "?";
    req.pszDescription  = szDescription ? szDescription : "?";
    req.pszSubsystem    = szCategory ? szCategory : s_szDefaultCategory;
    req.bValueNotRate   = (stType == MONITOR_VALUE);

    // if descriptor allocation succeeded, register the stat with
    // the driver, and insert the descriptor into the list. If the
    // VxD is not present use an internal area to store the stat:
    
    if (pStat) {
        if (statIsInstalled()) {
              DRIVER_REQUEST(s_hProxyDev, 
                STATDIOC_REGISTER_STAT, &req,
                    &pStat->idStat, (LPDWORD)&pStat->idStat);
              if (pStat->idStat.index>=0) 
                pStat->pMonitoree = s_pArea + pStat->idStat.index; 
         }
         else
                pStat->pMonitoree = (PDWORD)&pStat->idStat;
         pStat->pNext = s_pStatsList;
         s_pStatsList = pStat;
         return (HSTAT)pStat;       // OK!
    }
    else                            // memory problem, use dummy
        return (HSTAT)&s_Dummy;
}

///////////////////////////////////////////////////////////
// statDestroy():   Destroy a SYSMON statistics item
// =============
// Parameters:
//      hStat:      Statistics handle
// Returns:
//      TRUE if success, FALSE - bad handle
APPSTATS_API BOOL __stdcall statDestroy(const HSTAT hStat)
{
    PHSTAT_t pStat = (PHSTAT_t)hStat;

    // Remove the stat descriptor from the list and
    // deregister it with the driver:
    if (!pStat) return FALSE;
    if (s_pStatsList==pStat)
        s_pStatsList = s_pStatsList->pNext;
    else {  PHSTAT_t p = s_pStatsList;
        while (p)
            if (p->pNext == pStat) 
               { p->pNext = pStat->pNext; break; }
            else p = p->pNext;
         if (!p)  return FALSE; // bad handle
    }

    if (statIsInstalled())
        DRIVER_REQUEST(s_hProxyDev, STATDIOC_UNREGISTER_STAT,
          &pStat->idStat, &pStat->idStat, (LPDWORD)&pStat->pNext);      

    free(pStat); 
    return TRUE;
}


///////////////////////////////////////////////////////////
// statSet()/Get(): Assign/retrieve the value to/of
// ===============  a statistics item.
// Parameters:
//      hStat:      Statistics handle
//      dwVal:      The value to assign
// Returns:
//      New value of the statistics item
APPSTATS_API DWORD __stdcall statSet(const HSTAT hStat, const DWORD dwVal)
{
    return *((PHSTAT_t)hStat)->pMonitoree = dwVal;
}

APPSTATS_API DWORD __stdcall statGet(const HSTAT hStat)
{
    return *((PHSTAT_t)hStat)->pMonitoree;
}

///////////////////////////////////////////////////////////
// statInc/Dec():   Increment/Decrement a statistics item
// =============
// Parameters:
//      hStat:      Statistics handle
//      dwVal:      The value of increment/decrement
// Returns:
//      New value of the statistics item
APPSTATS_API DWORD __stdcall statInc(const HSTAT hStat, 
									 const DWORD dwVal)
{
    return *((PHSTAT_t)hStat)->pMonitoree +=  dwVal;
}

APPSTATS_API DWORD __stdcall statDec(const HSTAT hStat,
									 const DWORD dwVal)
{
    return *((PHSTAT_t)hStat)->pMonitoree -=  dwVal;
}


///////////////////////////////////////////////////////////
// statIsInstalled():   Checks if the AppStats subsystem
// =================    is available (the proxy VxD is loaded).
// Parameters:
//      none
// Returns:
//      TRUE if the AppStats is available. FALSE would mean that
//      the statistics won't be seen at SYSMON; the AppStats API is
//      still safe to use, though,- no conditional code is required
APPSTATS_API BOOL __stdcall statIsInstalled(void)
{ return s_hProxyDev != INVALID_HANDLE_VALUE; }
