/*
 * %W% %E% %U%  [EXTREL_1.2]
 *
 * VersaTrack orbit calculations are based on those that appear in Dr. Manfred
 * Bester's sattrack program (the Unix(tm) versions 1 and 2).
 *
 * The data from which the maps where generated come from "xsat", an
 * X-Windows program by David A. Curry (N9MSW).
 *
 * Site coordinates come from various sources, including a couple of
 * World Almanacs, and also from both of the programs mentioned above.
 *
 * The following are authors' applicable copyright notices:
 *
 *                                                                               
 * Copyright (c) 1992, 1993, 1994 Manfred Bester. All Rights Reserved.        
 *                                                                           
 * Permission to use, copy, modify, and distribute this software and its      
 * documentation for educational, research and non-profit purposes, without   
 * fee, and without a written agreement is hereby granted, provided that the  
 * above copyright notice and the following three paragraphs appear in all    
 * copies.                                                                    
 *                                                                              
 * Permission to incorporate this software into commercial products may be    
 * obtained from the author, Dr. Manfred Bester, 1636 M. L. King Jr. Way,     
 * Berkeley, CA 94709, USA.                                                   
 *                                                                             
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,  
 * SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF    
 * THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE AUTHOR HAS BEEN ADVISED   
 * OF THE POSSIBILITY OF SUCH DAMAGE.                                         
 *                                                                             
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT       
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A    
 * PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"       
 * BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,  
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.                                   
 *                                                                             
 *                                                                             
 * Copyright 1992 by David A. Curry                                            
 *                                                                             
 * Permission to use, copy, modify, distribute, and sell this software and its 
 * documentation for any purpose is hereby granted without fee, provided that  
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation.  The  
 * author makes no representations about the suitability of this software for  
 * any purpose.  It is provided "as is" without express or implied warranty.   
 *                                                                             
 * David A. Curry, N9MSW                                                       
 * Purdue University                                                           
 * Engineering Computer Network                                                
 * 1285 Electrical Engineering Building                                        
 * West Lafayette, IN 47907                                                    
 * davy@ecn.purdue.edu                                                         
 *                                                                             
 * VersaTrack Copyright (c) 1993, 1994 Siamack Navabpour. All Rights Reserved.
 *
 * Permission is hereby granted to copy, modify and distribute VersaTrack
 * in whole, or in part, for educational, non-profit and non-commercial use
 * only, free of charge or obligation, and without agreement, provided that
 * all copyrights and restrictions noted herein are observed and followed, and
 * additionally, that this and all other copyright notices listed herein
 * appear unaltered in all copies and in all derived work.
 *
 * This notice shall not in any way void or supersede any of the other authors
 * rights or privileges.
 *
 * VersaTrack IS PRESENTED FREE AND "AS IS", WITHOUT ANY WARRANTY OR SUPPORT.
 * YOU USE IT AT YOUR OWN RISK. The author(s) shall not be liable for any
 * direct, indirect, incidental, or consequential damage, loss of profits or
 * other tangible or intangible losses or benefits, arising out of or related
 * to its use. VersaTrack carries no warranty, explicit or implied, including
 * but not limited to those of merchantablity and fitness for a particular
 * purpose.
 *
 * Siamack Navabpour, 12342 Hunter's Chase Dr. Apt. 2114, Austin, TX 78729.
 * sia@bga.com or sia@realtime.com.
 */


 /* FOLLOWING ADAPTED FROM Sattrack (the Unix Version by Dr. M. Bester) */

#include <windows.h>
#include <math.h>
#include <stdlib.h>

#include "vstdefs.h"
#include "constant.h"
#include "vsttype.h"
#include "vstextrn.h"

/*
 * eqxnutation: finds nutation in longitude and obliquity, and the equation
 *           of the equinoxes.
 */
 
double
eqxnutation(tp)
track_t *tp;
{
    double tu, tusq, tucb, eps, l, lp, f, d, o;
    double totPsi, totEps;
    extern void nuteseries(double, double, double, double, double,
        double, double *, double *);

    tu   = (tp->juliandate - JULDAT2000) / JULCENT;
    tusq = tu*tu;
    tucb = tusq*tu;
    /*
     * calculate the mean obliquity of the ecliptic (eps), based on the
     * epoch J2000.0
     *
     * For reference see the "Astronomical Almanac", 1986, pages B24 - B31.
     * The equation for calculation of eps has been taken from the
     * "Astronomical Almanac", 1984, page S21 (in the Supplement section)
     */
     
    eps = (84381.448 - 46.815*tu - 0.00059*tusq + 0.001813*tucb) * CASR;

    /*
     * compute the nutation in longitude (totPsi) and in obliquity (totEps)
     * thus giving the true obliquity of the ecliptic (trueEps) and from
     * that, the equation of the equinoxes, also based on the epoch J2000.0.
     *
     * Calculation of the nutation parameters totPsi and totEps is performed
     * by using the "1980 IAU Theory of Nutation" (see "Astronomical
     * Almanac", 1984, pp. S21-S26)
     *
     * Fundamental arguments of the Sun and the Moon are : l, lp, f, d, o
     * (time-dependent arguments) from the "Astronomical Almanac", 1984,
     * p. S26. The function reduce() reduces their values into the interval
     * (0, 2pi). The nutation in longitude (totPsi) and in obliquity (totEps)
     * are updated when the update time interval is elapsed.
     */
     
    l   = reduce ( (485866.733 + 1717915922.633 * tu + 31.310 * tusq +
          0.064 * tucb) * CASR, ZERO, TWOPI);
 
    lp  = reduce ( (1287099.804 +  129596581.224 * tu - 0.577 *  tusq -
          0.012 * tucb) * CASR, ZERO, TWOPI);

    f   = reduce ( (335778.877 + 1739527263.137 * tu -13.257 * tusq +
          0.011 * tucb) * CASR, ZERO, TWOPI);

    d   = reduce ( (1072261.307 + 1602961601.328 * tu - 6.891 * tusq +
          0.019 * tucb) * CASR, ZERO, TWOPI);

    o   = reduce ( (450160.280 - 6962890.539 * tu + 7.455 * tusq +
          0.008*tucb) * CASR, ZERO, TWOPI);

    nuteseries(tu, l, lp, f, d, o, &totPsi, &totEps);

    return totPsi * cos(eps + totEps);                     /* [rad] */
}

/*
 *  nuteseries: calculates sine and cosine series of nutation
 */

static void
nuteseries(tu,l,lp,f,d,o, Psip, Epsp)
double tu, l, lp, f, d, o;
double *Psip, *Epsp;
{
    /* BEWARE STACK OVERFLOW */
    double b[106], sinb[106], c[106], cosc[106], bc[106];
    double sip, psp;
    int n;

    /*
     * Coefficients and arguments of the sine and cosine terms for the
     * Fourier Series representation (106 terms) of the nutation in longitude
     * (sine) and of the nutation in obliquity (cosine), respectively.
     * All known long and short-term variations are included.
     *
     * totEps  = b(0)*cos(B(0)) + b(1)*cos(B(1)) + ... + b(105)*cos(B(105))
     *
     * with b(n) = aat(n,0) + aat(n,1)*tu  for n = [0..14]   (time-dependent)
     * and  B(n) = at(n,0)*l + at(n,1)*lp + at(n,2)*f + at(n,3)*d + at(n,4)*o
     *
     * totPsi    = c(0)*sin(C(0)) + c(1)*sin(C(1)) + ... + c(105)*sin(C(105))
     *
     * with c(n) = aat(n,1) + aat(n,2)*tu  for n = [0..14]  (time-dependent)
     * and  c(n) = aa(n,1)              for n = [15..105] (time-independent)
     * and  C(n) = B(n)
     *
     *
     * First: 15 terms with time-dependent arguments (either nutation in
     * longitude or in obliquity is time-dependent.)
     *
     * The data field at[][] contains the code for the linear combinations
     * of the arguments of Sun and Moon (l, lp, f, d, o)
     */
     
    static double at[15][5] = {
         {  0,  0,  0,  0,  1},
         {  0,  0,  0,  0,  2},
         {  0,  0,  2, -2,  2},
         {  0,  1,  0,  0,  0},
         {  0,  1,  2, -2,  2},
         {  0, -1,  2, -2,  2},
         {  0,  0,  2, -2,  1},
         {  0,  2,  0,  0,  0},
         {  0,  2,  2, -2,  2},
         {  0,  0,  2,  0,  2},
         {  1,  0,  0,  0,  0},
         {  0,  0,  2,  0,  1},
         {  1,  0,  2,  0,  2},
         {  1,  0,  0,  0,  1},
         { -1,  0,  0,  0,  1}
    };

    static double aat[15][4] = {
         { -171996,  -174.2,   92025,     8.9},
         {    2062,     0.2,    -895,     0.5},
         {  -13187,    -1.6,    5736,    -3.1},
         {    1426,    -3.4,      54,    -0.1},
         {    -517,     1.2,     224,    -0.6},
         {     217,    -0.5,     -95,     0.3},
         {     129,     0.1,     -70,     0.0},
         {      17,    -0.1,       0,     0.0},
         {     -16,     0.1,       7,     0.0},
         {   -2274,    -0.2,     977,    -0.5},
         {     712,     0.1,      -7,     0.0},
         {    -386,    -0.4,     200,     0.0},
         {    -301,     0.0,     129,    -0.1},
         {      63,     0.1,     -33,     0.0},
         {     -58,    -0.1,      32,     0.0}
    };

    /*
     * Second : 91 terms with non-time dependent arguments
     */
     
    static double a[91][5] = {
         { -2,  0,  2,  0,  1},
         {  2,  0, -2,  0,  0},
         { -2,  0,  2,  0,  2},
         {  1, -1,  0, -1,  0},
         {  0, -2,  2, -2,  1},
         {  2,  0, -2,  0,  1},
         {  2,  0,  0, -2,  0},
         {  0,  0,  2, -2,  0},
         {  0,  1,  0,  0,  1},
         {  0, -1,  0,  0,  1},
         { -2,  0,  0,  2,  1},
         {  0, -1,  2, -2,  1},
         {  2,  0,  0, -2,  1},
         {  0,  1,  2, -2,  1},
         {  1,  0,  0, -1,  0},
         {  2,  1,  0, -2,  0},
         {  0,  0, -2,  2,  1},
         {  0,  1, -2,  2,  0},
         {  0,  1,  0,  0,  2},
         { -1,  0,  0,  1,  1},
         {  0,  1,  2, -2,  0},
         {  1,  0,  0, -2,  0},
         { -1,  0,  2,  0,  2},
         {  0,  0,  0,  2,  0},
         { -1,  0,  2,  2,  2},
         {  1,  0,  2,  0,  1},
         {  0,  0,  2,  2,  2},
         {  2,  0,  0,  0,  0},
         {  1,  0,  2, -2,  2},
         {  2,  0,  2,  0,  2},
         {  0,  0,  2,  0,  0},
         { -1,  0,  2,  0,  1},
         { -1,  0,  0,  2,  1},
         {  1,  0,  0, -2,  1},
         { -1,  0,  2,  2,  1},
         {  1,  1,  0, -2,  0},
         {  0,  1,  2,  0,  2},
         {  0, -1,  2,  0,  2},
         {  1,  0,  2,  2,  2},
         {  1,  0,  0,  2,  0},
         {  2,  0,  2, -2,  2},
         {  0,  0,  0,  2,  1},
         {  0,  0,  2,  2,  1},
         {  1,  0,  2, -2,  1},
         {  0,  0,  0, -2,  1},
         {  1, -1,  0,  0,  0},
         {  2,  0,  2,  0,  1},
         {  0,  1,  0, -2,  0},
         {  1,  0, -2,  0,  0},
         {  0,  0,  0,  1,  0},
         {  1,  1,  0,  0,  0},
         {  1,  0,  2,  0,  0},
         {  1, -1,  2,  0,  2},
         { -1, -1,  2,  2,  2},
         { -2,  0,  0,  0,  1},
         {  3,  0,  2,  0,  2},
         {  0, -1,  2,  2,  2},
         {  1,  1,  2,  0,  2},
         { -1,  0,  2, -2,  1},
         {  2,  0,  0,  0,  1},
         {  1,  0,  0,  0,  2},
         {  3,  0,  0,  0,  0},
         {  0,  0,  2,  1,  2},
         { -1,  0,  0,  0,  2},
         {  1,  0,  0, -4,  0},
         { -2,  0,  2,  2,  2},
         { -1,  0,  2,  4,  2},
         {  2,  0,  0, -4,  0},
         {  1,  1,  2, -2,  2},
         {  1,  0,  2,  2,  1},
         { -2,  0,  2,  4,  2},
         { -1,  0,  4,  0,  2},
         {  1, -1,  0, -2,  0},
         {  2,  0,  2, -2,  1},
         {  2,  0,  2,  2,  2},
         {  1,  0,  0,  2,  1},
         {  0,  0,  4, -2,  2},
         {  3,  0,  2, -2,  2},
         {  1,  0,  2, -2,  0},
         {  0,  1,  2,  0,  1},
         { -1, -1,  0,  2,  1},
         {  0,  0, -2,  0,  1},
         {  0,  0,  2, -1,  2},
         {  0,  1,  0,  2,  0},
         {  1,  0, -2, -2,  0},
         {  0, -1,  2,  0,  1},
         {  1,  1,  0, -2,  1},
         {  1,  0, -2,  2,  0},
         {  2,  0,  0,  2,  0},
         {  0,  0,  2,  4,  2},
         {  0,  1,  0,  1,  0}
    };

    static double aa[91][2] = {
         {   46,   -24},
         {   11,     0},
         {   -3,     1},
         {   -3,     0},
         {   -2,     1},
         {    1,     0},
         {   48,     1},
         {  -22,     0},
         {  -15,     9},
         {  -12,     6},
         {   -6,     3},
         {   -5,     3},
         {    4,    -2},
         {    4,    -2},
         {   -4,     0},
         {    1,     0},
         {    1,     0},
         {   -1,     0},
         {    1,     0},
         {    1,     0},
         {   -1,     0},
         { -158,    -1},
         {  123,   -53},
         {   63,    -2},
         {  -59,    26},
         {  -51,    27},
         {  -38,    16},
         {   29,    -1},
         {   29,   -12},
         {  -31,    13},
         {   26,    -1},
         {   21,   -10},
         {   16,    -8},
         {  -13,     7},
         {  -10,     5},
         {   -7,     0},
         {    7,    -3},
         {   -7,     3},
         {   -8,     3},
         {    6,     0},
         {    6,    -3},
         {   -6,     3},
         {   -7,     3},
         {    6,    -3},
         {   -5,     3},
         {    5,     0},
         {   -5,     3},
         {   -4,     0},
         {    4,     0},
         {   -4,     0},
         {   -3,     0},
         {    3,     0},
         {   -3,     1},
         {   -3,     1},
         {   -2,     1},
         {   -3,     1},
         {   -3,     1},
         {    2,    -1},
         {   -2,     1},
         {    2,    -1},
         {   -2,     1},
         {    2,     0},
         {    2,    -1},
         {    1,    -1},
         {   -1,     0},
         {    1,    -1},
         {   -2,     1},
         {   -1,     0},
         {    1,    -1},
         {   -1,     1},
         {   -1,     1},
         {    1,     0},
         {    1,     0},
         {    1,    -1},
         {   -1,     0},
         {   -1,     0},
         {    1,     0},
         {    1,     0},
         {   -1,     0},
         {    1,     0},
         {    1,     0},
         {   -1,     0},
         {   -1,     0},
         {   -1,     0},
         {   -1,     0},
         {   -1,     0},
         {   -1,     0},
         {   -1,     0},
         {    1,     0},
         {   -1,     0},
         {    1,     0}
    };

    /*
     * Calculate the linear combinations of the arguments of Sun and Moon
     *
     * B(n) = C(n) = bc(n) = ....
     */
     
    for (n = 0; n <= 14; n++) {
        bc[n] = at[n][0]*l + at[n][1]*lp + at[n][2]*f + at[n][3]*d + at[n][4]*o;
        bc[n] = reduce( bc[n], ZERO, TWOPI);
    }

    for (n = 15; n <= 105; n++) {
        bc[n] = a[n-15][0]*l + a[n-15][1]*lp + a[n-15][2]*f
                             + a[n-15][3]*d  + a[n-15][4]*o;
        bc[n] = reduce( bc[n], ZERO, TWOPI);
    }

    /*
     * Calculate the nutation in longitude (totPsi) (sine series) and
     * calculate the nutation in obliquity (totEps) (cosine series)
     */
     
    for (n = 0; n <= 14; n++) {
        b[n]    = aat[n][0] + aat[n][1]*tu;
        c[n]    = aat[n][2] + aat[n][3]*tu;
    }

    for (n = 15; n <= 105; n++) {
        b[n]    = aa[n-15][0];
        c[n]    = aa[n-15][1];
    }

    for (n = 0; n <= 105; n++) {
        sinb[n] = sin (bc[n]);
        cosc[n] = cos (bc[n]);
    }

    sip = 0.0;
    psp = 0.0;

    for (n = 0; n <= 105; n++) {
        sip += b[n] * sinb[n];
        psp += c[n] * cosc[n];
    }

    *Psip = sip * (CASR / 10000.0);                                /* [rad] */
    *Epsp = psp * (CASR / 10000.0);                                /* [rad] */
}
