//  $Id: mts.cc 1.16 1996/11/09 18:36:52 hardy Exp $
//
//  This progam/module was written by Hardy Griech based on ideas and
//  pieces of code from Chin Huang (cthuang@io.org).  Bug reports should
//  be submitted to rgriech@ibm.net.
//
//  This file is part of soup++ for OS/2.  Soup++ including this file
//  is freeware.  There is no warranty of any kind implied.  The terms
//  of the GNU Gernal Public Licence are valid for this piece of software.
//
//  Problems
//  --------
//  -  the emx09c semaphores are likely to deadlock, because the signal handler
//     is illegal (streams!)...
//  -  sprintf/vsprintf are thread-safe functions and reentrant
//  -  sometimes the functions in this module are calling directly library functions.
//     Those functions are placed here to indicate that they are checked with respect
//     to thread-safety
//  -  all IO should be done without streams, see class TFile!  Although this class
//     seems to be not so nice as e.g. streams, it is doing the job quite well.  Be
//     cautious, if you want to use this class for other projects...
//     E.g. simultaneous access to one TFile is not possible.  If there is a gets()
//     active and the file will be closed at the same time, an access violation
//     will happen (most likely...).  Anyway this type of operation is undefined!
//


#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <io.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "mts.hh"
#include "sema.hh"



//--------------------------------------------------------------------------------



TFile::TFile( void )
{
    mode  = mnotopen;
    fname = NULL;
}   // TFile::TFile



TFile::~TFile()
{
    close();
}   // TFile::~TFile



void TFile::close( int killif0, int forceKill )
{
#ifdef TRACE_ALL
    printfT( "TFile::close(%d,%d) %s\n",killif0,forceKill,fname );
#endif

    if (mode != mnotopen) {
        int kill = (killif0  &&  (::tell(handle) == 0))  ||  forceKill;
	::close( handle );
	if (kill)
	    ::remove( fname );
    }

    if (fname != NULL) {
	delete fname;
	fname = NULL;
    }
    if (mode == mread  ||  mode == mreadwrite) {
	delete buffer;
	buffer = NULL;
	buffSize = 0;
	invalidateBuffer();
    }
    mode   = mnotopen;
    handle = -1;
}   // TFile::close



void TFile::remove( void )
{
    close( 1,1 );
}   // TFile::remove



int TFile::open( const char *name, TMode mode, OMode textmode, int create )
//
//  in case of mwrite, the file pointer is positioned at end of file
//  returns 0, if not successful
//
{
    int flag;

#ifdef TRACE_ALL
    printfT( "TFile::open(%s,%d,%d,%d)\n",name,mode,textmode,create );
#endif

    if (TFile::mode != mnotopen)
	close(0,0);

    if (textmode == otext)
	flag = O_TEXT;
    else
	flag = O_BINARY;

    if (mode == mread)
	flag |= O_RDONLY;
    else if (mode == mwrite)
	flag |= O_WRONLY;
    else if (mode == mreadwrite)
	flag |= O_RDWR;
    else {
	errno = EPERM;
	return 0;
    }
    
    if (create)
	flag |= O_CREAT | O_TRUNC;

    handle = ::open( name, flag, S_IREAD|S_IWRITE );
    if (handle < 0)
	return 0;

    if (mode == mwrite)
	seek( 0,SEEK_END );
    
    TFile::mode = mode;
    xstrdup( &fname, name );
    buffSize = 0;
    buffer = NULL;
    if (mode == mread  ||  mode == mreadwrite) {
	buffSize = 4096;
	buffer = new unsigned char[buffSize+10];
    }
    invalidateBuffer();
    return 1;
}   // TFile::open



int TFile::openTmpfile( const char *pattern, TMode mode, OMode textmode )
{
    char *tname;

    for (;;) {
	if ((tname = tempnam(NULL,pattern)) == NULL)
	    return 0;
	if (open(tname,mode,textmode,1)) {
	    free( tname );
	    break;
	}
	free( tname );
    }
    return 1;
}   // TFile::openTmpfile



int TFile::truncate( long length )
{
    int res = ::ftruncate( handle, length );
    seek( 0, SEEK_END );
    return res;
}   // TFile::truncate



long TFile::seek( long offset, int origin )
{
#ifdef TRACE_ALL
    printfT( "TFile::seek(%ld,%d)\n",offset,origin );
#endif
    invalidateBuffer();
    return ::lseek( handle,offset,origin );
}   // TFile::seek



long TFile::tell( void )
{
    return ::tell( handle );
}   // TFile::tell



int TFile::write( const void *buff, int bufflen )
{
    if (bufflen <= 0)
	return 0;
    invalidateBuffer();
    return ::write( handle,buff,bufflen );
}   // TFile::write



int TFile::putcc( char c )
{
    return (write(&c,1) == 1) ? 1 : EOF;
}   // TFile::putc



int TFile::fputs( const char *s )
{
    int len, res;

    len = strlen(s);
    res = write( s,len );
    return (res == len) ? len : EOF;
}   // TFile::fputs



int TFile::printf( const char *fmt, ... )
{
    va_list ap;
    char buf[BUFSIZ];
    int  buflen;

    va_start( ap, fmt );
    buflen = vsprintf( buf, fmt, ap );
    va_end( ap );
    return write( buf,buflen );
}   // TFile::printf



int TFile::vprintf( const char *fmt, va_list arg_ptr )
{
    char buf[BUFSIZ];
    int buflen;

    buflen = vsprintf( buf,fmt,arg_ptr );
    return write( buf,buflen );
}   // TFile::vprintf



int TFile::read( void *buff, int bufflen )
{
    invalidateBuffer();
    return ::read( handle,buff,bufflen );
}   // TFile::read



int TFile::fillBuff( void )
{
    int blk;
    int toRead;

#ifdef TRACE_ALL
    printfT( "TFile::fillBuff()\n" );
#endif

    if (buffEof)
	return buffNdx < buffEnd;

    if (buffEnd-buffNdx >= buffSize/2)
	return 1;

#ifdef TRACE_ALL
    printfT( "TFile::fillBuff() -- 2\n" );
#endif

    memmove( buffer, buffer+buffNdx, buffEnd-buffNdx );
    buffEnd = buffEnd - buffNdx;
    buffNdx = 0;
    toRead = (buffSize-buffEnd) & 0xfc00;
    blk = ::read( handle, buffer+buffEnd, toRead );
    if (blk > 0)
	buffEnd += blk;
    else
	buffEof = 1;            // -> nothing left to read!
    buffer[buffEnd] = '\0';

#ifdef DEBUG_ALL
    printfT( "TFile::fillBuff() = %d,%d,%d\n",blk,buffNdx,buffEnd );
#endif
    return buffEnd > 0;
}   // TFile::fillBuff



int TFile::getcc( void )
{
    if (buffNdx >= buffEnd) {
	if ( !fillBuff())
	    return -1;
    }
    return buffer[buffNdx++];
}   // TFile::getcc



char *TFile::fgets( char *buff, int bufflen, int skipCrLf )
//
//  '\0' will always be skipped!
//  If SkipCrLf, then also \r & \n are skipped
//
{
    char *p;
    int  n;
    int  r;

#ifdef TRACE_ALL
    printfT( "TFile::fgets(.,%d) %s\n",bufflen,fname );
#endif

    p = buff;
    n = 0;
    for (;;) {
	if (n >= bufflen-2) {     // if Buffer exhausted return (there is no \n at eob)
	    if (skipCrLf) {
		do
		    r = getcc();
		while (r > 0  &&  r != '\n');
	    }
	    break;
	}

	r = getcc();
	if (r < 0) {              // EOF!
	    if (n == 0) {
		*p = '\0';
		return( NULL );
	    }
	    else
		break;
	}
	else if (r != 0) {
	    if ( ! (skipCrLf  &&  (r == '\r'  ||  r == '\n'))) {
		*(p++) = (unsigned char)r;
		++n;
	    }
	    if (r == '\n')
		break;
	}
    }
    *p = '\0';
    return buff;
}   // TFile::fgets



#if 0   ////
int  TFile::scanf( const char *fmt, ... )
//
//  trick:  the internal read buffer is always kept at least half full,
//          therefor scanf can work just like sscanf!
//
{
    va_list ap;
    int fields;

#ifdef TRACE_ALL
    printfT( "TFile::scanf(%s,...)\n",fmt );
#endif

    if ( !fillBuff())
	return EOF;

    va_start( ap, fmt );
    fields = vsscanf( (char *)buffer+buffNdx, fmt, ap );  // jetzt mu noch aus fields Anzahl der Zeichen berechnet werden...
    va_end( ap );
    return fields;
}   // TFile::scanf
#endif



int TFile::scanf( const char *fmt, void *a1 )
//
//  Scan one argument via sscanf.  Extension of this function to any number
//  of arguments does not work, because "%n" must be added (and an argument...)
//  Perhaps one argument is also ok, because what happened if one wants to
//  scan two arguments, but the second failed (what will "%n" display?)
//  -  sollen besondere Zeichen gescannt werden, so sollte das immer mit "%[...]"
//     gemacht werden, da ein einzelnes Zeichen, da nicht pat die Zuweisung "%n"
//     verhindert!
//  trick:  the internal read buffer is always kept at least half full,
//          therefor scanf can work just like sscanf!
//
{
    int  fields;
    int  cnt;
    char fmt2[200];

#ifdef TRACE_ALL
    printfT( "TFile::scanf1(%s,...) %d,%d\n",fmt,buffNdx,buffEnd );
#endif

    if ( !fillBuff())
	return EOF;

    strcpy( fmt2,fmt );
    strcat( fmt2,"%n" );

    cnt = 0;
    fields = sscanf( (char *)buffer+buffNdx, fmt2, a1,&cnt );
#ifdef DEBUG_ALL
    printfT( "TFile::scanf1()='%.20s' %d,%d\n",buffer+buffNdx,cnt,fields );
#endif
    if (cnt == 0)
	fields = 0;
    else
	buffNdx += cnt;
    return fields;
}   // TFile::scanf



//--------------------------------------------------------------------------------



static TSemaphor sysSema;



int hprintfT( int handle, const char *fmt, ... )
{
    va_list ap;
    char buf[BUFSIZ];
    int  buflen;

    va_start( ap, fmt );
    buflen = vsprintf( buf, fmt, ap );
    va_end( ap );
    if (buflen > 0)
	buflen = write( handle,buf,buflen );
    return buflen;
}   // hprintfT



int hputsT( const char *s, int handle )
{
    int len, res;

    len = strlen(s);
    res = write( handle,s,len );
    return (res == len) ? len : EOF;
}   // hputsT



int sprintfT( char *dst, const char *fmt, ... )
{
    va_list ap;
    int  buflen;

    *dst = '\0';
    va_start( ap, fmt );
    buflen = vsprintf( dst, fmt, ap );
    va_end( ap );
    
    return buflen;
}   // sprintfT



int printfT( const char *fmt, ... )
{
    va_list ap;
    char buf[BUFSIZ];
    int  buflen;

    sysSema.Request();
    va_start( ap, fmt );
    buflen = vsprintf( buf, fmt, ap );
    va_end( ap );
    if (buflen > 0)
	write( STDOUT_FILENO, buf,buflen );
    sysSema.Release();
    return buflen;
}   // printfT



int vprintfT( const char *fmt, va_list arg_ptr )
{
    char buf[BUFSIZ];
    int buflen;

    sysSema.Request();
    buflen = vsprintf( buf,fmt,arg_ptr );
    if (buflen > 0)
	write( STDOUT_FILENO, buf,buflen );
    sysSema.Release();
    return buflen;
}   // vprintfT



int vsprintfT( char *dst, const char *fmt, va_list arg_ptr )
{
    int buflen;

    buflen = vsprintf( dst,fmt,arg_ptr );
    return buflen;
}   // vsprintfT



int sscanfT( const char *src, const char *fmt, ... )
{
    va_list ap;
    int fields;

    va_start( ap, fmt );
    fields = vsscanf( src, fmt, ap );
    va_end( ap );
    return fields;
}   // sscanfT



//--------------------------------------------------------------------------------



size_t fwriteT( const void *buffer, size_t size, size_t count, FILE *out )
{
    size_t res;

    sysSema.Request();
    res = fwrite( buffer,size,count,out );
    sysSema.Release();
    return res;
}   // fwriteT



FILE *popenT( const char *command, const char *mode )
{
    FILE *res;

#ifdef TRACE_ALL
    printfT( "popenT(%s,%s)\n", command,mode );
#endif

    sysSema.Request();
    res = popen( command,mode );
    sysSema.Release();
    return res;
}   // popenT



int pcloseT( FILE *io )
{
    int res;

#ifdef TRACE_ALL
    printfT( "pcloseT(%p)\n", io );
#endif

    sysSema.Request();
    res = pclose( io );
    sysSema.Release();
    return res;
}   // pcloseT



int removeT( const char *fname )
{
    int res;

    sysSema.Request();
    res = remove( fname );
    sysSema.Release();
    return res;
}   // removeT



int renameT( const char *oldname, const char *newname )
{
    int res;

#ifdef TRACE_ALL
    printfT( "renameT(%s,%s)\n",oldname,newname );
#endif

    sysSema.Request();
    res = rename( oldname,newname );
    sysSema.Release();
    return res;
}   // renameT



//--------------------------------------------------------------------------------



static TSemaphor regSema;



regexp *regcompT( const char *exp )
{
    regexp *res;

    regSema.Request();
    res = regcomp( exp );
    regSema.Release();
    return res;
}   // regcompT



int regexecT( const regexp *cexp, const char *target )
{
    int res;

    regSema.Request();
    res = regexec( cexp,target );
    regSema.Release();
    return res;
}   // regexecT



//--------------------------------------------------------------------------------



const char *xstrdup( const char *src )
{
    int len;
    char *dst;

    len = strlen(src)+1;
    dst = new char [len];
    if (dst != NULL)
	memcpy( dst,src,len );
    else {
	hprintfT( STDERR_FILENO,"\nxstrdup failed\nprogram aborted\n" );
	exit( 3 );
    }
    return dst;
}   // xstrdup



void xstrdup( const char **dst, const char *src )
{
    if (*dst != NULL)
	delete *dst;

    if (src != NULL)
	*dst = xstrdup( src );
    else
	*dst = NULL;
}   // xstrdup
