//  $Id: kill.cc 1.15 1996/11/09 18:36:00 hardy Exp $
//
//  This progam/module was written by Hardy Griech based on ideas and
//  pieces of code from Chin Huang (cthuang@io.org).  Bug reports should
//  be submitted to rgriech@ibm.net.
//
//  This file is part of soup++ for OS/2.  Soup++ including this file
//  is freeware.  There is no warranty of any kind implied.  The terms
//  of the GNU Gernal Public Licence are valid for this piece of software.
//
//  Kill file processing
//


#include <ctype.h>
#include <fcntl.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <regexp.h>
#include <unistd.h>

#include "kill.hh"
#include "mts.hh"



TKillFile::TKillFile( void )
{
    globalKill = groupKillList = cacheGroupKill = NULL;
    cacheGroupName = xstrdup("");
}   // TKillFile::TKillFile



void TKillFile::killGroup( Group *gp )
{
    Exp *ep1, *ep2;
    
    if (gp == NULL)
	return;

    ep1 = gp->expList;
    while (ep1 != NULL) {
	if (ep1->re != NULL) {
////	    delete ep1->re;
	}
	ep2 = ep1->next;
////	delete ep1;
	ep1 = ep2;
    }
////    delete gp->name;
////   delete gp;
}   // TKillFile::killGroup



TKillFile::~TKillFile()
{
    Group *gp1, *gp2;

////    delete cacheGroupName;

    killGroup( globalKill );
    gp1 = groupKillList;
    while (gp1 != NULL) {
	gp2 = gp1->next;
	killGroup( gp1 );
	gp1 = gp2;
    }
}   // TKillFile::~TKillFile



TKillFile::Exp *TKillFile::genRegExp(const char *searchIn, const char *searchFor)
{
    char exp[BUFSIZ];
    Exp *result;

    if (searchFor[0] == '\0')
	return NULL;

    if (stricmp(searchIn, "header") == 0) {
	strcpy(exp, searchFor);
    } else {
	strcpy( exp, "^" );
	strcat( exp, searchIn );
	strcat( exp, ":.*" );
	strcat( exp, searchFor );
    }

    result = new Exp;
    result->next = NULL;
    strlwr(exp);
    result->re = regcompT(exp);
    return result;
}   // TKillFile::genRegExp



void TKillFile::stripBlanks( char *line )
{
    char *p1, *p2;
    int  len;

    p1 = line;
    while (*p1 == ' '  ||  *p1 == '\t')
	++p1;
    p2 = line + strlen(line) - 1;
    while (p2 >= p1  &&  (*p2 == ' '  ||  *p2 == '\t'))
	--p2;
    len = p2-p1+1;
    if (len > 0) {
	memmove( line,p1,len );
	line[len] = '\0';
    }
    else
	line[0] = '\0';
}   // TKillFile::stripBlanks



int TKillFile::readLine( char *line, int n, TFile &inf, int &lineNum )
//
//  fetch the next line from file
//  blanks are stripped
//  blank lines & lines with '#' in the beginning are skipped
//  on EOF NULL is returned
//    
{
    for (;;) {
	if (inf.fgets(line,n,1) == NULL)
	    return 0;
	++lineNum;
	stripBlanks( line );
	if (line[0] != '\0'  &&  line[0] != '#')
	    break;
    }
    return 1;
}   // TKillFile::readLine



int TKillFile::readFile( const char *killFile, int maxLines )
//
//  Read kill file and compile regular expressions.
//  Return:  -1 -> file not found, 0 -> syntax error, 1 -> ok
//  Nicht so hanz das optimale:  besser wre es eine Zustandsmaschine
//  zusammenzubasteln...
//
{
    char buf[1000], name[1000], tmp[1000];
    char searchIn[1000], searchFor[1000];
    TFile inf;
    Group *pGroup, *pLastGroup;
    Exp *pExp, *pLastExp;
    char ok;
    int lineNum;

    globalKill = groupKillList = NULL;
    TKillFile::maxLines = maxLines;

    if ( !inf.open(killFile,TFile::mread,TFile::otext))
	return -1;

    sema.Request();

    pLastGroup = NULL;
    ok = 1;

    //
    //  read newsgroup name
    //
    lineNum = 0;
    while (ok  &&  readLine(buf,sizeof(buf),inf,lineNum)) {
#ifdef DEBUG_ALL
	printfT( "line: '%s'\n",buf );
#endif
	if (sscanfT(buf,"%s%s",name,tmp) == 1)
	    readLine(tmp,sizeof(tmp),inf,lineNum);

	if (tmp[0] != '{' || tmp[1] != '\0') {
	    ok = 0;
	    break;
	}

	if (stricmp(name, "all") == 0) {
	    //
	    //  Allocate global kill entry.
	    //
	    if (globalKill == NULL) {
		globalKill = new Group;
		globalKill->name = NULL;
		globalKill->expList = NULL;
	    }
	    pGroup = globalKill;
	} else {
	    //
	    //  Allocate group kill entry (if it does not exist already)
	    //
	    pGroup = getGroupP( name );
	    if (pGroup == NULL) {
		pGroup = new Group;
		pGroup->expList = NULL;
		pGroup->name = xstrdup(name);
		pGroup->next = NULL;

		if (pLastGroup == NULL)
		    groupKillList = pGroup;
		else
		    pLastGroup->next = pGroup;
		pLastGroup = pGroup;
	    }
	}

	//
	//  pLastExp points to last element of list or to NULL
	//
	if (pGroup->expList == NULL)
	    pLastExp = NULL;
	else {
	    for (pLastExp = pGroup->expList;  pLastExp->next != NULL;  pLastExp = pLastExp->next)
		;
	}

	//
	//  Read kill expressions until closing brace.
	//
	while (readLine(buf,sizeof(buf),inf,lineNum)) {
	    *searchIn = *searchFor = '\0';
	    sscanfT( buf,"%s%[^\n]",searchIn,searchFor );
	    stripBlanks( searchFor );

	    if (searchIn[0] == '}'  &&  searchIn[1] == '\0') {
		ok = (searchFor[0] == '\0');
		break;
	    }
	    if (searchFor[0] == '\0') {
		ok = 0;
		break;
	    }
#ifdef DEBUG_ALL
	    printfT( "search: '%s', '%s'\n",searchIn,searchFor );
#endif
	    if ((pExp = genRegExp(searchIn,searchFor)) == NULL) {
		ok = 0;
		break;
	    }

	    //
	    //  append entry to list
	    //
	    if (pLastExp == NULL)
		pGroup->expList = pExp;
	    else
		pLastExp->next = pExp;
	    pLastExp = pExp;
	}
    }
    sema.Release();

    inf.close();

    if (!ok)
	hprintfT( STDERR_FILENO, "error in kill file %s,\n\tsection %s, line %d\n",
		  killFile,name,lineNum);
    return ok;
}   // TKillFile::readFile



TKillFile::Group *TKillFile::getGroupP( const char *groupName )
//
//  return group kill for *groupName
//
{
    Group *p;
    Group *res;

#ifdef TRACE_ALL
//    printfT( "TKillFile::getGroupP(%s)\n",groupName );
#endif

    sema.Request();
    if (stricmp(groupName,cacheGroupName) != 0) {
	for (p = groupKillList; p != NULL; p = p->next) {
	    if (stricmp(p->name, groupName) == 0) {
		xstrdup( &cacheGroupName, p->name );
		cacheGroupKill = p;
		break;
	    }
	}
	if (p == NULL) {
	    xstrdup( &cacheGroupName, "" );
	    cacheGroupKill = NULL;
	}
    }
    res = cacheGroupKill;
    sema.Release();
    return res;
}   // TKillFile::getGroupP



int TKillFile::matchExp(Group *pGroup, const char *buf)
{
    Exp *pExp;

    if (pGroup == NULL)
	return 0;

    sema.Request();
    for (pExp = pGroup->expList; pExp != NULL; pExp = pExp->next) {
	if (regexecT(pExp->re, buf)) {
	    sema.Release();
	    return 1;
	}
    }
    sema.Release();
    return 0;
}   // TKillFile::matchExp



int TKillFile::matchLine( const char *groupName, const char *line )
//
//  Check if line matches kill criteria.
//  Return TRUE if article should be killed.
//
{
    char *buf;
    int match = 0;

    //
    //  remove trailing '\n' and make lower cases
    //
    buf = (char *)xstrdup( line );
    strlwr(buf);

    //
    //  article too long ?
    //
    if (maxLines > 0 && strncmp(buf, "lines: ", 7) == 0) {
	match = (atoi(buf+7) > maxLines);
    }

    if ( !match) {
	//
	//  is there any match with the killFile ?
	//
	match = matchExp(globalKill, buf)  ||  matchExp(getGroupP(groupName), buf);
    }
    delete buf;
    return match;
}   // TKillFile::matchLine



int TKillFile::doKillQ( const char *groupName )
{
    return maxLines > 0  ||  globalKill != NULL  ||  getGroupP(groupName) != NULL;
}   // doKillQ
