/* $Id: pop3.cc 1.9 1996/09/01 12:11:23 hardy Exp $
 *
 * This module has been modified for souper.
 */

/* Copyright 1993,1994 by Carl Harris, Jr.
 * All rights reserved
 *
 * Distribute freely, except: don't remove my name from the source or
 * documentation (don't take credit for my work), mark your changes (don't
 * get me blamed for your possible bugs), don't alter or remove this
 * notice.  May be sold if buildable source is provided to buyer.  No
 * warrantee of any kind, express or implied, is included with this
 * software; use at your own risk, responsibility for damages (if any) to
 * anyone resulting from the use of this software rests entirely with the
 * user.
 *
 * Send bug reports, bug fixes, enhancements, requests, flames, etc., and
 * I'll try to keep a version up to date.  I can be reached as follows:
 * Carl Harris <ceharris@vt.edu>
 */

//
//  This progam/module was written by Hardy Griech based on ideas and
//  pieces of code from Chin Huang (cthuang@io.org).  Bug reports should
//  be submitted to rgriech@ibm.net.
//
//  This file is part of soup++ for OS/2.  Soup++ including this file
//  is freeware.  There is no warranty of any kind implied.  The terms
//  of the GNU Gernal Public Licence are valid for this piece of software.
//
//  NNTP client routines
//

/***********************************************************************
  module:       pop3.c
  program:      popclient
  SCCS ID:      @(#)pop3.c      2.4  3/31/94
  programmer:   Carl Harris, ceharris@vt.edu
  date:         29 December 1993
  compiler:     DEC RISC C compiler (Ultrix 4.1)
  environment:  DEC Ultrix 4.3 
  description:  POP2 client code.
 ***********************************************************************/

 
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <ctype.h>
#include <string.h>

#include "areas.hh"
#include "global.hh"
#include "mts.hh"
#include "pop3.hh"
#include "socket.hh"


/* exit code values */

enum PopRetCode {ps_success,      // successful receipt of messages
                 ps_socket,       // socket I/O woes
                 ps_protocol,     // protocol violation
                 ps_error         // some kind of POP3 error condition
};


/*********************************************************************
  function:      POP3_ok
  description:   get the server's response to a command, and return
                 the extra arguments sent with the response.
  arguments:     
    argbuf       buffer to receive the argument string (==NULL -> no return)
    socket       socket to which the server is connected.

  return value:  zero if okay, else return code.
  calls:         SockGets
 *********************************************************************/

static PopRetCode POP3_ok(char *argbuf, TSocket &socket)
{
    PopRetCode ok;
    char buf[BUFSIZ];
    char *bufp;

    if (socket.gets(buf, sizeof(buf))) {
	bufp = buf;
	if (*bufp == '+' || *bufp == '-')
	    bufp++;
	else
	    return(ps_protocol);

	while (isalpha(*bufp))
	    bufp++;
	*(bufp++) = '\0';
	
	if (strcmp(buf,"+OK") == 0)
	    ok = ps_success;
	else if (strcmp(buf,"-ERR") == 0)
	    ok = ps_error;
	else
	    ok = ps_protocol;
	
	if (argbuf != NULL)
	    strcpy(argbuf,bufp);
    }
    else 
	ok = ps_socket;
    
    return(ok);
}   // POP3_ok



/*********************************************************************
  function:      POP3_Auth
  description:   send the USER and PASS commands to the server, and
                 get the server's response.
  arguments:     
    userid       user's mailserver id.
    password     user's mailserver password.
    socket       socket to which the server is connected.

  return value:  non-zero if success, else zero.
  calls:         SockPrintf, POP3_ok.
 *********************************************************************/

static PopRetCode POP3_Auth(const char *userid, const char *password, TSocket &socket) 
{
    PopRetCode ok;
    char buf[BUFSIZ];

    socket.printf("USER %s\r\n",userid);
    if ((ok = POP3_ok(buf,socket)) == ps_success) {
	socket.printf("PASS %s\r\n",password);
	if ((ok = POP3_ok(buf,socket)) == ps_success) 
	    ;  //  okay, we're approved.. 
	else
	    areas.mailPrintf1( 1,"%s\n",buf);
    }
    else
	areas.mailPrintf1(1,"%s\n",buf);
    
    return(ok);
}   // POP3_Auth




/*********************************************************************
  function:      POP3_sendQuit
  description:   send the QUIT command to the server and close 
                 the socket.

  arguments:     
    socket       socket to which the server is connected.

  return value:  none.
  calls:         SockPuts, POP3_ok.
 *********************************************************************/

static PopRetCode POP3_sendQuit(TSocket &socket)
{
    char buf[BUFSIZ];
    PopRetCode ok;

#ifdef DEBUG
    printfT( "POP3_sendQuit(): QUIT\n" );
#endif
    socket.puts("QUIT");
    ok = POP3_ok(buf,socket);
    if (ok != ps_success)
	areas.mailPrintf1( 1,"%s\n",buf);

    return(ok);
}   // POP3_sendQuit



/*********************************************************************
  function:      POP3_sendStat
  description:   send the STAT command to the POP3 server to find
                 out how many messages are waiting.
  arguments:     
    count        pointer to an integer to receive the message count.
    socket       socket to which the POP3 server is connected.

  return value:  return code from POP3_ok.
  calls:         POP3_ok, SockPrintf
 *********************************************************************/

static PopRetCode POP3_sendStat(int *msgcount, TSocket &socket)
{
    PopRetCode ok;
    char buf[BUFSIZ];
    int totalsize;
    
    socket.puts("STAT");
    ok = POP3_ok(buf,socket);
    if (ok == ps_success)
	sscanfT(buf,"%d %d",msgcount,&totalsize);
    else
	areas.mailPrintf1( 1,"%s\n",buf);

    return(ok);
}   // POP3_sendStat




/*********************************************************************
  function:      POP3_sendRetr
  description:   send the RETR command to the POP3 server.
  arguments:     
    msgnum       message ID number
    socket       socket to which the POP3 server is connected.

  return value:  return code from POP3_ok.
  calls:         POP3_ok, SockPrintf
 *********************************************************************/

static PopRetCode POP3_sendRetr(int msgnum, TSocket &socket)
{
    PopRetCode ok;
    char buf[BUFSIZ];

    socket.printf("RETR %d\n",msgnum);
    ok = POP3_ok(buf,socket);
    if (ok != ps_success)
	areas.mailPrintf1( 1,"%s\n",buf);

    return(ok);
}   // POP3_sendRetr



/*********************************************************************
  function:      POP3_sendDele
  description:   send the DELE command to the POP3 server.
  arguments:     
    msgnum       message ID number
    socket       socket to which the POP3 server is connected.

  return value:  return code from POP3_ok.
  calls:         POP3_ok, SockPrintF.
 *********************************************************************/

static PopRetCode POP3_sendDele(int msgnum, TSocket &socket)
{
    PopRetCode ok;
    char buf[BUFSIZ];

    socket.printf("DELE %d\n",msgnum);
    ok = POP3_ok(buf,socket);
    if (ok != ps_success)
	areas.mailPrintf1(1,"%s\n",buf);

    return(ok);
}   // POP3_sendDele



/*********************************************************************
  function:      POP3_readmsg
  description:   Read the message content as described in RFC 1225.
                 RETR with reply evaluation has been done before
  arguments:     
    socket       ... to which the server is connected.
    mboxfd       open file descriptor to which the retrieved message will
                 be written.  
    topipe       true if we're writing to the system mailbox pipe.

  return value:  zero if success else PS_* return code.
  calls:         SockGets.
 *********************************************************************/

static PopRetCode POP3_readmsg(TSocket &socket, TAreasMail &outf)
{
    char buf[BUFSIZ];
    char *bufp;
    time_t now;
    static int  BytesRead = 0;
    int OldKbRead;

    outf.msgStart();
    //
    //  Unix mail folder format requires the Unix-syntax 'From' header.
    //  POP3 doesn't send it, so we fake it here.
    //    
    now = time(NULL);
    strftime(buf, sizeof(buf), "From POPmail %a %b %d %H:%M %Z %Y\n",
	     localtime(&now));
    outf.msgPrintf( "%s",buf );

    //
    //  read the message content from the server
    //
    OldKbRead = -1;
    for (;;) {
	if (socket.gets(buf,sizeof(buf)) == NULL) {
	    outf.msgStop();
	    return ps_socket;
	}
	bufp = buf;
	if (*bufp == '.') {
	    bufp++;
	    if (*bufp == 0)
		break;     // end of message
	}

	if (strncmp(bufp, "From ", 5) == 0)
	    outf.msgPrintf( ">%s\n",bufp );
	else
	    outf.msgPrintf( "%s\n",bufp );

	BytesRead += strlen(bufp);
	if (BytesRead / 1000 != OldKbRead) {
	    printfT( "(%05dk)\b\b\b\b\b\b\b\b", BytesRead / 1000 );
	    OldKbRead = BytesRead / 1000;
	}
    }

    outf.msgStop();
    outf.msgFlush();
    return ps_success;
}   // POP3_readmsg



/*********************************************************************
  function:      getMail
  description:   retrieve messages from the specified mail server
                 using Post Office Protocol 3.

  arguments:     
    options      fully-specified options (i.e. parsed, defaults invoked,
                 etc).

  return value:  exit code from the set of PS_.* constants defined in 
                 popclient.h
  calls:
 *********************************************************************/

int getMail (const char *host, const char *userid, const char *password)
{
    PopRetCode ok;
    TSocket socket;
    int number,count;
    int percent, lastPercent;

#ifdef TRACE
    printfT( "getMail(%s,%s,.)\n", host,userid );
#endif

    if (host == NULL) {
	areas.mailPrintf1( 1,"%s: no pop3 server defined\n", progname );
	return 0;
    }
    if (socket.open( host,"pop3","tcp" ) < 0) {
	areas.mailPrintf1( 1,"%s: cannot connect to pop3 server %s\n", progname,host);
	return 0;
    }

    areas.mailPrintf1( 1,"%s: connected to pop3 server %s\n",progname,host );
    
    ok = POP3_ok(NULL, socket);
    if (ok != ps_success) {
	if (ok != ps_socket)
	    POP3_sendQuit(socket);
	socket.close();
	return 0;
    }

    /* try to get authorized */
    ok = POP3_Auth(userid, password, socket);
    if (ok != ps_success)
	goto cleanUp;

    /* find out how many messages are waiting */
    ok = POP3_sendStat(&count, socket);
    if (ok != ps_success) {
	goto cleanUp;
    }

    /* show them how many messages we'll be downloading */
    areas.mailPrintf1( 1,"%s: you have %d mail message%s\n", progname, count,
	    (count == 1) ? "" : "s");

    if (count > 0) { 
	/* Open the mailbox file. */
	areas.msgOpen( "Email","mn" );
	
	lastPercent = 0;
	for (number = 1;  number <= count;  number++) {
	    percent = (number * 100) / count;
	    if (percent != lastPercent) {
		printfT("\r%d%%  ", percent);
		lastPercent = percent;
	    }

	    ok = POP3_sendRetr(number,socket);
	    if (ok != ps_success)
		goto cleanUp;
	    
	    ok = POP3_readmsg(socket, areas);
	    if (ok != ps_success)
		goto cleanUp;
	    
	    if ( !readOnly) {
		ok = POP3_sendDele(number,socket);
		if (ok != ps_success)
		    goto cleanUp;
	    }
	}
	areas.msgClose();
    }
    ok = POP3_sendQuit(socket);
    socket.close();
    return 1;
    
cleanUp:
    //
    //  Fehlerfall
    //
    if (ok != ps_success  &&  ok != ps_socket)
	POP3_sendQuit(socket);
    
    if (ok == ps_socket) 
	perror("doPOP3: cleanUp");
    
    return 0;
}   // getMail
