//  $Id: nntpcl.cc 1.14 1996/09/02 13:28:02 hardy Exp $
//
//  This progam/module was written by Hardy Griech based on ideas and
//  pieces of code from Chin Huang (cthuang@io.org).  Bug reports should
//  be submitted to rgriech@ibm.net.
//
//  This file is part of soup++ for OS/2.  Soup++ including this file
//  is freeware.  There is no warranty of any kind implied.  The terms
//  of the GNU Gernal Public Licence are valid for this piece of software.
//
//  NNTP client routines
//


#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "mts.hh"
#include "nntp.hh"
#include "nntpcl.hh"
#include "socket.hh"



//
//  is this a misfeature of GCC, is there a better way to do it??
//
#define STR2(x) #x
#define STR(x)  STR2(x)



static TSemaphor cntSema;     // static in class tut nicht (gcc2.7.0)
static TProtCounter bytesRcvd;



//--------------------------------------------------------------------------------



TNntp::TNntp( void )
{
#ifdef TRACE_ALL
    printfT( "TNntp::TNntp()\n" );
#endif
    tmpF = NULL;
    tmpN = NULL;
    xrefHook  = NULL;
    killQHook = NULL;
    actGroup = xstrdup("");
    selGroup = xstrdup("");
    user     = xstrdup("");
    passwd   = xstrdup("");
    strcpy( lastErrMsg, "unknown error condition" );
}   // TNntp::TNntp



TNntp::~TNntp()
{
#ifdef TRACE_ALL
    printfT( "TNntp::~TNntp()\n" );
#endif
    close( 0 );
////    delete actGroup;
////    delete selGroup;
    //// delete user;
    //// delete passwd;
}   // TNntp::~TNntp



void TNntp::setHelper( void (*xref)(const char *xrefLine),
		       int (*killQ)(const char *groupName, const char *headerLine ) )
{
    xrefHook  = xref;
    killQHook = killQ;
}   // TNntp::setHelper



TNntp::Res TNntp::request( const char *cmd, char *reply, size_t replySize,
			   int expReply )
//
//  Send a request to NNTP server and check the result (cmd must not end with \n)
//  If the server request authentication, the AUTHINFO procedure according to
//  RFC977-extension will be executed.
//
{
    int retcode;
    int loopCnt;

    *reply = '\0';
    loopCnt = 0;
    for (;;) {
	//
	//  three retries for the command
	//
	if (loopCnt++ >= 3) {
	    strcpy( lastErrMsg,"nntp server is in a loop requesting AUTHINFO..." );
	    return nok;
	}

	//
	//  transmit the command & fetch the result
	//
	puts( cmd );
	if (gets(reply,replySize) == NULL) {
	    sprintfT( lastErrMsg,"%s:  no reply", cmd );
	    return nok;
	}
	bytesRcvd += strlen(reply);
	if (reply[0] == CHAR_FATAL) {
	    sprintfT( lastErrMsg,"%s:  fatal(%s)", cmd,reply );
	    return nok;
	}

	//
	//  if return code != ERR_NOAUTH, we are done (-> check the result)
	//
	retcode = atoi(reply);
	if (retcode != ERR_NOAUTH)
	    break;

#ifndef NDEBUG
	fprintfT( stderr,"authentication requested\n" );
#endif
	//
	//  otherwise do the authentication
	//
	printf( "AUTHINFO USER %s\n",user );
	if (gets(reply,replySize) == NULL) {
	    strcpy( lastErrMsg,"AUTHINFO USER:  no reply" );
	    return nok;
	}
	retcode = atoi(reply);
	if (retcode == OK_AUTH)
	    continue;
	if (retcode != NEED_AUTHDATA) {
	    sprintfT( lastErrMsg,"AUTHINFO USER:  %s",reply );
	    return nok;
	}
	
	printf( "AUTHINFO PASS %s\n",passwd );
	if (gets(reply,replySize) == NULL) {
	    strcpy( lastErrMsg,"AUTHINFO PASS:  no reply" );
	    return nok;
	}
	retcode = atoi(reply);
	if (retcode != OK_AUTH) {
	    sprintfT( lastErrMsg,"AUTHINFO PASS:  %s",reply );
	    return nok;
	}
#ifndef NDEBUG
	fprintfT( stderr,"authentication ok\n" );
#endif
    }

    if (retcode != expReply) {
	sprintfT( lastErrMsg,"%s:  %s", cmd,reply );
	return nok;
    }
    return ok;
}   // TNntp::request



TNntp::Res TNntp::open( const char *nntpServer, const char *nntpUser,
			const char *nntpPasswd )
{
    char buf[NNTP_STRLEN];
    int  response;

#ifdef TRACE_ALL
    printfT( "TNntp::open(%s)\n",nntpServer );
#endif
    readOnly = 0;
    xstrdup(   &user,nntpUser );
    xstrdup( &passwd,nntpPasswd );

    if (nntpServer == NULL  ||  *nntpServer == '\0') {
	strcpy( lastErrMsg,"no news server defined" );
	return nok;
    }

    if (TSocket::open( nntpServer,"nntp","tcp" ) < 0) {
	strcpy( lastErrMsg,"cannot open socket" );
	return nok;
    }

    if (gets(buf, sizeof(buf)) == NULL) {
	strcpy( lastErrMsg,"connect:  no reply" );
#ifdef DEBUG
	printfT( "TNntp::open():  socket: %s\n",buf );
#endif
	return nok;
    }
    else {
	response = atoi(buf);
	switch (response) {

	case OK_NOPOST:
	    readOnly = 1;
	    break;
	    
	case OK_CANPOST:
	    break;
	    
	case ERR_ACCESS:
	    sprintfT( lastErrMsg,"connect:  no permission, %s",buf );
	    return nok;
	    
	default:
	    sprintfT( lastErrMsg,"connect:  ill response, %s",buf );
#ifdef DEBUG
	    printfT( "TNntp::open():  illresp: %s\n",buf );
#endif
	    return nok;
	}
    }

    //
    // This is for INN (result is ignored)
    //
    request( "MODE READER",buf,sizeof(buf),OK_CANPOST );
#ifdef DEBUG_ALL
    printfT( "TNntp::open():  antwort auf mode reader: %s\n",buf );
#endif

    //
    //  create temporary file
    //
    if (tmpF == NULL) {
	sysSema.Request();
	tmpN = tempnam(NULL,"soup");
	tmpF = fopenT(tmpN,"w+b");
	sysSema.Release();
	if (tmpF == NULL) {
	    sprintfT( lastErrMsg,"create of '%s' failed",tmpN );
	    return nok;
	}
    }
    return ok;
}   // TNntnp::open



void TNntp::close( int sendQuit )
{
#ifdef TRACE_ALL
    printfT( "TNntp::close(%d)\n",sendQuit );
#endif
    if (sendQuit) {
	char buf[100];

#ifdef TRACE_ALL
	printfT( "TNntp::close(): QUIT\n" );
#endif
	request( "QUIT",buf,sizeof(buf), OK_GOODBYE );
	TSocket::close();
    }

    //
    //  remove temporary file
    //
    if (tmpF != NULL) {
	fcloseT( tmpF );
	tmpF = NULL;
	removeT( tmpN );
	//// delete tmpN;
	tmpN = NULL;
    }
}   // TNntp::close



const char *TNntp::getLastErrMsg( void )
{
    return lastErrMsg;
}   // TNntp::getLastErrMsg



TNntp::Res TNntp::getNewGroups( const char *nntpTimeFile, int changeFile )
//
//  - send listing of new newsgroups as an email
//  - add new newsgroups to newsrc
//
{
    char oldTime[80], nntpTime[80], buf[NNTP_STRLEN];
    FILE *dateF;
    int getall;
    char *p;

#ifdef TRACE
    printfT( "getNewGroups()\n" );
#endif

    //
    //  get current date/time from NNTP server
    //
    if (request("DATE",buf,sizeof(buf),INF_DATE) == ok)
	sscanfT( buf+4, "%s", nntpTime );
    else {
	time_t now = time(NULL);
	strftime( nntpTime, sizeof(nntpTime), "%Y%m%d%H%M%S", gmtime(&now) );
    }
	
    //
    //  Get last date/time we checked for new newsgroups.
    //
    getall = 0;
    if ((dateF = fopenT(nntpTimeFile, "r")) != NULL) {
	fgetsT( oldTime, sizeof(oldTime), dateF);
	fclose( dateF );
    }
    else {
	//
	//  This is probably the first we checked for new newsgroups.
	//  yes -> get all available newsgroups!
	//
	memset( oldTime,0,sizeof(oldTime) );
	getall = 1;
    }

    //
    //  Request new newsgroups.
    //
    {
	char cmd[100];

	if (getall)
	    strcpy( cmd,"LIST" );
	else
	    sprintfT( cmd,"NEWGROUPS %-6.6s %-6.6s GMT", oldTime+2, oldTime+8);
	if (request(cmd,buf,sizeof(buf),getall ? OK_GROUPS : OK_NEWGROUPS) != ok)
	    return nok;
    }

    ftruncateT( tmpF,0L );
    while (gets(buf, sizeof(buf)) != NULL) {
#ifdef DEBUG_ALL
	printfT( "rcv: %s\n",buf );
#endif
	bytesRcvd += strlen(buf)+1;

	if (buf[0] == '.')
	    break;
	if ((p = strchr(buf, ' ')) != NULL)
	    *p = '\0';

	fputsT( buf,tmpF );
	fputcT( '\n',tmpF );
    }

    //
    //  Save current date/time.
    //
    if (changeFile) {
	if ((dateF = fopenT(nntpTimeFile,"w")) != NULL) {
	    fprintfT( dateF,"%s\n",nntpTime );
	    fcloseT(dateF);
	}
    }
    return ok;
}   // TNntp::getNewGroups



TNntp::Res TNntp::getOverview( long first, long last )
//
//  Attention:  those overview lines are sometimes VERY long (references...)
//
{
    char buf[NNTP_STRLEN];
    char cmd[100];

#ifdef TRACE_ALL
    printfT( "TNntp::getOverview(%ld,%ld,%s)\n",first,last,selGroup );
#endif
    if (first < last)
	sprintfT( cmd,"XOVER %ld-%ld", first, last );
    else
	sprintfT( cmd,"XOVER %ld-", first );
    if (request(cmd,buf,sizeof(buf),OK_XOVER) != ok)
	return nok;

    ftruncateT( tmpF,0L );
    while (gets(buf, sizeof(buf)) != NULL) {
	bytesRcvd += strlen(buf)+1;
	if (buf[0] == '.')
	    break;
	fputsT( buf,tmpF );
#ifdef TRACE_ALL
//	printfT( "%s\n",buf );
#endif
	fputcT( '\n',tmpF );
    }
    return ok;
}   // TNntp::getOverview



TNntp::Res TNntp::setActGroup( const char *group, long &cnt, long &lo, long &hi )
//
//  activate nntp group
//  returns:  ok,nok
//
{
    char buf[NNTP_STRLEN];
    char cmd[100];
    long l1,l2,l3;

#ifdef TRACE_ALL
    printfT( "TNntp::setActGroup(%s,..)\n",group );
#endif

    xstrdup( &actGroup,group );
    selNntpArticle = -1;

    sprintfT( cmd,"GROUP %s",group );
    if (request(cmd,buf,sizeof(buf),OK_GROUP) != ok) {
	xstrdup( &actGroup,"" );
	return nok;
    }

    sscanfT(buf+4, "%ld %ld %ld", &l1, &l2, &l3);
    cnt = l1;  lo = l2;  hi = l3;
////	selNntpArticle = l2;  wre korrekt, bringt es aber nicht so fr den NEXT
    nntpGroupHi = l3;
    
#ifdef TRACE_ALL
    printfT( "TNntp::setActGroup(%s,%ld,%ld,%ld)\n",group,cnt,lo,hi );
#endif
    return ok;
}   // TNntp::setActGroup



TNntp::Res TNntp::nextArticle( long *next )
//
//  Get next article in group.
//  Return ok if successful.
//
{
    char buf[NNTP_STRLEN];

    if (request("NEXT",buf,sizeof(buf),OK_NOTEXT) != ok) {
	*next = selNntpArticle = nntpGroupHi;
	return nok;                            // no next article
    }
    *next = selNntpArticle = atol(buf+4);

#ifdef DEBUG_ALL
    printfT( "nntpNext() -> %ld\n",*next );
#endif
    return ok;
}   // TNntp::nextArticle



void TNntp::selectArticle( const char *grpname, long artNum, int doKill )
{
#ifdef TRACE_ALL
    printfT( "selectArticle(%s,%ld,%d)\n",grpname,artNum,doKill );
#endif

    if (strcmp(selGroup,grpname) != 0)
	xstrdup( &selGroup,grpname );

    selArticle  = artNum;
    killEnabled = doKill;
}   // TNntp::selectArticle



TNntp::Res TNntp::_getHead( void )
//
//  Get the articles header and write it to a temporary file (tmpF)
//  killing & cross referencing is handled here
//  return:  ok,nok,killed,notavail
//
{
    char buf[NNTP_STRLEN];
    char cmd[100];
    char gotXref, artKilled;

#ifdef TRACE_ALL
    printfT( "_getHead(): %ld\n",selArticle );
#endif

    //
    //  request article (head)
    //
    sprintfT( cmd,"%s %ld", killEnabled ? "HEAD" : "ARTICLE",selArticle );
    if (request(cmd,buf,sizeof(buf),killEnabled ? OK_HEAD : OK_ARTICLE) != ok)
	return (buf[0] == CHAR_FATAL) ? nok : notavail;
    selNntpArticle = selArticle;
    
    artKilled = 0;
    gotXref = 0;

    //
    //  Get lines of article head.
    //
    while (gets(buf, sizeof(buf)) != NULL) {
	char *bufp = buf;

#ifdef DEBUG_ALL
	printfT( "--1: %ld '%s'\n",selArticle,bufp );
#endif

	bytesRcvd += strlen(buf)+1;
	
	if (killEnabled) {
	    if (buf[0] == '.')
		if (*(++bufp) == '\0')
		    break;
	}
	else if (*bufp == '\0')
	    break;
	
	fputsT(bufp, tmpF);
	fputcT('\n', tmpF);

	if (killEnabled  &&  !artKilled) {
	    if (killQHook != NULL  &&  killQHook(selGroup,bufp)) {
		if (strlen(bufp) > 50)
		    strcpy( bufp+50," [..]" );
		sprintfT( lastErrMsg,"article killed %s", bufp );
		artKilled = 1;
	    }
	}

	if (xrefHook != NULL  &&  !gotXref  &&  strnicmp(bufp, "xref: ", 6) == 0) {
	    xrefHook(bufp+6);
	    gotXref = 1;           // why is only one Xref allowed ?
	}
    }

    //
    //  Don't process anymore if article was killed.
    //
    if (artKilled) {
	assert( killEnabled != 0 );
	return killed;
    }

    //
    //  Put empty line separating head from body.
    //
    fputcT('\n', tmpF);
    return ok;
}   // TNntp::_getHead



TNntp::Res TNntp::_getBody( void )
//
//  Get the articles body and write it to a temporary file (tmpF)
//  should not be called, if article is going to be killed
//  return:  nok, ok
//
{
    char buf[NNTP_STRLEN];

#ifdef TRACE_ALL
    printfT( "_getBody(): %ld\n",selArticle );
#endif
    
    if (killEnabled) {
	char cmd[100];

	sprintfT( cmd,"BODY %ld", selArticle );
	if (request(cmd,buf,sizeof(buf),OK_BODY) != ok)
	    return (buf[0] == CHAR_FATAL) ? nok : notavail;
	selNntpArticle = selArticle;
    }

    //
    //  Retrieve article body.
    //
    while (gets(buf, sizeof(buf)) != NULL) {
	char *bufp = buf;

	bytesRcvd += strlen(buf)+1;

	if (buf[0] == '.') {
	    if (*(++bufp) == '\0')
		break;                  // -> end of article !
	}
	fputsT(bufp, tmpF);
	fputcT('\n', tmpF);
#ifdef DEBUG_ALL
	printfT( "--2: %ld '%s'\n",selArticle,bufp );
#endif
    }
    return ok;
}   // _getBody



TNntp::Res TNntp::getArticle( void )
//
//  Get the article and write it to a temporary file (tmpF)
//  killing & cross referencing is handled here
//  return:  ok,nok,notvail,killed
//  calls: _getHead, _getBody
//
{
    Res res;

#ifdef TRACE_ALL
    printfT( "getArticle(): %ld\n",selArticle );
#endif

#ifdef TRACE_ALL
    printfT( "--0: %ld\n",selArticle );
#endif

    //
    //  select the group, if required
    //
    if (strcmp(actGroup,selGroup) != 0) {
	long d0,d1,d2;
	res = setActGroup( selGroup, d0,d1,d2 );
	if (res != ok)
	    return res;
    }

#ifdef TRACE_ALL
    printfT( "--1: %ld\n",selArticle );
#endif

    //
    //  Get article to temporary file.
    //
    fseekT(tmpF, 0L, SEEK_SET);

    res = _getHead();
    if (res != ok)
	return res;

#ifdef TRACE_ALL
    printfT( "--2: %ld\n",selArticle );
#endif

    res = _getBody();

#ifdef TRACE_ALL
    printfT( "--3: %ld\n",selArticle );
#endif
    return res;
}   // TNntp::getArticle



long TNntp::getBytesRcvd( void )
{
    return bytesRcvd;
}   // TNntp::getBytesRcvd



TNntp::Res TNntp::postArticle( FILE *inf, size_t bytes )
//
//  Post article to NNTP server.
//  on entry:  file pointer to '*inf' points to beginning of message
//             'bytes' contains message size
//  on exit:   file pointer to '*inf' is set to end of message
//  Return ok if successful.
//
{
    char buf[NNTP_STRLEN], *s;
    size_t count;
    long offset;

#ifdef TRACE
    printfT( "TNntp::postArticle(.,%ld)\n",bytes );
#endif
    if (request("POST",buf,sizeof(buf),CONT_POST) != ok)
	return nok;

    offset = ftellT(inf);
    count = bytes;
    while (fgetsT(buf, sizeof(buf), inf)  &&  count > 0) {
	count -= strlen(buf);
	if ((s = strchr(buf, '\n')) != NULL)
	    *s = '\0';
	printf( "%s%s\n", (buf[0] == '.') ? "." : "", buf );
    }
    fseekT(inf, offset+bytes, SEEK_SET);

    if (request(".",buf,sizeof(buf),OK_POSTED) != ok) {
	if (atoi(buf) == ERR_POSTFAIL)
	    sprintfT( lastErrMsg, "POST:  article not accepted by server; not posted\n\t(%s)",buf );
	else
	    sprintfT( lastErrMsg, "POST:  %s",buf );

	//
	//  if the server replied with a 'dont resend' or so,
	//  an ok-condition is faked
	//
	if (strstr(buf,STR(ERR_GOTIT))    != NULL  ||
	    strstr(buf,STR(ERR_XFERRJCT)) != NULL)
	    return ok;
	return nok;
    }
    return ok;
}   // TNntp::postArticle
