//  $Id: news.cc 1.23 1996/09/02 13:25:10 hardy Exp $
//
//  This progam/module was written by Hardy Griech based on ideas and
//  pieces of code from Chin Huang (cthuang@io.org).  Bug reports should
//  be submitted to rgriech@ibm.net.
//
//  This file is part of soup++ for OS/2.  Soup++ including this file
//  is freeware.  There is no warranty of any kind implied.  The terms
//  of the GNU Gernal Public Licence are valid for this piece of software.
//
//  Get news from NNTP server.
//
//  rg270596:
//  - multithreading support for OS/2
//
//  Und nun mal ein paar Erfahrungen mit multithreading (in diesem Zusammenhang):
//  -----------------------------------------------------------------------------
//
//  Probleme mit CHANGI:
//  - die ursprngliche CHANGI Version "spann" bei vielfachem Zugriff
//    Update nach changi09m brachte erhebliche Erleichterung
//  - nntp-NEXT tut nicht, wenn gleichzeitig ein Artikel gelesen wird,
//    oder wenn mehrere NEXTs unterwegs sind (knnte CHANGI-Problem sein)
//    120696:  war eher meine Bldheit und hatte u.U. was mit den Signalen zu tun...
//             Wahrscheinlich braucht der NEXT recht lange und der entsprechende
//             gets() wurde dann mit hoher Wahrscheinlichkeit durch ein Signal
//             unterbrochen...
//
//  Probleme mit EMX-GCC09a:
//  - das signal-handling scheint fragwrdig (signal mu aber abgefangen werden...):
//    - nextchar in socket (wird durch recv/read gemacht) kommt u.U. mit EINTR
//      zurck - doch wie setze ich dann wieder auf???
//    - _beginthread kommt u.U. mit EINVAL zurck, was jedoch ebenso auf einen
//      unterbrochenen Aufruf schlieen lt (d.h. Fehlerauswertung nicht vollstndig)
//    --> Signale nicht dazu verwenden, um die Beendigung eines Threads anzuzeigen,
//        sondern nur in absoluten Notfllen!!!
//  - einmal (?) hatte ich im (durch Semaphor geschtzten) StdOut einen greren
//    Block doppelt.  Das Programm hat den Block unmglich (?) produzieren knnen,
//    also kommt nur EMX-GCC in Frage bzw. das OS
//  - new/delete knnen nicht selbst definiert werden
//  - wie komme ich bitte an _threadid ? (stddef.h war nicht fr C++)
//  - unlink steht nicht in stdio.h, sondern unistd.h
//  - tmpfile() / tempnam() durch Semaphor geschtzt ??
//  --> htte ich Zugriff, wrde ich sofort nach 09b updaten!!!
//
//  Hausgemachte Probleme:
//  - ein mehrfacher Request von einem MutexSemaphor in EINEM Thread hlt diesen
//    NICHT an.  Nur ein anderer Thread kann das Semaphor nicht mehr anfordern...
//  - Zustandsmaschine war durch 'mode reader' nicht mehr korrekt (es wurde schon ein
//    'waiting' angezeigt, obwohl noch 'init' war...)
//    - in nntpMtGetFinished wurde der Zustand zweimal abgefragt und dann noch in der
//      Reihenfolge 'finished'?, 'running'?.  Dieser bergang wird aber in einem Thread
//      gemacht -> Thread war u.U. noch nicht 'finished', aber auch nicht mehr 'running'.
//      Dies ergibt ein leicht inkonsistentes Bild der Zustnde!
//  - wird ein Zhler im Thread hochgezhlt und mu hinterher ausgewertet
//    werden, so empfiehlt sich mindestens ein Semaphor (vielleicht auch noch
//    volatile) (bytesRcvd)
//  - die Threads mssen auch einen Signal-Handler fr z.B. SIGPIPE haben, sonst
//    gibt es bei Abbruch u.U. einen doppelten Fehler! (das kommt daher, wie ein
//    Programm abgebrochen wird)
//  - ein Event-Semaphor will auch zurckgesetzt werden!  Die 'Kinder' laufen sonst
//    echt Gefahr zu verhungern...
//  - stream-I/O mu konsequent durch MuxSema geschtzt werden (ein bichen Disziplin
//    bitte)
//  - regexp hat statische Variablen
//  - um Klassen, die was mit Listen oder so zu tun haben, am besten auch ein
//    individuelles Semaphor legen
//  - stimmt der makefile nicht, und es wird ein Datentyp gendert, so kommt es
//    klarerweise zu seltsamen Effekten (die Objekte werden nicht neu angelegt, etc.)
//  - beachte Zuweisung eines 'char' von 0xff (== -1) an einen int!!!
//


//
//  um das Signal-Problem hinzubekommen einfach definieren!
//  - nextchar() in socket liefert bei read bzw recv EINTR zurck
//  - _beginthread wird u.U. auch unterbrochen (recht unwahrscheinlich, aber wahr)
//
//#define SIGNAL_PROBLEM

//
//  Testroutine, die die Angelegenheit ein bichen behindern soll, einbinden
//
//#define INSERT_MTTEST


#include <assert.h>
#include <process.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "areas.hh"
#include "global.hh"
#include "kill.hh"
#include "mts.hh"
#include "news.hh"
#include "newsrc.hh"
#include "nntp.hh"
#include "nntpcl.hh"
#include "socket.hh"



static TSemaphor   divSema;
static TEvSemaphor threadFinito;
static TEvSemaphor disconnectDone;

static TProtCounter artsRcvd;         // articles received...
static TProtCounter artsKilled;       // articles killed...

static TKillFile killF;               // kill file handling

static TProtCounter getArt_artRcvd;   // globals for mtGetArticle()
static long getArt_grpHi;             //    (I hate those f* globals -> restructure)
static int doingProcessSendme;        //    currently doing processSendme

static long catchupNumKeep;

//
//  thread states (init must be 0)
//  starting is for debugging and not absolutely required...
////  (are static vars initialized to zero ?)
//
enum NntpStates { init,connecting,failed,waiting,starting,running,runningspecial };

//
//  these are the thread connections to the news server
//
#ifndef INSERT_MTTEST
static TNntp nntp[ MAXNNTPTHREADS ];             // no pointer (because of imlicit destructor)
static NntpStates nntpS[MAXNNTPTHREADS ] = {init};
#else
static TNntp nntp[ MAXNNTPTHREADS+1 ];
static NntpStates nntpS[MAXNNTPTHREADS+1 ] = {init };
#define TESTTHREAD (MAXNNTPTHREADS)
#endif



//--------------------------------------------------------------------------------
//
//  utility functions
//



#ifdef SIGNAL_PROBLEM
static void signalHandler( int sig )
//
//  nur dazu da, um sleep abzubrechen
//
{
    signal( sig, SIG_ACK );
}   // signalHandler
#endif



#if defined(DEBUG)  ||  defined(TRACE_ALL)  ||  defined(TRACE)  ||  defined(TRACE_ALL)
static void printThreadState( const char *pre, int maxNo=maxNntpThreads )
{
    int i;
    char b[100];

    assert( maxNo <= maxNntpThreads );

    sprintfT( b,"%s: ",pre );
    for (i = 0;  i < maxNo;  i++) {
	switch (nntpS[i]) {
	case init:
	    strcat(b,"[i]");
	    break;
	case connecting:
	    strcat(b,"[c]");
	    break;
	case starting:
	    strcat(b,"[s]");
	    break;
	case waiting:
	    strcat(b,"[w]");
	    break;
	case running:
	    strcat(b,"[r]");
	    break;
	case runningspecial:
	    strcat(b,"[R]");
	    if (maxNo < maxNntpThreads)
		++maxNo;
	    break;
	case failed:
	    strcat(b,"[E]");
	    if (maxNo < maxNntpThreads)
		++maxNo;
	    break;
	}
    }
    printfT( "%s\n",b );
}
#endif



static int killArticleQ( const char *groupName, const char *headerLine )
//
//  this is a hook function for TNntp...
//
{
    return killF.matchLine( groupName,headerLine );
}   // killArticleQ



static void processXref( const char *s )
//
//  Process an Xref line.
//  format: 'Xref: '<host-name> <grp-name[: ]grp-num>(\b<grp-name[: ]grp-num>)*
//  - s points behind 'Xref: '
//  - \b may be blank or \t
//  
//  rg260596:  the new version works with sscanf (before strtok).  Hopefully this version
//             is ok for multithreading
//
//  hook function for TNntp...
//
//
{
    const char *p, name[FILENAME_MAX];
    int num, cnt;

#ifdef DEBUG_ALL
    printfT( "XREF: '%s'\n",s );
#endif

    //
    //  Skip the host field
    //
    p = strpbrk( s," \t" );
    if (p == NULL)
	return;

    //
    //  Look through the rest of the fields
    //  (note:  the %n does not count in the sscanf-result)
    //
    while (sscanfT(p,"%*[ \t]%[^ \t:]%*[ \t:]%d%n",name,&num,&cnt) == 2) {
#ifdef DEBUG_ALL
	printfT( "xref: '%s' %d\n",name,num );
#endif
	newsrc.artMarkRead( name,num );
	p += cnt;
    }
}   // processXref



static int writeArticle( TAreasMail &msgF, FILE *inF )
//
//  Copy article from temporary file to TAreas-msgfile.
//  Return TRUE if article was copied (successfully).
//
{
    long artSize;
    long toRead, wasRead;
    char buf[4096];   // 4096 = good size for file i/o
    int  res;

    //
    //  Get article size.
    //
    artSize = ftellT(inF);
#ifdef DEBUG
    if (artSize <= 0)
	printfT( "writeArticle(): ftellT: %p,%ld,%ld\n",inF,artSize,ftellT(inF) );
#endif
    if (artSize <= 0)
	return 0;	// Skip empty articles

    msgF.msgStart();
    
    //
    //  Write "rnews" line
    //
    msgF.msgPrintf("#! rnews %ld\n", artSize);
#ifdef DEBUG_ALL
    printfT( "writeArticle(): %ld\n", artSize );
#endif

    //
    //  Copy article body.
    //
    fseekT(inF, 0L, SEEK_SET);
    res = 1;
    while (artSize > 0) {
	toRead = ((size_t)artSize < sizeof(buf)) ? artSize : sizeof(buf);
	wasRead = freadT(buf, sizeof(char), toRead, inF);
	if (wasRead != toRead) {
	    perror("read article");
#ifdef DEBUG
	    printfT( "writeArticle: read article: %p,%lu,%lu,%p,%lu,%lu\n",buf,toRead,wasRead,inF,ftellT(inF),artSize );
#endif
	    res = 0;
	    break;
	}
	assert( wasRead > 0 );
	if (msgF.msgWrite(buf, sizeof(char), wasRead) != (size_t)wasRead) {
	    perror("write article");
	    res = 0;
	    break;
	}
	artSize -= wasRead;
    }

    msgF.msgStop();
    return res;
}   // writeArticle



//--------------------------------------------------------------------------------


static void signalHandlerThread( int sig )
{
#ifndef NDEBUG
    fprintfT( stderr,"\nthread received signal %d\n",sig );
#endif
    _endthread();
}   // signalHandlerThread



static void mtInitSignals( void )
//
//  very important to call this function during init of a THREAD ////???
//
{
#ifdef TRACE_ALL
    printfT( "mtInitSignals()\n" );
#endif
////    signal(SIGINT,   signalHandlerThread );     // ^C
////    signal(SIGBREAK, signalHandlerThread );     // ^Break
    signal(SIGPIPE,  signalHandlerThread );     // broken pipe
    signal(SIGSEGV,  signalHandlerThread );
    signal(SIGTERM,  signalHandlerThread );     // kill (lt der sich doch catchen?)
}   // mtInitSignals



#ifdef INSERT_MTTEST
static void mtTest( void *threadNo )
// TestoMat als Thread
{
    int no = (int)threadNo;
    int i;
    int n = 0;
    long cnt,lo,hi;

    mtInitSignals();

    for (;;) {
	if (nntpS[no] == waiting) {
	    nntpS[no] = running;
	}
	else if (nntpS[no] == running) {
	    if (nntp[no].setActGroup( (n & 1) ? "comp.os.os2.announce" : "fido.ger.os2",
					cnt, lo, hi ) != TNntp::ok) {
		printfT( "mtTest(): setActGroup() failed\n" );
	    }
	    ++n;
	    for (i = 0;  i < 9;  i++) {
		long l;
		nntp[no].nextArticle( &l );
#ifdef SIGNAL_PROBLEM	       
		kill( getpid(), SIGUSR1 );
#endif
	    }
#ifdef SIGNAL_PROBLEM	       
	    for (i = 0;  i < 100;  i++)
		kill( getpid(), SIGUSR1 );
#endif
	}
	else
	    sleep( 1 );
    }
}   // mtTest
#endif



static void mtGetArticle( void *threadNo )
//
//  states:  starting -> running -> waiting
//
{
    static int artNotAvail = 0;
    int no = (int)threadNo;
    TNntp::Res res;

#ifdef TRACE_ALL
    printfT( "mtGetArticle(%d): running\n",no );
#endif
    assert( nntpS[no] == starting );
    mtInitSignals();

    //
    //  get article
    //
    nntpS[no] = running;
    res = nntp[no].getArticle();

    //
    //  if successfully retrieved, write article
    //
    switch (res) {
    case TNntp::ok:
	//
	//  article successfully received
	//
	writeArticle( areas,nntp[no].getTmpF() );
	newsrc.artMarkRead( nntp[no].groupName(), nntp[no].article() );
	++getArt_artRcvd;
	++artsRcvd;
	artNotAvail = 0;
	break;
    case TNntp::killed:
	newsrc.artMarkRead( nntp[no].groupName(), nntp[no].article() );
	areas.mailPrintf1( 1,"%s: %s\n",progname,nntp[no].getLastErrMsg() );
	++getArt_artRcvd;
	++artsKilled;
	artNotAvail = 0;
	break;
    case TNntp::notavail:
	//
	//  Article not available.  Look for next available article.
	//
	if (doingProcessSendme) {
	    newsrc.artMarkRead( nntp[no].groupName(), nntp[no].article() );
	    areas.mailPrintf1( 1,"%s: %s: %s\n", progname,
			       nntp[no].groupName(),nntp[no].getLastErrMsg() );
	}
	else {
	    //
	    //  hier kann was sehr hliches passieren:
	    //  es wurde noch kein Article gelesen und es wird notavail
	    //  zurckgegeben.  Dann gibt NEXT nmlich grpLo+irgendwas
	    //  (den zweiten verfgbaren Artikel zurck).  Was bedeutet,
	    //  da man sich mit NEXT langsam durch alle gelesenen
	    //  Artikel durchqult, bis man den jetzigen erreicht hat...
	    //  Abhilfe:  nntpArticle<0  ->  noch keinen gelesen...
	    //  Damit die Sache nicht bei 'kleinen' Lchern in der Newsgroup die
	    //  ganze Zeit in NEXTs verfllt, wird mitgezhlt, wieviele Artikel
	    //  hintereinander nicht da waren.  Wird ein bestimmter Wert
	    //  berschritten, wird mit NEXT weitergearbeitet (vorher nicht!)
	    //
#ifndef NDEBUG
	    printfT( "not avail: %ld\n", nntp[no].article() );
#endif
	    if (++artNotAvail < 10)
		newsrc.artMarkRead( nntp[no].groupName(), nntp[no].article() );
	    else if ( !newsrc.artIsRead(nntp[no].groupName(),nntp[no].article())) {
		artNotAvail = 0;
		newsrc.artMarkRead( nntp[no].groupName(), nntp[no].article() );
		if (nntp[no].nntpArticle() >= 0) {
		    long artLo, artHi;
		    long nextArt;

		    artLo = nntp[no].article()+1;
		    artHi = getArt_grpHi;
		    while (nntp[no].nextArticle(&nextArt) == TNntp::ok) {
			artHi = nextArt-1;
#ifndef NDEBUG
			printfT("next! %ld-%ld  \n",artLo,artHi);
#endif
			if (nextArt > nntp[no].article())
			    break;
			while (artLo <= artHi)
			    newsrc.artMarkRead( nntp[no].groupName(), artLo++ );
			artLo = artHi+2;
		    }

		    //
		    //  mark articles from artLo..artHi as read
		    //  at least required, if there is no next article
		    //
#ifndef NDEBUG
		    printfT("NEXT! %ld-%ld  \n",artLo,artHi);
#endif
		    while (artLo <= artHi)
			newsrc.artMarkRead( nntp[no].groupName(), artLo++ );
		}
	    }
	}
	break;
    default:
	areas.mailPrintf1( 1,"%s: %s: %s\n", progname,
			   nntp[no].groupName(),nntp[no].getLastErrMsg() );
	break;
    }

//    printfT("."); ////
#ifdef TRACE_ALL
    printfT( "mtGetArticle(%d): finished, %ld\n",no,nntp[no].article() );
#endif
    nntpS[no] = waiting;    // article handling finished
    threadFinito.Post();
}   // mtGetArticle



static void mtGetNewGroups( void *threadNo )
//
//  states:
//
{
    int no = (int)threadNo;
    char nntpTimePath[FILENAME_MAX];

#ifdef TRACE_ALL
    printfT( "mtGetNewGroups(%d)\n",no );
#endif
    assert( nntpS[no] == starting );
    nntpS[no] = runningspecial;

    mtInitSignals();

    sprintfT( nntpTimePath, "%s/%s", homeDir,FN_NEWSTIME );
    if (nntp[no].getNewGroups(nntpTimePath,!readOnly) != TNntp::ok) {
	areas.mailPrintf1( 0,"can not get new groups:\n\t%s\n",
			   nntp[no].getLastErrMsg() );
	nntpS[no] = waiting;
	return;
    }

    {
	FILE *in = nntp[no].getTmpF();
	int mailOpened = 0;
	char buf[BUFSIZ];
    
	fseekT(in, 0L, SEEK_SET);
    
	mailOpened = 0;
	while (fgets(buf,sizeof(buf),in) != NULL) {
	    buf[strlen(buf)-1] = '\0';             // remove '\n'
#ifdef DEBUG_ALL
	    printfT( "mtGetNewGroups: %s\n",buf );
#endif

	    //
	    //  scan to see if we know about this one
	    //
	    if (newsrc.grpExists(buf))
		continue;
	
	    newsrc.grpAdd( buf );

	    //
	    //  beim ersten neuen Namen eine Mail ffnen
	    //
	    if ( !mailOpened) {
		//
		//  Open message file.
		//
		mailOpened = 1;
		areas.mailStart();
		areas.mailPrintf( "new newsgroups:\n\n", buf );
		genStsMail = 1;                                   // force generation of status mail
	    }

	    //
	    //  neuen Namen in die Mail schreiben
	    //
	    areas.mailPrintf( "%s\n",buf );
	}
	if (mailOpened)
	    areas.mailStop();
    }

    nntpS[no] = waiting;
}   // mtGetNewGroups



static void _nntpMtConnect( void *threadNo )
//
//  set up single connection to news server (could be started as a thread)
//  states:  init -> connecting -> waiting  ||
//           init -> connecting -> failed
//  give it three tries on problem...
//
{
    int i;
    int no = (int)threadNo;
    static int readOnlyDisplayed = 0;

    assert( nntpS[no] == init );

    mtInitSignals();
    nntp[no].setHelper( doXref ? processXref : NULL, killArticleQ );

    for (i = 0;  i < 3;  i++) {
	nntpS[no] = connecting;
	if (nntp[no].open(nntpInfo.host,nntpInfo.user,nntpInfo.passwd) == TNntp::ok) {
	    if (!readOnlyDisplayed  &&  nntp[no].isReadOnly()) {
		readOnlyDisplayed = 1;
		areas.mailPrintf1( 1,"%s: you cannot post to news server %s\n",progname,nntpInfo.host );
	    }
	    nntpS[no] = waiting;
	    break;
	}
	nntpS[no] = failed;
    }
}   // _nntpMtConnect



static void nntpConnect( int maxThreads )
//
//  set up connection to news server
//
{
    int i;

#ifdef TRACE_ALL
    printfT( "nntpConnect(%d)\n",maxThreads );
#endif

    assert( maxThreads <= maxNntpThreads );

    for (i = 0;  i < maxThreads;  i++)
	BEGINTHREAD( _nntpMtConnect, (void *)i );
}   // nntpConnect



static void nntpMtDisconnect( void *maxThreads )
//
//  (explicit) disconnect from news server
//
{
    int maxNo = (int)maxThreads;
    int i;
    
#ifdef TRACE_ALL
    printfT( "nntpDisconnect(%d)\n",maxNo );
#endif

    assert( maxNo <= maxNntpThreads );

    mtInitSignals();
    for (i = 0;  i < maxNo;  i++) {
	if (nntpS[i] != failed)
	    nntp[i].close();
	nntpS[i] = init;
    }
    disconnectDone.Post();
}   // nntpDisconnect



static void nntpMtCheckTimeout( void *dummy )
//
//  check transfer...
//
{
    int timeoutCnt;
    long bytesRcvd;

    mtInitSignals();

    timeoutCnt = -TIMEOUT;
    bytesRcvd  = TNntp::getBytesRcvd();
    for (;;) {
	sleep( 1 );
	if (bytesRcvd != TNntp::getBytesRcvd()) {
	    bytesRcvd  = TNntp::getBytesRcvd();
	    timeoutCnt = 0;
	}
	else {
	    if (++timeoutCnt > TIMEOUT) {
		areas.mailStart(1);
		areas.mailPrintf( "no data received for %ds -> timeout\nsignal %d generated\n",
				  TIMEOUT,SIGINT );
		areas.mailStop();
#ifdef TRACE_ALL
		printfT( "nntpMtCheckTimeout():  TIMEOUT\n" );
#endif
		kill( getpid(),SIGINT );
		break;
	    }
	}
    }
}   // nntpMtCheckTimeout



static int nntpMtWaitConnect( int maxThreads )
//
//  wait until one of the threads has successfully connected, or all of them
//  have failed.  On failure return 0 (timeout after 60s)
//  Running threads are not taken into consideration...
//
{
    int i;
    int conFailed = 0;
    long time = 0;

    assert( maxThreads <= maxNntpThreads );

    while ( !conFailed) {
#ifdef TRACE_ALL
	printThreadState( "nntpMtWaitConnect()",maxThreads );
#endif
	conFailed = 1;
	for (i = 0;  i < maxThreads;  i++) {
	    switch (nntpS[i]) {
	    case waiting:
		return 1;                   // -> connected !
		break;
	    case failed:
		break;                      // -> do nothing
	    case init:
	    case connecting:
		conFailed = 0;              // -> not failed
		break;
	    case running:
	    case runningspecial:
	    case starting:
		conFailed = 0;              // -> not failed
		time = 0;                   //    + reset timeout
		break;
	    }
	}
	//
	//  wait ~100ms
	//
	if ( !conFailed) {
	    _sleep2( 100 );
	    time += 100;
	    conFailed = (time > TIMEOUT*1000);     // timeout after TIMEOUT s
	}
    }
#ifdef TRACE_ALL
    printfT( "nntpMtWaitConnect():  TIMEOUT!\n" );
#endif
    return 0;
}   // nntpMtWaitConnect



static int nntpMtGetWaiting( int maxThreads=maxNntpThreads )
//
//  look for waiting thread & return ndx
//  return -1, if none is waiting
//  failed/runningspecial threads are skipped
//
{
    int i;

#ifdef TRACE_ALL
    printThreadState( "nntpMtGetWaiting()",maxThreads );
#endif

    assert( maxThreads <= maxNntpThreads );

    for (i = 0;  i < maxThreads;  i++) {
	switch (nntpS[i]) {
	case waiting:
	    return i;
	case runningspecial:
	case failed:
	    if (maxThreads < maxNntpThreads)
		++maxThreads;
	    break;
	default:
	    break;
	}
    }
    return -1;
}   // nntpMtGetWaiting



static int nntpMtAnyRunning( int checkSpecial, int maxThreads=maxNntpThreads )
//
//  return 1, if one thread is 'running', otherwise 0
//  - failed threads are skipped
//  - if checkSpecial is activated, the nntpMtAnyRunning return true also, if
//    there is a 'runningspecial' thread.  Otherwise these threads are skipped
//
{
    int i;

#ifdef TRACE_ALL
    printThreadState( "nntpMtAnyRunning()", maxThreads );
#endif

    assert( maxThreads <= maxNntpThreads );

    for (i = 0;  i < maxThreads;  i++) {
	switch (nntpS[i]) {
	case starting:
	case running:
	    return 1;
	case runningspecial:
	    if (checkSpecial)
		return 1;
	    else if (maxThreads <= maxNntpThreads)
		++maxThreads;
	    break;
	case failed:
	    if (maxThreads <= maxNntpThreads)
		++maxThreads;
	    break;
	default:
	    break;
	}
    }
    return 0;
}   // nntpMtAnyRunning



//--------------------------------------------------------------------------------



static int readNewsrc( const char *name )
{
    if ( !newsrc.readFile(name)) {
	areas.mailPrintf1( 1,"there is no %s file\n",name );
	return 0;
    }
    return 1;
}   // readNewsrc



static void statusInfo( int printIt )
//
//  write status info to mail file.
//  status info contains bytes received, throughput etc.
//  if called with printIt==0, the (internal) stop watch will be reset
//
{
    static clock_t startTime;
    clock_t deltaTime;
    long deltaTimeS10;
    int  i, ok;
    
    if ( !printIt) {
	startTime = clock();
    }
    else {
	//
	//  print status (to mail & console)
	//
	deltaTime = clock() - startTime;
	deltaTimeS10 = (10*deltaTime) / CLOCKS_PER_SEC;
	if (deltaTimeS10 == 0)
	    deltaTimeS10 = 1;
	areas.mailStart( 1 );
	areas.mailPrintf( "%s: totally %ld bytes received, %ld article%s read",
			  progname, TNntp::getBytesRcvd(),
			  (long)artsRcvd, (artsRcvd != 1) ? "s" : "" );
	if (artsKilled != 0)
	    areas.mailPrintf( ", %ld article%s killed", (long)artsKilled,(artsKilled != 1) ? "s" : "" );
	areas.mailPrintf( "\n" );
	areas.mailPrintf( "%s: %ld.%ds elapsed, throughput %ld bytes/s\n",progname,
			  deltaTimeS10 / 10, deltaTimeS10 % 10,
			  (10*TNntp::getBytesRcvd()) / deltaTimeS10 );

	ok = 0;
	for (i = 0;  i < maxNntpThreads;  ++i) {
	    if (nntpS[i] == waiting)
		++ok;
	}
	areas.mailPrintf( "%s: %d threads were connected successfully\n",progname,ok );

	areas.mailStop();
    }
}   // statusInfo



static int checkNntpConnection( int maxThreads, const char *msg )
//
//  check connection to NNTP server
//  if failed return 0, on success return 1
//
{
#ifdef TRACE_ALL
    printfT( "checkNntpConnection(%d)\n",maxThreads );
#endif
    if ( !nntpMtWaitConnect(maxThreads)) {
	areas.mailPrintf1( 1,"%s: cannot connect to news server %s (%s):\n\t%s\n",
			   progname, (nntpInfo.host != NULL) ? nntpInfo.host : "\b", msg,
			   nntp[0].getLastErrMsg() );
	return 0;
    }
    areas.mailPrintf1( 1,"%s: connected to news server %s (%s)\n",
		       progname,nntpInfo.host,msg );
    return 1;
}   // checkNntpConnection



//--------------------------------------------------------------------------------



static int getGroup( const char *groupName )
//
//  Get articles from the newsgroup.
//  Return TRUE if successful.
//
{
    long grpCnt, grpLo, grpFirst, artNum;
    int killEnabled;
    int thread;
    int artRequested;
    int somethingDone;
    int maxThreads;

#ifdef TRACE_ALL
    printfT( "getGroup(%s)\n",groupName );
#endif

    thread = nntpMtGetWaiting();
    assert( thread >= 0 );
#ifdef TRACE_ALL
    printfT( "thread: %d\n",thread );
#endif

    //
    //  Select group name from news server.
    //
    if (nntp[thread].setActGroup(groupName, grpCnt, grpLo, getArt_grpHi) != TNntp::ok) {
	areas.mailPrintf1( 1,"cannot select %s:\n\t%s\n\tunsubscribe group manually\n",
			   groupName,nntp[thread].getLastErrMsg() );
	return 0;
    }

    killEnabled = killF.doKillQ( groupName );
    areas.msgOpen( groupName,"un" );
    
    //
    //  Fix the read article number list (with lo/hi received thru group selection)
    //
    newsrc.grpFixReadList( groupName,grpLo,getArt_grpHi );

#ifdef DEBUG_ALL
    printfT( "group selected: %s %ld-%ld\n",groupName,grpLo,getArt_grpHi );
#endif

    grpFirst = newsrc.grpFirstUnread( groupName,grpLo );
#ifdef DEBUG_ALL
    printfT( "first unread: %ld\n",grpFirst );
#endif
    {
	//
	//  calculate number of articles to fetch (pessimistic version)
	//
	long artCnt = getArt_grpHi-grpFirst+1;
	
	if (getArt_grpHi-grpLo+1 != grpCnt) {
	    if (artCnt > grpCnt)
		artCnt = grpCnt;
	}
	grpCnt = artCnt;
    }
    areas.mailPrintf1( 1,"%s: %4ld unread article%c in %s\n", progname, grpCnt,
		       (grpCnt == 1) ? ' ' : 's', groupName);

    //
    //  calculate reasonable number of threads
    //
#if 0
    maxThreads = grpCnt/2;
    if (maxThreads > maxNntpThreads)
	maxThreads = maxNntpThreads;
    else if (maxThreads < 1)
	maxThreads = 1;
#else
    maxThreads = maxNntpThreads;
#endif
#ifdef DEBUG_ALL
    printfT( "maxThreads: %d\n",maxThreads );
#endif

    //
    //  Look through unread articles
    //
    artNum = grpFirst;
    getArt_artRcvd = 0;
    artRequested = 1;
    while (artNum <= getArt_grpHi  ||  !artRequested  ||  nntpMtAnyRunning(0,maxThreads)) {
#ifdef DEBUG_ALL
	printThreadState( "getGroup()",maxThreads );
#endif
	//
	//  display progress & do timeout
	//
	{
	    static int oldPercent = -1;
	    static long oldKsRcvd = -1;
	    int percent = 100;
	    long ksRcvd;

	    ksRcvd = TNntp::getBytesRcvd() / 1000;
	    if (artNum <= getArt_grpHi)
		percent = ((artNum-grpFirst) * 100) / (getArt_grpHi-grpFirst+1);
	    if (ksRcvd != oldKsRcvd  ||  percent != oldPercent) {
#ifdef DEBUG_ALL
		printfT("%d%% (%05ldk)\n", percent,ksRcvd);
#else
	        printfT("%d%% (%05ldk)\r", percent,ksRcvd);
#endif
		oldKsRcvd  = ksRcvd;
		oldPercent = percent;
	    }
	}

	somethingDone = 0;

	//
	//  if there is a waiting thread, then receive the next article with that one
	//
	thread = nntpMtGetWaiting( maxThreads );
	if (thread >= 0) {
	    //
	    //  find next unread article number
	    //
	    while (artRequested  &&  artNum <= getArt_grpHi) {
		if (newsrc.artIsRead(groupName,artNum)) {
#ifndef NDEBUG
		    printfT( "skip! %ld  \n",artNum );  ////
#endif
		    ++artNum;
		}
		else
		    artRequested = 0;
	    }

	    if ( !artRequested) {
		nntp[thread].selectArticle( groupName,artNum,killEnabled );
		nntpS[thread] = starting;
#ifdef DEBUG_ALL
		printfT( "---------- thread %d/%d(%ld) ----------\n",
			 thread,maxThreads,artNum );
		{
		    int r = BEGINTHREAD( mtGetArticle,(void *)thread );
		    printfT( "thread %d activated: %d,%d\n",thread,r,errno );
		}
#else
		BEGINTHREAD( mtGetArticle,(void *)thread );
#endif
		artRequested = 1;
		somethingDone = 1;
		++artNum;
	    }
	}

#ifdef SIGNAL_PROBLEM
	{
	    int i;
	    for (i = 0;  i < 100;  i++)
		kill( getpid(),SIGUSR1 );
	}
#endif

	if ( !somethingDone) {
	    threadFinito.Wait( 1000 );
	}

	//
	//  Check if too many blocks already
	//  das mu anders gemacht werden... (vielleicht wird dann das lesen von neuen Artikeln verhindert?)
	//
	if (maxBytes > 0  &&  TNntp::getBytesRcvd() >= maxBytes  &&  artNum <= getArt_grpHi) {
	    printfT("%s: maximum packet size exceeded\n", progname);
	    artNum = getArt_grpHi+1;    // trick: initiation of article reading disabled 
	}
    }

    assert( artNum > getArt_grpHi );
    assert( artRequested );
    assert( !nntpMtAnyRunning(0,maxThreads) );
#ifdef DEBUG_ALL
    printfT( "articles read: %s, %ld, %ld\n", groupName,(long)getArt_artRcvd,
	     TNntp::getBytesRcvd() );
#else
    printfT( "\r                              \r" );
#endif

    areas.msgClose();
    return 1;
}   // getGroup



//--------------------------------------------------------------------------------
//
//  handle COMMANDS file
//



static int processSendme( FILE *cmdF )
{
    long grpCnt, grpLo, grpHi, artNum;
    int thread;
    int c;
    int artRequested;
    int somethingDone;
    char buf[BUFSIZ];
    const char *groupName;
    int killEnabled;

#ifdef TRACE_ALL
    printfT( "processSendme()\n" );
#endif

    //
    //  Read newsgroup name.
    //
    if (fscanfT(cmdF, "%s", buf) != 1) {
	fgetsT(buf, sizeof(buf), cmdF);
	return 0;
    }
    groupName = xstrdup( buf );

    thread = nntpMtGetWaiting();
    assert( thread >= 0 );
#ifdef TRACE_ALL
    printfT( "thread: %d\n",thread );
#endif

    //
    //  Select group name from news server.
    //
    if (nntp[thread].setActGroup(groupName, grpCnt,grpLo,grpHi) != TNntp::ok) {
	areas.mailPrintf1( 1,"cannot select %s:\n\t%s\n\tunsubscribe group manually\n",
			   groupName,nntp[thread].getLastErrMsg() );
	fgetsT(buf, sizeof(buf), cmdF);
////    delete groupName;	
	return 0;
    }

    //
    //  if group does not exist in newsrc, add it
    //
    if ( !newsrc.grpExists(groupName)) {
	newsrc.grpAdd( groupName,1 );
	areas.mailPrintf1( 1,"%s added to %s\n",groupName,newsrcFile );
    }

    //
    //  rem:  if articles are selected manually, we assume, that
    //        the user knows which article he/she selects...
    //
//    killEnabled = killF.doKillQ( groupName );
    killEnabled = 0;
    areas.msgOpen( groupName,"un" );

    //
    //  Fix the read article number list
    //
    newsrc.grpFixReadList( groupName, grpLo, grpHi);

#ifdef DEBUG_ALL
    printfT( "group selected: %s %ld-%ld\n",groupName,grpLo,grpHi );
#endif

    printfT( "%s: %s selected\n", progname,groupName );

    c = fgetcT(cmdF);
    artRequested = 1;
    artNum = -1;
    while (artNum < 0  ||  !artRequested  ||  nntpMtAnyRunning(0)) {
#ifdef DEBUG
	printThreadState( "processSendme()" );
#endif
	//
	//  display progress & handle timeout
	//
	{
	    static long oldArtNum = -1;
	    static long oldKsRcvd = -1;
	    long ksRcvd;

	    ksRcvd = TNntp::getBytesRcvd() / 1000;
	    if (ksRcvd != oldKsRcvd  ||  artNum != oldArtNum) {
		if (artNum != -1)
		    printfT( "%05ldk: %ld         \r",ksRcvd,artNum);
		oldKsRcvd = ksRcvd;
		oldArtNum = artNum;
	    }
	}

	somethingDone = 0;

	//
	//  if there is a waiting thread, then receive the article with that one
	//
	thread = nntpMtGetWaiting();
	if (thread >= 0) {
	    //
	    //  get next article number (if any exists)
	    //
	    while (artRequested  &&  c != EOF  &&  c != '\r'  &&  c != '\n') {
		if (fscanfT(cmdF, "%[^ \t\r\n]", buf) != 1) {
		    fgetsT(buf, sizeof(buf), cmdF);
		    c = EOF;
		}
		else {
		    c = fgetcT(cmdF);
		    artNum = atol(buf);
		    if (artNum >= 0) {
			if ( !newsrc.artIsRead(groupName,artNum))
			    artRequested = 0;
		    }
		}
	    }

	    if ( !artRequested) {
		nntp[thread].selectArticle( groupName,artNum,killEnabled );
		nntpS[thread] = starting;
#ifdef DEBUG_ALL
		printfT( "------------- thread %d/%d ------------\n",thread,maxNntpThreads );
#endif
		BEGINTHREAD( mtGetArticle,(void *)thread );
		artRequested = 1;
		somethingDone = 1;
	    }
	}

	if ( !somethingDone)
	    threadFinito.Wait( 1000 );

	//
	//  check if too many block already received
	//
	if (maxBytes > 0  &&  TNntp::getBytesRcvd() >= maxBytes) {
	    printfT( "%s:  maximum packet size exceeded\n",progname );
	    fgetsT(buf, sizeof(buf), cmdF);
	    c = EOF;     // trick:  stop further reading of file...
	}
    }

    assert( artRequested );
    assert( !nntpMtAnyRunning(0) );
////    delete groupName;	
    printfT( "\r                                     \r" );
    areas.msgClose();
    return 1;
}   // processSendme



//--------------------------------------------------------------------------------



int getNews( void )
//
//  If a COMMANDS file exists in the current directory, fetch the articles
//  specified by the sendme commands in the file, otherwise fetch unread
//  articles from newsgroups listed in the newsrc file.
//
{
    FILE *cmdF;

    statusInfo( 0 );

    //
    //  start connecting to nntpServer
    //
    nntpConnect( maxNntpThreads );

    //
    //  Read .newsrc file (may take a while)
    //
    if ( !readNewsrc(newsrcFile))
	return 0;

    //
    //  Read kill file (error msg only, if file was given thru cmdline parameter)
    //
    if (killF.readFile(killFile,maxLines) == -1  &&  killFileOption)
	fprintfT( stderr,"%s: kill file %s not found.\n", progname,killFile );

    //
    //  check connection
    //
    if ( !checkNntpConnection(maxNntpThreads,"getNews"))
	return 0;

#ifdef INSERT_MTTEST
    BEGINTHREAD( _nntpMtConnect, (void *)TESTTHREAD );
    BEGINTHREAD( mtTest, (void *)TESTTHREAD );
#endif
#ifdef SIGNAL_PROBLEM
    signal( SIGUSR1, signalHandler );
#endif

#ifdef DEBUG_ALL
    printfT( "waiting: %d\n", nntpMtGetWaiting() );
#endif
    assert( nntpMtGetWaiting() >= 0 );

#if defined(OS2)  &&  defined(__MT__)
    BEGINTHREAD( nntpMtCheckTimeout, NULL );
#endif

    //
    //  Check for new newsgroups.
    //
    if (doNewGroups) {
	int thread = nntpMtGetWaiting();
	nntpS[thread] = starting;
	BEGINTHREAD( mtGetNewGroups, (void *)thread );
	if ( !nntpMtWaitConnect(maxNntpThreads)) {
	    assert( 0 );  /// darf nicht sein!!!
	}
    }
    assert( nntpMtGetWaiting() >= 0 );

    artsRcvd = 0;
    artsKilled = 0;

    if ((cmdF = fopenT(FN_COMMAND, "rb")) != NULL) {
	//
	//  Process command file containing sendme commands.
	//
	char buf[BUFSIZ];

	doingProcessSendme = 1;
	while (fscanfT(cmdF, "%s", buf) == 1) {
	    if (stricmp(buf, "sendme") == 0) {
		processSendme(cmdF);
		if (maxBytes > 0  &&  TNntp::getBytesRcvd() >= maxBytes)
		    break;
	    }
	    else
		fgetsT(buf, sizeof(buf), cmdF);
	}
	fcloseT(cmdF);
	if ( !readOnly)
	    removeT(FN_COMMAND);
    } else {
	//
	//  For each subscribed newsgroup in .newsrc file
	//
	const char *groupName;

	doingProcessSendme = 0;
	groupName = newsrc.grpFirst();
	while (groupName != NULL) {
	    assert( newsrc.grpSubscribed(groupName) );
	    getGroup(groupName);
	    if (maxBytes > 0  &&  TNntp::getBytesRcvd() >= maxBytes) {
		printfT( "%s: ok, we've read enough...\n",progname );
		break;
	    }
	    groupName = newsrc.grpNext( groupName );
	}
    }

    while (nntpMtAnyRunning(1))
	sleep( 1 );
    BEGINTHREAD( nntpMtDisconnect,(void *)maxNntpThreads );

    statusInfo( 1 );

    if ( !readOnly)
	newsrc.writeFile();

    disconnectDone.Wait( 5000 );    // wait for disconnect (maximum of 5s)
    return 1;
}   // getNews



//--------------------------------------------------------------------------------



static char *nextField(char **ppCur)
//
//  Return next field in record. */
//
{
    char *pEnd;
    char *pStart = *ppCur;

    if ((pEnd = strchr(pStart, '\t')) != NULL) {
	*pEnd++ = '\0';
	*ppCur = pEnd;
    }
    return pStart;
}   // nextField



static void mtSumGroup( void *threadNo )
{
    int no = (int)threadNo;
    long grpCnt,grpLo,grpHi,grpFirst;

#ifdef TRACE
    printfT( "mtSumGroup(%d)\n",no );
#endif
    assert( nntpS[no] == starting );

    mtInitSignals();

    if (nntp[no].setActGroup( nntp[no].groupName(), grpCnt,grpLo,grpHi ) != TNntp::ok) {
	areas.mailPrintf1( 1,"cannot select %s (sumnews):\n\t%s\n",
			   nntp[no].groupName(), nntp[no].getLastErrMsg() );
	goto THREAD_FINISHED;
    }

    //
    //  Fix up the read article number list
    //
    newsrc.grpFixReadList( nntp[no].groupName(),grpLo,grpHi );
    grpFirst = newsrc.grpFirstUnread( nntp[no].groupName(),grpLo );
    {
	//
	//  calculate number of articles to fetch (pessimistic version)
	//
	long artCnt = grpHi-grpFirst+1;
	
	if (grpHi-grpLo+1 != grpCnt) {
	    if (artCnt > grpCnt)
		artCnt = grpCnt;
	}
	grpCnt = artCnt;
    }
    areas.mailPrintf1( 1,"%s: %4ld unread article%c in %s (sumnews)\n", progname, grpCnt,
		       (grpCnt == 1) ? ' ' : 's', nntp[no].groupName());

    if (grpFirst > grpHi)
	goto THREAD_FINISHED;

    nntpS[no] = running;
    if (nntp[no].getOverview(grpFirst,grpHi) != TNntp::ok) {
	areas.mailPrintf1( 1,"cannot get overview of %s (sumnews):\n\t%s\n",
			   nntp[no].groupName(), nntp[no].getLastErrMsg() );
	goto THREAD_FINISHED;
    }

    //
    //  write the collected data to index file
    //
    {
	FILE *inF = nntp[no].getTmpF();
	char buf[BUFSIZ];

#ifdef TRACE_ALL
	printfT( "writing idx of %s\n",nntp[no].groupName() );
#endif
	areas.msgOpen( nntp[no].groupName(),"ic" );
	areas.msgStart();

	fseek( inF,0L,SEEK_SET );
	while (fgets(buf,sizeof(buf),inF) != NULL) {
	    char *cur = buf;
	    long artNum;
	    char *s;

	    buf[strlen(buf)-1] = '\0';                  // remove '\n'
	    artNum = atol(nextField(&cur));          	// article number
	    if ( !newsrc.artIsRead(nntp[no].groupName(),artNum)) {
		newsrc.artMarkRead(nntp[no].groupName(),artNum);    // avoid twice appearance of article !
		s = nextField(&cur);
		areas.msgPrintf( "\t%s\t",s );		// Subject
		s = nextField(&cur);
		areas.msgPrintf( "%s\t",s );		// From
		s = nextField(&cur);
		areas.msgPrintf( "%s\t",s );		// Date
		s = nextField(&cur);
		areas.msgPrintf( "%s\t",s );		// Message-ID
		s = nextField(&cur);
		areas.msgPrintf( "%s\t",s );		// References
		s = nextField(&cur);
		areas.msgPrintf( "0\t" );		// bytes
		s = nextField(&cur);
		areas.msgPrintf( "%s\t",s );		// lines
		areas.msgPrintf( "%ld\n",artNum);	// article number
	    }
	}
	
	areas.msgStop();
	areas.msgClose();
#ifdef TRACE_ALL
	printfT( "writing done of %s\n",nntp[no].groupName() );
#endif
    }

THREAD_FINISHED:
    nntpS[no] = waiting;
    threadFinito.Post();
}   // mtSumGroup



static void sumGroup( const char *groupName )
//
//  on entrance there must be at least one waiting thread
//  on exit this is assured
//
{
#ifdef TRACE
    printfT( "sumGroup(%s)\n",groupName );
#endif
    if (groupName != NULL) {
	int thread = nntpMtGetWaiting();
	assert( thread >= 0 );
	nntp[thread].selectArticle( groupName );
	nntpS[thread] = starting;
	BEGINTHREAD( mtSumGroup,(void *)thread );
    }

    while ((groupName != NULL  &&  nntpMtGetWaiting() < 0)  ||
	   (groupName == NULL  &&  nntpMtAnyRunning(0))) {
	threadFinito.Wait( 1000 );
    }
}   // sumGroup



int sumNews( void )
//
//  Create news summary.
//
{
    const char *groupName;

    statusInfo( 0 );

    //
    //  start connecting to nntpServer
    //
    nntpConnect( maxNntpThreads );

    //
    //  Read .newsrc file (may take a while)
    //
    if ( !readNewsrc(newsrcFile))
	return 0;

    //
    //  check connection
    //
    if ( !checkNntpConnection(maxNntpThreads,"sumNews"))
	return 0;

    assert( nntpMtGetWaiting() >= 0 );

#if defined(OS2)  &&  defined(__MT__)
    BEGINTHREAD( nntpMtCheckTimeout, NULL );
#endif

    //
    //  For each subscribed newsgroup in the .newsrc file
    //
    groupName = newsrc.grpFirst();
    while (groupName != NULL) {
	assert( newsrc.grpSubscribed(groupName) );
	sumGroup( groupName );
	groupName = newsrc.grpNext( groupName );
    }
    sumGroup( NULL );

    BEGINTHREAD( nntpMtDisconnect, (void *)maxNntpThreads );
    statusInfo( 1 );
    if ( !readOnly)
	newsrc.writeFile();
    disconnectDone.Wait( 5000 );
    return 1;
}   // sumNews



//--------------------------------------------------------------------------------



static void mtCatchup( void *threadNo )
{
    int no = (int)threadNo;
    long grpCnt,grpLo,grpHi;

#ifdef TRACE
    printfT( "mtCatchup(%d)\n",no );
#endif
    assert( nntpS[no] == starting );

    mtInitSignals();

    nntpS[no] = running;

    if (nntp[no].setActGroup( nntp[no].groupName(), grpCnt,grpLo,grpHi ) != TNntp::ok) {
	areas.mailPrintf1( 1,"cannot select %s (catchup):\n\t%s\n",
			   nntp[no].groupName(), nntp[no].getLastErrMsg() );
	goto THREAD_FINISHED;
    }

    //
    //  catch up the read article number list
    //
    if (grpHi-catchupNumKeep >= grpLo) {
	grpHi -= catchupNumKeep;
	newsrc.grpCatchup( nntp[no].groupName(), 1,grpHi );
	printfT("%s: catched up %s:  1-%ld\n", progname, nntp[no].groupName(),grpHi );
    }
    else
	newsrc.grpFixReadList( nntp[no].groupName(),grpLo,grpHi );

THREAD_FINISHED:
    threadFinito.Post();
    nntpS[no] = waiting;
}   // mtCatchup



int catchupNews( long numKeep )
//
//  Catch up in subscribed newsgroups.
//
{
    const char *groupName;

    statusInfo( 0 );
    
    catchupNumKeep = numKeep;             // nicht besonders fein...

    //
    //  start connecting to nntpServer
    //
    nntpConnect( maxNntpThreads );

    //
    //  read .newsrc file (may take a while)
    //
    if ( !readNewsrc(newsrcFile))
	return 0;

    //
    //  check connection
    //
    if ( !checkNntpConnection(maxNntpThreads,"catchupNews"))
	return 0;

    assert( nntpMtGetWaiting() >= 0 );

#if defined(OS2)  &&  defined(__MT__)
    BEGINTHREAD( nntpMtCheckTimeout, NULL );
#endif

    //
    //  For each subscribed newsgroup in the .newsrc file
    //
    groupName = newsrc.grpFirst();
    while (groupName != NULL) {
	int thread;

	assert( newsrc.grpSubscribed(groupName) );

	thread = nntpMtGetWaiting();
	assert( thread >= 0 );
	nntp[thread].selectArticle( groupName );
	nntpS[thread] = starting;
	BEGINTHREAD( mtCatchup,(void *)thread );

	while (nntpMtGetWaiting() < 0)
	    threadFinito.Wait( 1000 );

	groupName = newsrc.grpNext( groupName );
    }
    while (nntpMtAnyRunning(0))
	threadFinito.Wait( 1000 );

    BEGINTHREAD( nntpMtDisconnect, (void *)maxNntpThreads );
    statusInfo( 1 );

    if ( !readOnly)
	newsrc.writeFile();

    disconnectDone.Wait( 5000 );
    return 1;
}   // catchupNews
