// animation.cpp : implementation file
//

#include "stdafx.h"
#include "anim.h"

#include "error.h"

int CAnim::DisplayWidth = 640;  // defaults to 640 if not set

//-----------------------------------------------------------------------------
// SetDisplayWidth
//-----------------------------------------------------------------------------
void CAnim::SetDisplayWidth(int nDisplayWidth)
{
  CAnim::DisplayWidth = nDisplayWidth;
}

//-----------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------
CAnim::CAnim(LPDIRECTDRAW pDirectDraw,
             char* pFileRoot,
             char* pDirectoryName,
             int nNumFrames,
             BOOL bStoreInVideo) // = TRUE
{
  m_pDirectDraw = pDirectDraw;
  m_nNumFrames = nNumFrames;
  m_bInVideo = bStoreInVideo;

  // create new string 5 characters longer that dir name: 4 for
  // file name, and 1 for terminating NULL
  m_pFileRoot = new char[strlen(pDirectoryName) + 5];
  strcpy(m_pFileRoot, pDirectoryName);
  strncat(m_pFileRoot, pFileRoot, 4);

  m_pnRows = new int[nNumFrames];
  m_pnCols = new int[nNumFrames];

  m_pSurface = NULL;
}

//-----------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------
CAnim::~CAnim()
{
  delete [] m_pFileRoot;
  delete [] m_pnRows;
  delete [] m_pnCols;
}
                               
//-----------------------------------------------------------------------------
// Load
//-----------------------------------------------------------------------------
BOOL CAnim::Load()
{
  return InternalLoad(FALSE);
}

//-----------------------------------------------------------------------------
// Release
//-----------------------------------------------------------------------------
void CAnim::Release()
{
  m_pSurface->Release();
}

//-----------------------------------------------------------------------------
// Restore
//-----------------------------------------------------------------------------
BOOL CAnim::Restore()
{
  m_pSurface->Restore();

  return InternalLoad(TRUE);
}

//-----------------------------------------------------------------------------
// Render
//-----------------------------------------------------------------------------
HRESULT CAnim::Render(int nDestX,
                      int nDestY,
                      int nFrameNum,
                      LPDIRECTDRAWSURFACE pDestSurface,
                      BOOL bTransparent) // = TRUE
{
  HRESULT hr;
  CRect srcRect(m_pnCols[nFrameNum], m_pnRows[nFrameNum],
                m_pnCols[nFrameNum] + m_nWidth, m_pnRows[nFrameNum] + m_nHeight);

  if (bTransparent)
    hr = pDestSurface->BltFast(nDestX, nDestY, m_pSurface, &srcRect,
                               DDBLTFAST_SRCCOLORKEY);
  else
    hr = pDestSurface->BltFast(nDestX, nDestY, m_pSurface, &srcRect,
                               DDBLTFAST_NOCOLORKEY);

  #ifdef _DEBUG
  if (hr != DD_OK)
    DDrawError("CAnim::Render BltFast::", hr);
  #endif
  
  return hr;
}

//-----------------------------------------------------------------------------
// InternalLoad
//-----------------------------------------------------------------------------
BOOL CAnim::InternalLoad(BOOL bRestore)
{
  HBITMAP hBitmap;
  BITMAP bitmap;
  DDSURFACEDESC ddSurfDesc;
  HRESULT hResult;
  HDC hdcSource;
  HDC hdcDest;

  char* pFileName = new char[strlen(m_pFileRoot) + 9];

  for (int nIndex = 0; nIndex < m_nNumFrames; nIndex++)
  {
    wsprintf(pFileName, "%s%04d.bmp", m_pFileRoot, nIndex);

    // load bitmap
    hBitmap = (HBITMAP)LoadImage(NULL, pFileName, IMAGE_BITMAP,
                                 0, 0, LR_LOADFROMFILE|LR_CREATEDIBSECTION);
    if (!hBitmap)
    {
      if (m_pSurface)
        m_pSurface->Release();
      delete [] pFileName;
      return FALSE;
    }

    // create the dd surface the first time through
    if (0 == nIndex && !bRestore)
    {
      GetObject(hBitmap, sizeof(bitmap), &bitmap);
      m_nWidth = bitmap.bmWidth;
      m_nHeight = bitmap.bmHeight;

      if (m_nNumFrames * m_nWidth > CAnim::DisplayWidth)
      {
        m_nNumCols = CAnim::DisplayWidth / m_nWidth;
        m_nNumRows = int(float(m_nNumFrames) / float(m_nNumCols) + 1.0);
      }
      else
      {
        m_nNumCols = m_nNumFrames;
        m_nNumRows = 1;
      }

      ddSurfDesc.dwSize = sizeof(ddSurfDesc);
      ddSurfDesc.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
      ddSurfDesc.dwWidth = m_nWidth * m_nNumCols;
      ddSurfDesc.dwHeight = m_nHeight * m_nNumRows;

      if (m_bInVideo)
        ddSurfDesc.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
      else
        ddSurfDesc.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;

      hResult = m_pDirectDraw->CreateSurface(&ddSurfDesc, &m_pSurface, NULL);
      
      if (hResult != DD_OK)
      {
        DDrawError("CAnim::Load CreateSurface:", hResult);
        delete [] pFileName;
        return FALSE;
      }
      
      DDCOLORKEY ddck;
      ddck.dwColorSpaceLowValue = RGB(0,0,0);
      ddck.dwColorSpaceHighValue = RGB(0,0,0);
      m_pSurface->SetColorKey(DDCKEY_SRCBLT, &ddck);
    }

    m_pnRows[nIndex] = nIndex / m_nNumCols * m_nHeight;
    m_pnCols[nIndex] = nIndex % m_nNumCols * m_nWidth;

    hResult = m_pSurface->GetDC(&hdcDest);
    if (DD_OK == hResult)
    {
      ASSERT(m_pSurface != NULL);
      hdcSource = CreateCompatibleDC(hdcDest);
      SelectObject(hdcSource, hBitmap);

      StretchBlt(hdcDest, 
                 m_pnCols[nIndex], m_pnRows[nIndex],
                 m_nWidth, m_nHeight,
                 hdcSource,
                 0, 0,
                 m_nWidth, m_nHeight,
                 SRCCOPY);
      m_pSurface->ReleaseDC(hdcDest);
    }

    DeleteDC(hdcSource);
    DeleteObject(hBitmap);
  }

  // everything seems ok
  delete [] pFileName;
  return TRUE;
}
