{
                                 

  BOARD.CH  include module for CHESS.PAS
  Last modified:  10/29/85

  This module contains the types and variables used to represent
  a position on the board.

                                 

    Board and PieceTab represent the chess positions.
    Player contains the Player to Move.
    MovTab contains the moves played so far.

  These variables represent the state of the game.

  The procedure Perform makes a Move and updates Board and
  PieceTab.   Procedure Perform and Edit are the only ways
  these variables may be changed.
}

type  { Squarenumbers. a1=0, b1=1,..., a2=$10,..., h8=$77 }

      SquareType     =    0..$77;   { The 64 squares }
      EdgeSquareType = -$21..$98;

      ColorType      = (White, Black);
      PieceType      = (Empty,
                        King, Queen, Rook, Bishop, Knight, Pawn);

      IndexType      = 0..15;       { Index in PieceTab }

      BoardType      = record
                          Piece : PieceType;   { PieceType }
                          Color : ColorType;   { Color }
                          Index : 0..16;       { Index to PieceTab }
                       end;

var   { Board contains the Content of each Square,
        Square by Square }
      Board : array[SquareType] of BoardType;

      { PieceTab contains the squares of all the Pieces,
        Piece by Piece.
        Board and PieceTab is two different representations of the
        same position, and they are always changed simultaniously.

        No. 0 is the King,
        No. 1 - OfficerNo is the officers and evt. Pawns,
        No. OfficerNo + 1 - PawnNo is the pawns }

      PieceTab : array[ColorType,IndexType] of
                   record
                      ISquare : SquareType;  { Square and Index to Board }
                      IPiece :  PieceType;                   { PieceType }
                   end;

      { Indexes to PieceTab, used to speed Up the program a Bit }
      OfficerNo, PawnNo : array[ColorType] of - 1..15;

var
  Player, Opponent : ColorType;      { Player is always Next to Move }

type  { The MoveType, which is used to represent
        all moves in the program }
      MoveType = record
                    New1, Old : SquareType; { New1 and Old Square }
                    Spe:      boolean;    { Indicates special Move:
                                            case MovPiece of
                                             King: Castling
                                             Pawn: E.p. capture
                                             else : Pawnpromotion }
                    MovPiece,              { Moving Piece }
                    Content :  PieceType;  { Evt. captured Piece }
                 end;

const { The undefined Move. Testing is Done using MovPiece=Empty }
      ZeroMove : MoveType = (New1    : 8;
                            Old      : 8;
                            Spe      : False;
                            MovPiece : Empty;
                            Content  : Empty);

{ The played moves.
  MovTab[Back..Depth] contains all the moves played so far }
type  DepthType = Back..MaxPly;
var   Depth :  DepthType;
      MovTab : array[DepthType] of MoveType;

function EqMove(var a,b : MoveType) : boolean;
{ Compares two moves }
begin
   EqMove := False;
   if a.MovPiece = b.MovPiece then
     if a.New1 = b.New1 then
       if a.Old = b.Old then
         if a.Content = b.Content then
           if a.Spe = b.Spe then
             EqMove := true;
end; { EqMove }

function Min(a,b : integer) : integer;
begin
  if a < b then Min := a
  else Min := b;
end; { Min }

function Max(a,b : integer) : integer;
begin
  if a > b then Max := a
  else Max := b;
end; { Max }

procedure ClearBoard;
{ Clears the Board and initializes the Board-module }
var   Square : SquareType;
begin
  for Square := 0 to $77 do
    with Board[Square] do
    begin
      Piece := Empty;
      Color := White;
    end;
end { ClearBoard };

procedure CalcPieceTab;
{ Calculates PieceTab from scratch }
var   Square : SquareType;
      Piece1 : PieceType;

procedure ClearIndex;
{ Clears indexes in Board and PieceTab }
var Square : SquareType;
    Col :    ColorType;
    Index :  IndexType;
begin
   for Square := 0 to $77 do
      Board[Square].Index := 16;
   for Col := White to Black do for Index := 0 to 15 do
      PieceTab[Col,Index].IPiece := Empty;
   PawnNo[White] := -1;
   PawnNo[Black] := -1;
   OfficerNo := PawnNo;
end;

begin
  ClearIndex;
  for Piece1 := King to Pawn do    { Insert all the Pieces of the type }
  begin
     if Piece1 = Pawn then
        OfficerNo := PawnNo;      { Save Number of officers }
     Square := 0;
     repeat
       with Board[Square] do
         if Piece = Piece1 then
         begin
           PawnNo[Color] := PawnNo[Color] + 1;          { Count Pieces }
           with PieceTab[Color,PawnNo[Color]] do        { Insert Piece }
           begin
             IPiece := Piece1;
             ISquare := Square;
             Index := PawnNo[Color];
           end;
         end;
         Square := Square xor $77;     { Generate all squares from }
         if (Square and 4) = 0 then    {  border to center }
            if Square >= $70 then
               Square := (Square + $11) and $73
            else
               Square := Square + $10;
     until Square = 0;
   end;
end; { CalcPieceTab }

procedure GenCastSquare(New1 : SquareType;
            var CastSquare,
                CornerSquare : SquareType);
{ Calculates the squares for the Rook Move in a castling }
begin
   if (New1 and 7) >= 4 then     { Short }
   begin
      CastSquare := New1 - 1;
      CornerSquare := New1 + 1;
   end
   else
   begin                         { Long }
     CastSquare := New1 + 1;
     CornerSquare := New1 - 2;
   end;
end; { GenCastSquare }

procedure Perform(Move : MoveType; ResetMove : boolean);
{ Performs or takes Back Move (takes Back if ResetMove if true),
  and performs the updating of Board and PieceTab. Player must
  contain the Color of the moving Player, Opponent the Color
  of the Opponent.

  MovePiece, DeletePiece, InsertPiece and ChangeType
  is used to Update the Board module
}

   procedure MovePiece(New1,Old : SquareType);
   { Is used to Move a Piece }
   var   b : BoardType;
   begin
      b := Board[New1];
      Board[New1] := Board[Old];
      Board[Old] := b;
      with Board[New1] do
         PieceTab[Color,Index].ISquare := New1;
   end; { MovePiece }

   procedure DeletePiece(InSquare : SquareType);
   { Is used in captures. The Square must not be Empty }
   begin
      with Board[InSquare] do begin
         Piece := Empty;
         PieceTab[Color,Index].IPiece := Empty;
      end;
   end; { DeletePiece }

   procedure InsertPiece(InPiece : PieceType;
                         InColor : ColorType;
                        InSquare : SquareType);
   { Is used to take Back captures }
   begin
      with Board[InSquare],PieceTab[InColor,Index] do
      begin
         Piece := InPiece;
         Color := InColor;
         IPiece := InPiece;
         ISquare := InSquare;
      end;
   end; { InsertPiece }

   procedure ChangeType(NewType : PieceType; InSquare : SquareType);
   { Is used for Pawn promotion }
   begin
      with Board[InSquare] do
      begin
         Piece := NewType;
         PieceTab[Color,Index].IPiece := NewType;
         if OfficerNo[Color] < Index then
            OfficerNo[Color] := Index;
      end;
   end; { ChangeType }

var
  New1,CastSquare,CornerSquare,EpSquare : SquareType;
begin
  with Move do
  begin
    if ResetMove then              { Perform Move }
    begin
      MovePiece(Old,New1);
      if Content <> Empty then
        InsertPiece(Content,Opponent,New1);
    end
    else
    begin
      if Content <> Empty then DeletePiece(New1);
      MovePiece(New1,Old);
    end;
    if Spe then                   { Test if Move is special }
      if MovPiece = King then
      begin
        GenCastSquare(New1,CastSquare,CornerSquare); { Castling Move }
        if ResetMove then
          MovePiece(CornerSquare,CastSquare)
        else
          MovePiece(CastSquare,CornerSquare);
      end
      else
        if MovPiece = Pawn then
        begin
          EpSquare := (New1 and 7) + (Old and $70);    { E.p. capture }
          if ResetMove then
            InsertPiece(Pawn,Opponent,EpSquare)
          else
            DeletePiece(EpSquare);
        end
        else                                          { Pawn-promotion }
          if ResetMove then
            ChangeType(Pawn,Old)
          else
            ChangeType(MovPiece,New1);
  end; { with }
end; { Perform }
