/*
	squares.C

	Version 1.6

	Squares Game

	Copyright (C) 1987-1995, Geoff Friesen
	All rights reserved.

	Developed with: Borland C++ 3.1
*/

/*
	set ti=c:\borlandc\tsrlib
	t squares -hs -i -s512 -x

	The BIOS memory location 0040:0084 is set to the number of
	screen rows less one.  For example, if the screen has been
	set to 25 lines then the value 24 would be stored here.  A
	value of 49 would be stored if 50-line mode were in effect
	(on VGA adaptors).  This address is used by recent adaptor
	BIOSes.  Older BIOSes do not use it.  Therefore, assume it
	to be valid if it returns a 24, 42, or 49.  The 42 is used
	in EGA 43-line mode.
*/

#if !defined(__TINY__)
#error Tiny Memory Model Expected
#endif

#include "tsrlib.H"

char *title = "\r\n"

"ͻ\r\n"
"                                 \r\n"
"                                          \r\n"
"                                            \r\n"
"                   Version 1.6 \r\n"
"                                           \r\n"
"                                         \r\n"
"                                 \r\n"
"                                                       \r\n"
" Copyright (C) 1987-1995, Geoff Friesen                \r\n"
" All rights reserved.                                  \r\n"
"ͼ\r\n"
"$";

int sound_enabled = 1;          /* sound flag (nonzero means sound on)  */

int stars [] =                  /* Stars and Stripes Forever (WIN)      */
{
   589, 6, 589, 6, 525, 3, 495, 3, 495, 6, 467, 3, 495, 3, 495, 16,
   32767, 2, 467, 3, 495, 3, 495, 6, 467, 3, 495, 3, 589, 6, 495, 3,
   589, 3, 525, 12, 441, 6, 32767, 3, 441, 3, 441, 6, 416, 3, 441, 3,
   441, 6, 416, 3, 441, 3, 525, 16, 32767, 2, 495, 3, 441, 3, 495, 3,
   589, 9, 661, 9, 661, 3, 441, 16, 32767, 2, 589, 6, 589, 6, 525, 3,
   495, 3, 495, 6, 467, 3, 495, 3, 495, 16, 32767, 2, 467, 3, 495, 3,
   495, 6, 467, 3, 495, 3, 525, 3, 495, 3, 441, 5, 371, 1, 441, 12,
   393, 6, 32767, 3, 393, 3, 393, 6, 371, 3, 393, 3, 467, 6, 441, 3,
   393, 3, 786, 15, 32767, 3, 393, 3, 441, 3, 495, 3, 589, 1, 32767, 2,
   393, 3, 441, 3, 495, 3, 589, 1, 32767, 2, 294, 3, 330, 5, 495, 1,
   441, 12, 393, 1, -1
};

int funeral [] =                /* Funeral March of a Marionette (LOSE) */
{
   221, 1, 32767, 2, 147, 1, 32767, 5, 294, 3, 294, 3, 278, 3, 248, 3,
   278, 3, 32767, 3, 294, 3, 330, 3, 32767, 3, 221, 1, 32767, 2, 147, 1,
   32767, 5, 294, 3, 294, 3, 278, 3, 248, 3, 278, 3, 32767, 3, 294, 3,
   330, 3, 32767, 3, 221, 3, 294, 3, 32767, 3, 350, 3, 441, 6, 393, 3,
   350, 3, 32767, 3, 441, 3, 525, 6, 467, 3, 441, 3, 32767, 3, 556, 3,
   661, 6, 589, 3, 556, 3, 467, 3, 441, 3, 393, 3, 350, 3, 330, 3, 147, 1,
   32767, 5, 294, 3, 294, 3, 278, 3, 248, 3, 278, 3, 32767, 3, 294, 3,
   330, 3, 32767, 3, 221, 1, 32767, 2, 147, 1, 32767, 5, 294, 3, 294, 3,
   278, 3, 248, 3, 278, 3, 32767, 3, 294, 3, 330, 3, 32767, 3, 221, 3,
   350, 3, 32767, 3, 441, 3, 525, 6, 467, 3, 441, 3, 393, 3, 350, 3,
   312, 3, 393, 3, 467, 3, 294, 3, 278, 3, 294, 3, 330, 3, 32767, 3,
   350, 1, 32767, 2, 330, 9, 294, 1, -1
};

int square_colors [] =
{
    BLUE,
    GREEN,
    CYAN,
    MAGENTA,
    BROWN,
    LIGHTBLUE,
    LIGHTGREEN,
    LIGHTCYAN,
    LIGHTMAGENTA,
    YELLOW,
    RED
};

char squares [9] = { 0 };

char *template [] =
{
   "ͻ",
   "                    ",
   "͹",
   "                  ",
   "                  ",
   "͹",
   "                  ",
   "                  ",
   "͹",
   "                  ",
   "                  ",
   "͹",
   "                    ",
   "                    ",
   "ͼ"
};

#define TEMPLATE_COLS   22
#define TEMPLATE_ROWS   15

int srow = -1;                          /* suggested row */
int template_x = ((80-TEMPLATE_COLS)/2);
int template_y;

#define CELL_SIZE_W     6
#define CELL_SIZE_H     2

struct
{
   int x, y;
}
origins [9] =
{
   {  1, 9 },
   {  8, 9 },
   { 15, 9 },
   {  1, 6 },
   {  8, 6 },
   { 15, 6 },
   {  1, 3 },
   {  8, 3 },
   { 15, 3 }
};

int tindex = 0;                         /* timer index (tindex*30 sec) */

int bw [] =
{
   WHITE,                               /* template */
   LIGHTGRAY,                           /* square */
   WHITE,                               /* title and key ref */
   WHITE,                               /* help border */
   LIGHTGRAY << 4,                      /* help title */
   LIGHTGRAY                            /* help text and timer */
};

int color [] =
{
   GREEN,                               /* template */
   RED,                                 /* square */
   LIGHTBLUE,                           /* title and key ref */
   BLUE,                                /* help border */
   GREEN,                               /* help title */
   CYAN                                 /* help text and timer */
};

#define NATTR   (sizeof(bw)/sizeof(bw [0]))

int attr [NATTR];

char *help_text [] =
{
   "You play the SQUARES game on a 3x3 board.  The object is simple.  All",
   "squares must be set to BLACK.  Use the numeric keypad to toggle these",
   "squares.  Figure A shows the board layout.",
   " ",
   "Various squares are toggled on and off by pressing one of the keys on",
   "the numeric keypad.  For example, pressing the 1 key toggles the fol-",
   "lowing squares (those marked with asterisks).  See figure B.",
   " ",
   "In fact, any corner key toggles the four squares in that corner.  The",
   "edge keys (2, 4, 6, and 8) toggle the squares along that edge.  As an",
   "example, pressing 2 results in these squares (fig. C) being toggled.",
   "The central squares are toggled by pressing the 5 key (see figure D).",
   " ",
   "Press ENTER to scramble the board.  You can set your keypad to number",
   "mode by pressing NUMLOCK (the NUM LOCK light will appear in this mode",
   "for keyboards that contain NUM LOCK, CAPS LOCK, and the SCROLL LOCK",
   "lights).  Press C to change the square color.",
   " ",
   "A. ͻ B. ͻ C. ͻ D. ͻ",
   "    7  8  9                                   *    ",
   "   ͹    ͹    ͹    ͹",
   "    4  5  6      *  *                      *  *  * ",
   "   ͹    ͹    ͹    ͹",
   "    1  2  3      *  *         *  *  *         *    ",
   "   ͼ    ͼ    ͼ    ͼ",
   " ",
   "SQUARES v1.6 supports the mouse.  Press the left button on any square",
   "to toggle the appropriate combination of squares, on F1-Help to bring",
   "up the help screen, on ESC-Exit to exit SQUARES, or on C-Set Square",
   "Color to change the square color.  When the help screen is displayed,",
   "press the left button on the up/down arrow or ESC.  Press the right",
   "button to change the square color.  Holding down the left button will",
   "cause this button to auto repeat.",
   " ",
   "When running SQUARES from the DOS prompt (aka command line), a number",
   "of command line arguments can be specified including:",
   " ",
   "SQUARES /B0  disable hotkey trigger beep",
   "SQUARES /B1  enable hotkey trigger beep",
   "SQUARES /C0  set squares to BLUE color",
   "SQUARES /C1  set squares to GREEN color",
   "SQUARES /C2  set squares to CYAN color",
   "SQUARES /C3  set squares to MAGENTA color",
   "SQUARES /C4  set squares to BROWN color",
   "SQUARES /C5  set squares to LIGHTBLUE color",
   "SQUARES /C6  set squares to LIGHTGREEN color",
   "SQUARES /C7  set squares to LIGHTCYAN color",
   "SQUARES /C8  set squares to LIGHTMAGENTA color",
   "SQUARES /C9  set squares to YELLOW color",
   "SQUARES /Kx  change hotkey (x is A-Z or 0-9)",
   "SQUARES /R   reset certain values to defaults",
   "SQUARES /S0  disable game sound",
   "SQUARES /S1  enable game sound",
   "SQUARES /Tx  set timer (x is 1 to 9 for 30-270 secs or 0 to disable)",
   "SQUARES /U   uninstall",
   "SQUARES /Xx  set left-most column (1-58) of game window",
   "SQUARES /Yx  set upper-most row (1-35) of game window",
   " ",
   "Options can be combined.  Example: SQUARES /B1 /C3 /K5 /S0",
   NULL
};

void    help                    (char *title, char **ht, int tlattr,
				 int bdattr, int txattr);
int     message                 (char *title);
int     m_range                 (event *e, int x1, int y1, int x2, int y2);
void    paint_squares           (void);
int     play                    (int *song);
void    randomize_squares       (void);
void    reset                   (void);
void    update_timer            (void);

void main (void)
{
   event e;
   int cshape, i, nrows, vmode, x, y;
   static char buffer [(TEMPLATE_COLS+1)*(TEMPLATE_ROWS+1)*2];

   if ((vmode = v_getmode ()) != BW80 && vmode != C80 && vmode != MONO)
       return;

   e_open ();

   e_settrigger (6);

   randomize ();

   nrows = peekb(0x40, 0x84)+1;
   if (nrows != 25 && nrows != 43 && nrows != 50)
       nrows = 25;

   template_y = (srow == -1 || srow+TEMPLATE_ROWS > nrows)
		? (nrows-TEMPLATE_ROWS) >> 1 : srow;

   for (i = 0; i < NATTR; i++)
	attr [i] = (vmode == C80) ? color [i] : bw [i];

   cshape = v_getshape ();
   v_setshape (0x2000);         /* hide cursor */

   x = v_wherex ();
   y = v_wherey ();

   v_screen (SCREEN_SAVE, template_x, template_y,
	     TEMPLATE_COLS+1, TEMPLATE_ROWS+1, buffer);

   v_setattr (attr [0]);

   for (i = 0; i < TEMPLATE_ROWS; i++)
   {
	v_gotoxy (template_x, template_y+i);
	v_cputs (template [i]);
   }

   v_shadow (template_x, template_y, TEMPLATE_COLS, TEMPLATE_ROWS);

   v_setattr (attr [2]);
   v_gotoxy (template_x+((tindex) ? 2 : 5), template_y+1);
   v_cputs ("Squares v1.6");
   v_gotoxy (template_x+1, template_y+TEMPLATE_ROWS-3);
   v_cputs ("F1 - Help/ESC - Exit");
   v_gotoxy (template_x+1, template_y+TEMPLATE_ROWS-2);
   v_cputs ("C - Set Square Color");

   m_moveto (template_x << 3, template_y << 3);
   e_show ();
   reset ();

   while (1)
   {
      e_fetch (&e);

      if (e.type == TIMER)
      {
	  e_hide ();
	  update_timer ();
	  e_show ();

	  if (e.parm1 == 1)
	  {
	      if (message ("SORRY") == 'N')
		  break;

	      reset ();
	  }

	  continue;
      }

      if (e.type == KEY && toupper (e.parm1) == 'C')
      {
	  e.type = MOUSE;
	  e.parm1 = ME_RBP;
      }

      if (e.type == MOUSE)
      {
	  if (e.parm1 & ME_LBP)
	      if (m_range (&e, template_x+1, template_y+13,
			   template_x+20, template_y+13))
		  e.parm1 = ME_RBP;

	  if (e.parm1 & ME_RBP)
	  {
	      if (vmode == C80)
	      {
		  for (i = 0; i < 11; i++)
		       if (attr [1] == square_colors [i])
			   break;

		  if (++i > 10)
		      i = 0;

		  color [1] = attr [1] = square_colors [i];
	      }
	      else
		  bw [1] = attr [1] = (attr [1] == LIGHTGRAY) ? WHITE
			   : LIGHTGRAY;

	      paint_squares ();
	      continue;
	  }

	  if (e.parm1 & ME_LBP)
	  {
	      for (i = 0; i < 9; i++)
		   if (m_range (&e, template_x+origins [i].x,
				template_y+origins [i].y,
				template_x+origins [i].x+CELL_SIZE_W-1,
				template_y+origins [i].y+CELL_SIZE_H-1))
		       break;

	      if (i < 9)
	      {
		  e.type = KEY;
		  e.parm1 = '1'+i;
	      }
	      else
	      if (m_range (&e, template_x+1, template_y+12,
			   template_x+9, template_y+12))
	      {
		  e.type = KEY;
		  e.parm1 = F1;
	      }
	      else
	      if (m_range (&e, template_x+11, template_y+12,
			   template_x+20, template_y+12))
	      {
		  e.type = KEY;
		  e.parm1 = ESC;
	      }
	  }
      }

      if (e.type == KEY)
      {
	  if (e.parm1 == F1)
	  {
	      help (" SQUARES v1.6 HELP ", help_text, attr [4],
		    attr [3], attr [5]);

	      continue;
	  }

	  if (e.parm1 == ESC)
	      break;

	  if (e.parm1 == RET)
	  {
	      reset ();
	      continue;
	  }

	  if (e.parm1 < '1' || e.parm1 > '9')
	      continue;

	  switch (e.parm1)
	  {
	     case '1' : squares [0] = !squares [0];
			squares [1] = !squares [1];
			squares [3] = !squares [3];
			squares [4] = !squares [4];
			break;

	     case '2' : squares [0] = !squares [0];
			squares [1] = !squares [1];
			squares [2] = !squares [2];
			break;

	     case '3' : squares [1] = !squares [1];
			squares [2] = !squares [2];
			squares [4] = !squares [4];
			squares [5] = !squares [5];
			break;

	     case '4' : squares [0] = !squares [0];
			squares [3] = !squares [3];
			squares [6] = !squares [6];
			break;

	     case '5' : squares [0] = !squares [0];
			squares [2] = !squares [2];
			squares [4] = !squares [4];
			squares [6] = !squares [6];
			squares [8] = !squares [8];
			break;

	     case '6' : squares [2] = !squares [2];
			squares [5] = !squares [5];
			squares [8] = !squares [8];
			break;

	     case '7' : squares [3] = !squares [3];
			squares [4] = !squares [4];
			squares [6] = !squares [6];
			squares [7] = !squares [7];
			break;

	     case '8' : squares [6] = !squares [6];
			squares [7] = !squares [7];
			squares [8] = !squares [8];
			break;

	     case '9' : squares [4] = !squares [4];
			squares [5] = !squares [5];
			squares [7] = !squares [7];
			squares [8] = !squares [8];
	  }

	  paint_squares ();

	  for (i = 0; i < 9; i++)
	       if (squares [i])
		   break;

	  if (i == 9)
	  {
	      if (message ("CONGRATULATIONS") == 'N')
		  break;

	      reset ();
	  }
      }
   }

   e_hide ();

   v_screen (SCREEN_RESTORE, template_x, template_y,
	     TEMPLATE_COLS+1, TEMPLATE_ROWS+1, buffer);

   v_gotoxy (x, y);
   v_setshape (cshape);

   e_close ();
}

void help (char *title, char **ht, int tlattr, int bdattr, int txattr)
{
   event e;
   WINDOW w;
   static char buffer [12*72*2];
   int anchor = 0, oldtrigger;
   int finished = 0, i = 0, maxdlines = 12, maxlines = 0, minlines, x, y;

   e_tsuspend (0);
   e_tsuspend (1);

   oldtrigger = e_gettrigger ();
   e_settrigger (1);

   while (ht [i++])
      maxlines++;

   if (maxlines < maxdlines)
       maxdlines = maxlines;

   w.x = 5;
   w.y = 5;
   w.nx = 72;
   w.ny = maxdlines+2;
   w.bdattr = bdattr;
   w.tlattr = tlattr;
   w.txattr = txattr;
   w.flags = WINF_VBAR;
   w.title = &title [0];
   w.scrn_buffer = &buffer [0];

   e_hide ();

   w_create (&w);

   for (i = 0; i < maxdlines; i++)
   {
	w_gotoxy (&w, 1, i+1);
	w_puts (&w, ht [i]);
   }

   e_show ();

   e_flush ();

   while (!finished)
   {
      e_fetch (&e);

      if (e.type == MOUSE)
	  if (e.parm1 & ME_LBP)
	  {
	      if (m_range (&e, w.x+w.nx-1, w.y+1, w.x+w.nx-1, w.y+1))
	      {
		  e.type = KEY;
		  e.parm1 = UP;
	      }
	      else
	      if (m_range (&e, w.x+w.nx-1, w.y+w.ny-2, w.x+w.nx-1, 
		  w.y+w.ny-2))
	      {
		  e.type = KEY;
		  e.parm1 = DOWN;
	      }
	      else
	      if (m_range (&e, w.x+2, w.y, w.x+2, w.y))
	      {
		  e.type = KEY;
		  e.parm1 = ESC;
	      }
	      else
		  continue;
	  }

      if (e.type == KEY)
	  switch (e.parm1)
	  {
	     case DOWN : if (maxlines > maxdlines)
			 {
			     if (anchor+maxdlines == maxlines)
				 break;

			     anchor++;

			     e_hide ();

			     w_scroll (&w, SCROLL_UP);
			     w_setscrollpos (&w, WINF_VBAR, 
					     anchor*maxdlines/maxlines);

			     w_gotoxy (&w, 1, w.ny-2);
			     w_puts (&w, ht [anchor+maxdlines-1]);

			     e_show ();
			 }
			 break;

	     case ESC  : finished = 1;
			 break;

	     case UP   : if (maxlines > maxdlines)
			 {
			     if (!anchor)
				 break;

			     anchor--;

			     e_hide ();

			     w_scroll (&w, SCROLL_DN);
			     w_setscrollpos (&w, WINF_VBAR, 
					     anchor*maxdlines/maxlines);
					   
			     w_gotoxy (&w, 1, 1);
			     w_puts (&w, ht [anchor]);

			     e_show ();
			 }
	  }
   }

   e_hide ();

   w_destroy (&w);

   e_show ();

   e_settrigger (oldtrigger);

   e_tresume (0);
   e_tresume (1);
}

int message (char *title)
{
   event e;
   int key, len = 0;
   static char buffer [TEMPLATE_COLS*TEMPLATE_ROWS*2];

   e_tsuspend (0);
   e_tsuspend (1);

   e_hide ();

   while (title [len++])
      ;

   v_screen (SCREEN_SAVE, template_x, template_y,
	     TEMPLATE_COLS, TEMPLATE_ROWS, buffer);

   v_gotoxy (template_x+1, template_y+1);
   v_putch (' ', TEMPLATE_COLS-2);
   v_gotoxy (template_x+1, template_y+TEMPLATE_ROWS-3);
   v_putch (' ', TEMPLATE_COLS-2);
   v_gotoxy (template_x+1, template_y+TEMPLATE_ROWS-2);
   v_putch (' ', TEMPLATE_COLS-2);

   v_setattr (attr [2] | BLINK);
   v_gotoxy (template_x+(TEMPLATE_COLS-len)/2+1, template_y+1);
   v_cputs (title);
   v_gotoxy (template_x+3, template_y+TEMPLATE_ROWS-2);
   v_cputs ("TRY AGAIN (Y/N)?");

   e_flush ();

   if (sound_enabled)
       key = play ((title [0] == 'S') ? funeral : stars);
   else
      while (1)
      {
	 do
	    e_fetch (&e);
	 while (e.type == MOUSE && (e.parm1 != ME_LBP));

	 if (e.type == MOUSE)
	     if (e.parm1 & ME_LBP)
	     {
		 key = 'Y';
		 break;
	     }

	 if (e.type == KEY)
	     if ((key = toupper (e.parm1)) == 'Y' || key == 'N')
		 break;
      }

   v_screen (SCREEN_RESTORE, template_x, template_y,
	     TEMPLATE_COLS, TEMPLATE_ROWS, buffer);

   e_show ();

   e_tresume (0);
   e_tresume (1);

   return key;
}

int m_range (event *e, int x1, int y1, int x2, int y2)
{
   unsigned x, y;

   x = e->parm3 >> 3;   /* Convert from pixel to character coordinates. */
   y = e->parm4 >> 3;

   x++;
   y++;                 /* Make coordinates relative to (1, 1). */

   return (x >= x1 && x <= x2 && y >= y1 && y <= y2);
}

void paint_squares (void)
{
   int i;

   e_hide ();

   v_setattr (attr [1]);

   for (i = 0; i < 9; i++)
	v_paint ((squares [i] ? '' : ' '), template_x+origins [i].x,
		 template_y+origins [i].y, CELL_SIZE_W, CELL_SIZE_H);

   e_show ();
}

int play (int *song)
{
   event e;
   int frequency, i = 0, key, rest;

   do
   {
      do
	 e_fetch (&e);
      while (e.type == MOUSE && (e.parm1 != ME_LBP));

      if (e.type == MOUSE)
	  if (e.parm1 & ME_LBP)
	  {
	      key = 'Y';
	      break;
	  }
	  else
	      continue;

      if (e.type == KEY)
	  if ((key = toupper (e.parm1)) == 'Y' || key == 'N')
	      break;
	  else
	      continue;

      if ((frequency = song [i++]) == -1)
	  break;

      sound (frequency);
      pause (rest = song [i++]);

      if (frequency != 32767 || rest != 1)
      {
	  sound (32767);
	  pause (1);
      }
   }
   while (1);

   nosound ();

   return key;
}

void randomize_squares (void)
{
   int finished = 0, i;

   do
   {
      for (i = 0; i < 9; i++)
	   squares [i] = random(2);

      for (i = 0; i < 9; i++)
	   if (squares [i])
	   {
	       finished = 1;
	       break;
	   }
   }
   while (!finished);
}

void reset (void)
{
   randomize_squares ();
   paint_squares ();
   if (tindex)
   {
       e_settimer (0, 1);
       e_settimer (1, 546*tindex);
   }
   e_flush ();
}

void update_timer (void)
{
   char buffer [5];
   unsigned mins, secs;

   mins = (secs = (unsigned) e_gettimer (1)*10/182)/60;
   secs = secs%60;
   buffer [0] = '0'+mins;
   buffer [1] = ':';
   buffer [2] = '0'+secs/10;
   buffer [3] = '0'+secs%10;
   buffer [4] = '\0';
   v_setattr (attr [5]);
   v_gotoxy (template_x+16, template_y+1);
   v_cputs (buffer);
}

#ifndef INCL_EVENT
#include "event.C"
#endif

#ifndef INCL_GETVECT
#include "getvect.C"
#endif

#ifndef INCL_ISDIGIT
#include "isdigit.C"
#endif

#ifndef INCL_KISKEY
#include "kiskey.C"
#endif

#ifndef INCL_KFETCH
#include "kfetch.C"
#endif

#ifndef INCL_RANDOM
#include "random.C"
#endif

#ifndef INCL_SETVECT
#include "setvect.C"
#endif

#ifndef INCL_STRLEN
#include "strlen.C"
#endif

#ifndef INCL_TOUPPER
#include "toupper.C"
#endif

#ifndef INCL_VCPUTS
#include "vcputs.C"
#endif

#ifndef INCL_VGETATTR
#include "vgetattr.C"
#endif

#ifndef INCL_VGETMODE
#include "vgetmode.C"
#endif

#ifndef INCL_VGETSHAPE
#include "vgetshap.C"
#endif

#ifndef INCL_VPAINT
#include "vpaint.C"
#endif

#ifndef INCL_VSCROLL
#include "vscroll.C"
#endif

#ifndef INCL_VSETATTR
#include "vsetattr.C"
#endif

#ifndef INCL_VSETSHAPE
#include "vsetshap.C"
#endif

#ifndef INCL_VSHADOW
#include "vshadow.C"
#endif

#ifndef INCL_VWHEREX
#include "vwherex.C"
#endif

#ifndef INCL_VWHEREY
#include "vwherey.C"
#endif

#ifndef INCL_WINDOW
#include "window.C"
#endif

int xparse (int option)
{
   int accum, digit;

   if (option == 'C')   /* COLORS OPTION */
   {
       if (!isdigit ((digit = parse ())))
       {
	   _AH = 9;
	   _DX = FP_OFF("squares: digit expected after 'C'\r\n$");
	   geninterrupt(0x21);
	   return -1;
       }

       accum = square_colors [digit-'0'];
       setrseg ();
       color [1] = accum;
       setcseg ();
       return 0;
   }

   if (option == 'R')   /* RESET OPTION */
   {
       accum = (80-TEMPLATE_COLS)/2;
       setrseg ();
       color [1] = RED;
       sound_enabled = 1;
       tindex = 0;
       template_x = accum;
       srow = -1;
       setcseg ();
       return 0;
   }

   if (option == 'S')   /* SOUND OPTION */
   {
       if (!isdigit ((digit = parse ())))
       {
	   _AH = 9;
	   _DX = FP_OFF("squares: digit expected after 'S'\r\n$");
	   geninterrupt(0x21);
	   return -1;
       }

       accum = digit-'0';
       setrseg ();
       sound_enabled = accum;
       setcseg ();
       return 0;
   }

   if (option == 'T')   /* TIMER OPTION */
   {
       if (!isdigit ((digit = parse ())))
       {
	   _AH = 9;
	   _DX = FP_OFF("squares: digit expected after 'T'\r\n$");
	   geninterrupt(0x21);
	   return -1;
       }

       accum = digit-'0';

       setrseg ();
       tindex = accum;
       setcseg ();
       return 0;
   }

   if (option == 'X')   /* COLUMN OPTION */
   {
       if (!isdigit ((digit = parse ())))
       {
	   _AH = 9;
	   _DX = FP_OFF("squares: digit expected after 'X'\r\n$");
	   geninterrupt(0x21);
	   return -1;
       }

       accum = digit-'0';

       if (!isdigit ((digit = parse ())))
	   unparse ();
       else
       {
	   accum *= 10;
	   accum += (digit-'0');
       }

       if (accum < 1 || accum > (80-TEMPLATE_COLS))
       {
	   _AH = 9;
	   _DX = FP_OFF("squares: column out of range ...\r\n$");
	   geninterrupt(0x21);
	   return -1;
       }

       setrseg ();
       template_x = accum;
       setcseg ();
       return 0;
   }

   if (option == 'Y')   /* ROW OPTION */
   {
       if (!isdigit((digit = parse ())))
       {
	   _AH = 9;
	   _DX = FP_OFF("squares: digit expected after 'Y'\r\n$");
	   geninterrupt(0x21);
	   return -1;
       }

       accum = digit-'0';

       if (!isdigit ((digit = parse ())))
	   unparse ();
       else
       {
	   accum *= 10;
	   accum += (digit-'0');
       }

       if (accum < 1 || accum > (50-TEMPLATE_ROWS))
       {
	   _AH = 9;
	   _DX = FP_OFF("squares: row out of range ...\r\n$");
	   geninterrupt(0x21);
	   return -1;
       }

       setrseg ();
       srow = accum;
       setcseg ();
       return 0;
   }

   return -1;
}
