/*
	ascii.C

	Version 1.6

	ASCII Chart

	Copyright (C) 1987-1995, Geoff Friesen 
	All rights reserved.

	Developed with: Borland C++ 3.1
*/

/*
	set ti=c:\borlandc\tsr\tsrlib
	t ascii -ha -x

	The BIOS memory location 0040:0084 is set to the number of
	screen rows less one.  For example, if the screen has been
	set to 25 lines then the value 24 would be stored here.  A
	value of 49 would be stored if 50-line mode were in effect
	(on VGA adaptors).  This address is used by recent adaptor
	BIOSes.  Older BIOSes do not use it.  Therefore, assume it
	to be valid if it returns a 24, 42, or 49.  The 42 is used
	in EGA 43-line mode.
*/

#if !defined(__TINY__)
#error Tiny Memory Model Expected
#endif

#include "tsrlib.H"

char *title = "\r\n"

"ͻ\r\n"
"                          \r\n"
"                                   \r\n"
"                                     \r\n"
"                    Version 1.6 \r\n"
"                                     \r\n"
"                                   \r\n"
"                         \r\n"
"                                           \r\n"
" Copyright (C) 1987-1995, Geoff Friesen    \r\n"
" All rights reserved.                      \r\n"
"ͼ\r\n"
"$";

int bw [] =
{
   WHITE,                               /* border */
   LIGHTGRAY << 4,                      /* lightbar */
   LIGHTGRAY                            /* normal text */
};

int color [] =
{
   YELLOW,                              /* border */
   LIGHTGRAY << 4,                      /* lightbar */
   (BLUE << 4) | WHITE                  /* normal text */
};

#define NATTR   (sizeof(bw)/sizeof(bw [0]))

int attr [NATTR];

#define NCOLS   18
#define NROWS   16

char buffer [(NCOLS+3)*(NROWS+3)*2];

int column = 1;
int row;
int srow = -1;                          /* suggested row */
int currow = 1;
int cchoice = 0;
int nchoice = 128;

char    *asc            (int row);
void    down            (int nlines);
void    end             (void);
void    home            (void);
void    refresh         (void);
void    up              (int nlines);

void main (void)
{
   int cshape, i, key, nrows, vmode, x, y;

   if ((vmode = v_getmode ()) != BW80 && vmode != C80 && vmode != MONO)
       return;

   m_savectx ();

   nrows = peekb(0x40, 0x84)+1;
   if (nrows != 25 && nrows != 43 && nrows != 50)
       nrows = 25;

   row = (srow == -1 || srow+NROWS+2 > nrows) ? 1 : srow;

   for (i = 0; i < NATTR; i++)
	attr [i] = (vmode == C80) ? color [i] : bw [i];

   cshape = v_getshape ();
   v_setshape (0x2000);         /* hide cursor */

   x = v_wherex ();
   y = v_wherey ();

   v_screen (SCREEN_SAVE, column, row, NCOLS+3, NROWS+3, buffer);

   v_setattr (attr [0]);
   v_border (BLOCK_LINE, column, row, NCOLS+2, NROWS+2);
   v_shadow (column, row, NCOLS+2, NROWS+2);

   refresh ();

   do
   {
      if ((key = k_fetch ()) == ESC)
	  break;

      switch (key)
      {
	 case DOWN : down (1);
		     break;

	 case END  : end ();
		     break;

	 case HOME : home ();
		     break;

	 case PGDN : down (16);
		     break;

	 case PGUP : up (16);
		     break;

	 case UP   : up (1);
      }
   }
   while (1);

   v_screen (SCREEN_RESTORE, column, row, NCOLS+3, NROWS+3, buffer);
   v_gotoxy (x, y);
   v_setshape (cshape);

   m_restorectx ();
}

char *asc (int row)
{
   int i, temp;
   static char buffer [NCOLS+1], hex [16] = "0123456789abcdef";

   for (i = 0; i < NCOLS; i++)
	buffer [i] = ' ';

   buffer [NCOLS] = '\0';

   temp = row;

   for (i = 2; i >= 0; i--)
   {
	buffer [i] = temp % 10 + '0';
	temp /= 10;

	if (!temp)
	    break;
   }

   buffer [4] = hex [(row & 0xf0) >> 4];
   buffer [5] = hex [row & 15];
   buffer [7] = (!row) ? ' ' : row;

   temp = row += 128;

   for (i = 12; i >= 10; i--)
   {
	buffer [i] = temp % 10 + '0';
	temp /= 10;
   }

   buffer [14] = hex [(row & 0xf0) >> 4];
   buffer [15] = hex [row & 15];
   buffer [17] = row;

   return buffer;
}

void down (int nlines)
{
   int i, dnlines;

   if (cchoice == nchoice-1)
       return;

   v_setattr (attr [2]);
   v_gotoxy (column+1, row+currow);
   v_cputs (asc (cchoice));

   dnlines = min(nchoice-1-cchoice, nlines);

   for (i = 1; i <= dnlines; i++)
   {
	if (currow != NROWS)
	    currow++;
	else
	    v_scroll (column+1, row+1, NCOLS, NROWS, SCROLL_UP,
		      1, attr [2]);

	if (i == dnlines)
	    v_setattr (attr [1]);

	v_gotoxy (column+1, row+currow);
	v_cputs (asc (++cchoice));
   }
}

void end (void)
{
   if (cchoice == nchoice-1)
       return;

   cchoice = nchoice-1;
   currow = (nchoice < NROWS) ? nchoice : NROWS;
   refresh ();
}

void home (void)
{
   if (!cchoice)
       return;

   cchoice = 0;
   currow = 1;
   refresh ();
}

void refresh (void)
{
   int index, _row;

   index = cchoice-currow+1;

   for (_row = 1; _row <= NROWS; _row++)
   {
	v_gotoxy (column+1, row+_row);
	v_setattr (attr [(currow == _row) ? 1 : 2]);
	v_cputs (asc (index++));
   }
}

void up (int nlines)
{
   int i, uplines;

   if (!cchoice)
       return;

   v_setattr (attr [2]);
   v_gotoxy (column+1, row+currow);
   v_cputs (asc (cchoice));
   uplines = min(cchoice, nlines);

   for (i = 1; i <= uplines; i++)
   {
	if (currow != 1)
	    currow--;
	else
	    v_scroll (column+1, row+1, NCOLS, NROWS, SCROLL_DN,
		      1, attr [2]);

	if (i == uplines)
	    v_setattr (attr [1]);

	v_gotoxy (column+1, row+currow);
	v_cputs (asc (--cchoice));
   }
}

#ifndef INCL_ISDIGIT
#include "isdigit.C"
#endif

#ifndef INCL_KFETCH
#include "kfetch.C"
#endif

#ifndef INCL_MOUSE
#include "mouse.C"
#endif

#ifndef INCL_VBORDER
#include "vborder.C"
#endif

#ifndef INCL_VCPUTS
#include "vcputs.C"
#endif

#ifndef INCL_VGETMODE
#include "vgetmode.C"
#endif

#ifndef INCL_VGETSHAPE
#include "vgetshap.C"
#endif

#ifndef INCL_VSCROLL
#include "vscroll.C"
#endif

#ifndef INCL_VSETATTR
#include "vsetattr.C"
#endif

#ifndef INCL_VSETSHAPE
#include "vsetshap.C"
#endif

#ifndef INCL_VSHADOW
#include "vshadow.C"
#endif

#ifndef INCL_VWHEREX
#include "vwherex.C"
#endif

#ifndef INCL_VWHEREY
#include "vwherey.C"
#endif

int xparse (int option)
{
   int accum, digit;

   if (option == 'R')   /* RESET OPTION */
   {
       setrseg ();
       column = 1;
       srow = -1;
       setcseg ();
       return 0;
   }

   if (option == 'X')   /* COLUMN OPTION */
   {
       if (!isdigit ((digit = parse ())))
       {
	   _AH = 9;
	   _DX = FP_OFF("ascii: digit expected after 'X'\r\n$");
	   geninterrupt(0x21);
	   return -1;
       }

       accum = digit-'0';

       if (!isdigit ((digit = parse ())))
	   unparse ();
       else
       {
	   accum *= 10;
	   accum += (digit-'0');
       }

       if (accum < 1 || accum > (80-NCOLS-2))
       {
	   _AH = 9;
	   _DX = FP_OFF("ascii: column out of range ...\r\n$");
	   geninterrupt(0x21);
	   return -1;
       }

       setrseg ();
       column = accum;
       setcseg ();
       return 0;
   }

   if (option == 'Y')   /* ROW OPTION */
   {
       if (!isdigit ((digit = parse ())))
       {
	   _AH = 9;
	   _DX = FP_OFF("ascii: digit expected after 'Y'\r\n$");
	   geninterrupt(0x21);
	   return -1;
       }

       accum = digit-'0';

       if (!isdigit ((digit = parse ())))
	   unparse ();
       else
       {
	   accum *= 10;
	   accum += (digit-'0');
       }

       if (accum < 1 || accum > (50-NROWS-2))
       {
	   _AH = 9;
	   _DX = FP_OFF("ascii: row out of range ...\r\n$");
	   geninterrupt(0x21);
	   return -1;
       }

       setrseg ();
       srow = accum;
       setcseg ();
       return 0;
   }

   return -1;
}
