//
// TreeDo.cpp -- walks directory tree, and makes in each.
//
// mikel@oz.net <Mike Lempriere>  Feb-93
// last mod 04-Apr-95

const char * Version = "ver 1.3, 05-Apr-95";

// Notes.
//
// When a DOS program ends, you must specifically CD back to the starting
// dir if you have changed dir while running, or the command shell will be
// sitting in that new dir.  This is why we set "InitialDir" at outset,
// reset it on exit, and trap signals, so that we can reset it even on a
// ctrl-C exit.

// This stupid thing is because Borland's BSEDOS.H is not
// properly CONST'ed, so we must cast away the CONST'ness.
#ifdef __BORLANDC__
#define BORCAST (PSZ)
#define IBMCAST
#else
// This stupid thing is because IBM's CSET++ chdir() is not
// properly CONST'ed, so we must cast away the CONST'ness.
#define BORCAST
#define IBMCAST (char *)
#endif

#include <stdio.h>
#include <stdlib.h>


#ifdef os2
#define INCL_DOS
#include "os2.h"
#else

#define DO_SIGNAL_STUFF
#include <dos.h>
// This sets the stack size for non-Windows apps in Borland.
#ifdef __BORLANDC__
unsigned _Cdecl _stklen = 8192;
#endif
#include <dir.h>
#endif

#include <string.h>
#include <signal.h>
#include <io.h>         // Needed for dup2()
#include <ctype.h>      // Needed for toupper()
#include <direct.h>     // Needed for _chdrive()

#include <fstream.h>

// Modes...
static int Debug = 0;
static int Verbose = 0;
static int DoTheDup = 0;
static int NoRecurse = 0;
static int LeavesOnly = 0;
static int SpecialMakeMode = 0;
static int IfExistMode = 0;
static int DoUsageOnly = 0;

// Misc...
const char * argv0 = NULL;
#define MAX_CMD 256
static char Command[MAX_CMD] = "";
static char InitialDir[256];
static char *MakeFname = NULL;
static char *MakePattern = NULL;
static int MakePatLen = 0;
static unsigned short TotalCnt = 0;
static int PathArgSubNum = 0;
static const char * PathArgSubStr = NULL;
static const char *SetDir = NULL;


static void Usage(void) {

    fprintf(stderr, "TreeDo %s - Traverses dir tree, performing command in each dir.\n", Version);
    fprintf(stderr, "by Mike Lempriere <mikel@oz.net>\n");
    fprintf(stderr, "\nUsage: [switches] command [args]\n");
    if (!Verbose) {
        fprintf(stderr, "[Use \"%s -help -verbose\" for a more detailed usage screen]\n\n",
                argv0);
        fprintf(stderr, "Switches:\n");
        fprintf(stderr, "-h, -?\tthis usage screen\t-v\tVerbose\n");
        fprintf(stderr, "-o\tdup Output\t\t-l\tLeaves only\n");
        fprintf(stderr, "-n\tNo recurse\t\t-s\tSet Initial dir\n");
        fprintf(stderr, "-m r\tMakefile mode, rule='r'\t-M f r\tMakefile name='f', rule='r'\n");
        fprintf(stderr, "-e\tfile Exists mode\t-p n s\tPath arg 'n' substitute with 's'\n");
        fprintf(stderr, "\nSpecial args:\n");
        fprintf(stderr, "%%d\tCurr. Working drv\t%%p\tCurr. Working dir\n");
        fprintf(stderr, "%%i\tInitial dir\t\t%%s\tStart dir\n");
        fprintf(stderr, "%%m\tMakefile name\t\t%%r\tmakefile Rule\n");
    }
    else {
        fprintf(stderr, "Switches:\n");
        fprintf(stderr, "-h, -?\tThis Usage screen.\n");
        fprintf(stderr, "-v\tverbose mode -- displays each dirname before performing command.\n");
        fprintf(stderr, "-o\tdup Output mode -- copies stderr to stdout.\n");
        fprintf(stderr, "-l\tLeaves only mode -- Only performs command in bottommost dirs.\n");
        fprintf(stderr, "-n\tno recurse -- does not go into subdirs.\n");
        fprintf(stderr, "-s path\tSet initial drive & dir.\n");
        fprintf(stderr, "-M f r\tSpecial check for 'make'.  This checks each dir for file 'f', and, if\n");
        fprintf(stderr, "\tit exists, checks for rule 'r' in that file.  If both conditions\n");
        fprintf(stderr, "\tare met, treedo performs the given command, else dir. will be skipped.\n");
        fprintf(stderr, "\tNote that 'rule' is assumed to be anchored left, and is a simple\n");
        fprintf(stderr, "\tchar-by-char match (no wildcards) -- not a Regular Expression!\n");
        fprintf(stderr, "\tSample: treedo -v -m makefile.dos lib make -f makefile.dos lib\n");
        fprintf(stderr, "-m r\tSame as \"-M\" but assumes file is \"Makefile\".\n");
        fprintf(stderr, "\tAlso, if no command is supplied, assumes command is 'make' with a 1st arg\n");
        fprintf(stderr, "\t of 'r'.  Sample:  \"treedo -m clean\" does a \"make clean\" in every dir.\n");
        fprintf(stderr, "\twith a \"makefile\" file containing a \"clean\" rule.\n");
        fprintf(stderr, "-e f\tCheck for file existance.  This checks each dir for file 'f', and, if\n");
        fprintf(stderr, "\tit exists, performs the given command, else dir. will be skipped.\n");
        fprintf(stderr, "-p n s\tSubstitutes Path segment 'n' with 's'.  Sample: in the C:\\Prima\\Src\n");
        fprintf(stderr, "\tdir:  treedo -n -p 1 Foo echo S:%%p  displays S:\\Foo\\Src\n");
        fprintf(stderr, "\nSpecial args:\n");
        fprintf(stderr, "%%d\tcurrent working drive\n");
        fprintf(stderr, "%%p\tcurrent working directory (without drive)\n");
        fprintf(stderr, "%%i\tinitial working directory\n");
        fprintf(stderr, "%%s\tstarting directory (1st parm after -s)\n");
        fprintf(stderr, "%%m\tmakefile name (1st parm after -m) (-m mode only)\n");
        fprintf(stderr, "%%r\tmakefile rule (2nd parm after -m) (-m mode only)\n");
    }
    fprintf(stderr, "\nAll unrecognized args will be passed to the new subshell as the command\n");
    fprintf(stderr, "to perform and its args.\n");
    exit(1);
}                                               // Usage()


// CD's somewhere.  As supplied paths might be relative to the initial dir,
// which may not be the root dir, we cheat by always CD'ing there first,
// then to the relative dir.  This gets us around having to parse out stuff
// like ".." which should work in the middle of a path, but doesn't.
static void ChgDir(const char *CurrDir, const char *Cp) {

    int DosErr = chdir(IBMCAST CurrDir);
    if (DosErr) {
        perror(CurrDir);
        exit(1);
    }
    DosErr = chdir(IBMCAST Cp);
    if (DosErr) {
        perror(Cp);
        exit(1);
    }
}                                               // ChgDir()


// Replaces the percent-d syntax in a string with the
// supplied drive letter and colon.
static void ReplaceDir(const char *CurrDir, char *Cp) {

    // Directly overwrites "%d" with "x:".
    *Cp = *CurrDir;
    *(Cp + 1) = ':';
}                                               // ReplaceDir()


// This just inserts CurrDir into Cp, overwriting the first 2 chars
// in Cp, assuming they're the "%p" that got us here.
// Further, if the first 2 chars of CurrDir are "x:",
// they are NOT copied, as we seperate the drive and path.
static void ReplacePath(const char *CurrDir, char *Cp) {

if (Debug) fprintf(stderr, "ReplacePath: At entry, CurrDir=%s, Cp=%s\n",
CurrDir, Cp);

    int Len = strlen(CurrDir);
    int DrvSkip = 0;
    if (CurrDir[1] == ':') {
        DrvSkip = 2;    // Skip drive and colon
    }
    Len -= DrvSkip;
    // Shift latter part of str right
    int Cntr = strlen(Cp);
    while (Cntr > 0) {
        // -2 because of the "%p" we're overwriting.
        *(Cp + Len + Cntr - 2) = *(Cp + Cntr);
        Cntr--;
    }
    // Copy in path
    strncpy(Cp, CurrDir + DrvSkip, Len);
if (Debug) fprintf(stderr, "ReplacePath: At exit, Cp=%s\n", Cp);
}                                               // ReplaceDir()


// Finds and replaces given dir seg.
static void ReplacePathSeg(char * OStr) {

    char * Cp = OStr;
    short Cnt = 0;
    // Find the n'th backslash...
    while ((Cp != NULL) && (Cnt < PathArgSubNum)) {
        Cp = strchr(Cp, '\\');
        Cnt++;
    }
    // If there are that many segments (ie. backslashes)...
    if (Cp != NULL) {
        if ((*PathArgSubStr) != '\\') {
            // Skip past the preceeding backslash...
            Cp++;
        }

        // Determine length of space we're going into...
        char * Np = strchr(Cp + 1, '\\');
        short Space = Np - Cp;

        // Find size we're moving...
        short NewLen = strlen(PathArgSubStr);

        // If we're putting a short string into a wider gap...
        if (NewLen < Space) {
            // Copy in new string...
            memcpy(Cp, PathArgSubStr, NewLen);
            // crunch left the tail of the string, wiping out crud.
            memcpy(Cp + NewLen, Np, strlen(Np) + 1);
        }
        // else we're fitting a long string into a narrower space...
        else {
            // Copy the tail of the string right, making room for new...
            memmove(Cp + NewLen, Cp + Space, strlen(Cp) + 1);
            // Copy in the new string.
            memcpy(Cp, PathArgSubStr, NewLen);
        }
    }
}						// ReplacePathSeg()


// Replaces any of our known parms with it's value in a string.
// %p = path, %d = drive, %m = make filename (-m only), %% = %.
static void ParmSubst(const char *CurrDir, char *Cmd) {

    strcpy(Cmd, Command);
    while (Cmd != NULL) {
        Cmd = strchr(Cmd, '%');
        if (Cmd != NULL) {
            char Ch = (*(Cmd + 1));
            switch (Ch) {
                case 'd':
                    ReplaceDir(CurrDir, Cmd);
                    break;
                case 'p':
                    ReplacePath(CurrDir, Cmd);
                    break;
                case '%':
                    strcpy(Cmd, Cmd + 1);
                    Cmd++;
                    break;
                case 'm':
                    if (MakeFname != NULL) {
                        ReplacePath(MakeFname, Cmd);
                    }
                    else {
                        Cmd++;
                    }
                    break;
                case 'r':
                    if (MakePattern != NULL) {
                        ReplacePath(MakePattern, Cmd);
                    }
                    else {
                        Cmd++;
                    }
                    break;
                case 's':
                    if (SetDir != NULL) {
                        ReplacePath(SetDir, Cmd);
                    }
                    else {
                        Cmd++;
                    }
                    break;
                case 'i':
                    if (InitialDir != NULL) {
                        ReplacePath(InitialDir, Cmd);
                    }
                    else {
                        Cmd++;
                    }
                    break;
                default:
                    fprintf(stderr,
                           "ignoring unrecognized special arg \"%%%c\"\n", Ch);
                    Cmd++;
                    break;
            }
        }
    }
}                                               // ParmSubst()


// Checks for existance of 'MakeFname', and 'MakePattern' within.
// Returns non-zero if both conditions are met, else 0.
// Note!  'MakePattern' checking assumes left-anchored-char-by-char match!
static int SpecialMakeCheck(void) {

#define MAX_LINE 256

    ifstream CheckFile(MakeFname);
    int Found = (CheckFile != 0);
    if (!Found) {
        return(Found);
    }
    Found = 0;
    char *Line = new char[MAX_LINE];
    while ((!Found) && CheckFile) {
        CheckFile.getline(Line, MAX_LINE);
        Found = (strncmp(Line, MakePattern, MakePatLen) == 0);
    }
    delete [] Line;
    return(Found);
}                                               // SpecialMakeCheck()


static void HeresOne(const char *CurrDir, const char *Dir) {

    int DoItFlag = 1;

    if (IfExistMode) {
        DoItFlag = SpecialMakeCheck();
    }
    if (DoItFlag) {
        char *Cmd = new char[MAX_CMD];
        char *Tmp = new char[MAX_LINE];
        strcpy(Tmp, CurrDir);
        if (Dir != NULL) {
            int len = strlen(Tmp);
            if ((len > 0) && ( * (Tmp + len - 1) != '\\')) {
                strcat(Tmp, "\\");
            }
            strcat(Tmp, Dir);
        }
        ParmSubst(Tmp, Cmd);
        if (PathArgSubNum > 0) {
            ReplacePathSeg(Cmd);
        }
        if (Verbose) {
            printf("%s->%s\n", Tmp, Cmd);
            fflush(stdout);
        }
        int DosErr = system(Cmd);
        if (DosErr == 255) {
            fprintf(stderr, "Shell returned %d; aborting treedo\n", DosErr);
            chdir(InitialDir);
            exit(1);
        }
        delete [] Cmd;
        delete [] Tmp;
    }
    TotalCnt++;
}                                               // HeresOne()


static void DoGetCWD(char *CurrDir, int SizeOfCurrDir) {

    char * cwd = getcwd(CurrDir, SizeOfCurrDir);
    if (cwd == NULL) {
        char zzz[80];
        sprintf(zzz, "getcwd err %d", errno);
        perror(zzz);
        exit(1);
    }
}                                               // DoGetCWD()


#ifdef os2
static long GetFirstFile(
    char ** Cp,
    FILEFINDBUF3 *FileInfo,
    HDIR *DirHandle,
    ULONG *SearchCount) {

    long DosErr = DosFindFirst(BORCAST *Cp, DirHandle,
                               MUST_HAVE_DIRECTORY,
                               FileInfo, sizeof(FILEFINDBUF3),
                               SearchCount, FIL_STANDARD);
    if (!DosErr) {
        *Cp = FileInfo->achName;
    }
    if (Debug > 4) {
        fprintf(stderr, "First: err=%d, name=%s\n", DosErr, *Cp);
    }
    return DosErr;
}                                               // GetFirstFile()

#else
static long GetFirstFile(char ** Cp,
                         struct find_t *FileInfo) {

    unsigned DosErr = _dos_findfirst(*Cp, FA_DIREC, FileInfo);
    if (!DosErr) {
        *Cp = FileInfo->name;
    }
    if (Debug > 4) {
        fprintf(stderr, "First: err=%d, name=%s\n", DosErr, *Cp);
    }
    return DosErr;
}                                               // GetFirstFile()
#endif


#ifdef os2
static long FindNextFile(
    char ** Cp,
    FILEFINDBUF3 *FileInfo,
    HDIR *DirHandle,
    ULONG *SearchCount) {

    long DosErr = DosFindNext(*DirHandle, FileInfo, sizeof(FILEFINDBUF3),
                              SearchCount);
    if (!DosErr) {
        *Cp = FileInfo->achName;
    }
    if (Debug > 4) {
        fprintf(stderr, "Next: err=%d, name=%s\n", DosErr, *Cp);
    }
    return DosErr;
}                                               // FindNextFile()

#else
// DOS borland version.
static long FindNextFile(char ** Cp,
                         struct find_t *FileInfo) {

    long DosErr = _dos_findnext(FileInfo);
    if (!DosErr) {
        *Cp = FileInfo->name;
    }
    if (Debug > 4) {
        fprintf(stderr, "Next: err=%d, name=%s\n", DosErr, *Cp);
    }
    return DosErr;
}                                               // FindNextFile()
#endif


static void MainLoop(void) {

    char * Cp = "*.*";
    char *CurrDir = new char[MAX_LINE];
    DoGetCWD(CurrDir, MAX_LINE);

    if (!NoRecurse) {
        int Leaf = 1;

#ifdef os2
        FILEFINDBUF3 FileInfo;
        HDIR DirHandle = HDIR_CREATE;
        ULONG SearchCount = 1L;
        long DosErr = GetFirstFile(&Cp, &FileInfo, &DirHandle, &SearchCount);
#else
        struct find_t FileInfo;
        long DosErr = GetFirstFile(&Cp, &FileInfo);
#endif
        do {
            if (Debug > 4) {
                puts(Cp);
            }
#ifdef os2
            if (!DosErr && (FileInfo.attrFile & FILE_DIRECTORY)) {
#else
            if (!DosErr && (FileInfo.attrib & FA_DIREC)) {
#endif
                if ((strcmp(Cp, ".") != 0) && (strcmp(Cp, "..")) != 0) {
                    Leaf = 0;
                    ChgDir(CurrDir, Cp);
                    if (!LeavesOnly) {
                        HeresOne(CurrDir, Cp);
                    }
                    MainLoop();
                }
            }
            if (DosErr) {
                perror(Cp);
            }
#ifndef os2
            DosErr = FindNextFile(&Cp, &FileInfo);
#else
            DosErr = FindNextFile(&Cp, &FileInfo, &DirHandle, &SearchCount);
#endif
        } while (!DosErr);
        if (LeavesOnly && (Leaf && (Cp != NULL))) {
            HeresOne(CurrDir, NULL);
        }
#ifdef os2
        DosFindClose(DirHandle);
#else
#endif
        DosErr = chdir(InitialDir);
    }
    else {
        HeresOne(CurrDir, NULL);
        chdir(InitialDir);
    }
    delete [] CurrDir;
}                                               // MainLoop()


#ifdef DO_SIGNAL_STUFF
static void cdecl Kill(int SigNum) {

    fprintf(stderr, "signal %d recv'd\n", SigNum);
    // We ignore errors here, just try to go there.
    chdir(InitialDir);
    if (InitialDir[1] == ':') {
        char Drv = (char) toupper(InitialDir[0]);
        _chdrive(Drv - 'A' + 1);
    }
    exit(1);
}                                               // KillSig()
#endif


// This returning to original dir is only necessary for DOS.
static void MainCleanup(void) {

    int Err = chdir(InitialDir);
    if (Err) {
        perror(InitialDir);
    }
    if (InitialDir[1] == ':') {
        char Drv = (char) toupper(InitialDir[0]);
        int Err = _chdrive(Drv - 'A' + 1);
        if (Err) {
            fprintf(stderr, "Failed to return to drive %c\n", Drv);
        }
    }
}                                               // MainCleanup()


static void ParseArgs(int argc, char * const *argv) {

    int NoMoreSwitches = 0;
    for (int argi = 1; argi < argc; argi++) {
        if ((!NoMoreSwitches) && ((*argv[argi]) == '-')) {
            char Ch = *(argv[argi] + 1);
            switch (Ch) {
                case 'v':
                    Verbose = 1;
                    break;
                case 'o':
                    DoTheDup = 1;
                    break;
                case 'n':
                    NoRecurse = 1;
                    break;
                case 'l':
                    LeavesOnly = 1;
                    break;
                case 'm':
                    IfExistMode = 2;
                    MakeFname = "Makefile";
                    SpecialMakeMode = 2;
                    break;
                case 'M':
                    IfExistMode = 1;
                    SpecialMakeMode = 1;
                    break;
                case 'e':
                    IfExistMode = 1;
                    break;
                case 's':
                    SetDir = argv[++argi];
                    break;
                case 'd':
                    Debug = (argv[argi][2] - '0');
                    if (Debug < 1) {
                        Debug = 1;
                    }
                    break;
                case 'p':
                    argi++;
                    PathArgSubNum = (argv[argi][0] - '0');
                    argi++;
                    PathArgSubStr = argv[argi];
                    break;
                case '?':
                case 'h':
                    DoUsageOnly = 1;
                    break;
                default:
                    fprintf(stderr,
                            "ignoring unrecognized switch \"-%c\"\n", Ch);
                    break;
            }
        }
        else {
            NoMoreSwitches = 1;
            switch (IfExistMode) {
            case 1:
                IfExistMode = 2;
                MakeFname = argv[argi];
                break;
            case 2:
                if (SpecialMakeMode) {
                    IfExistMode = 3;
                    MakePattern = argv[argi];
                    // Strictly an optimization, only need do this once.
                    MakePatLen = strlen(MakePattern);
                    if (SpecialMakeMode == 2) {
                        if (argi == (argc - 1)) {
                            strcpy(Command, "make ");
                            strcat(Command, MakePattern);
                        }
                    }
                    break;
                }
                // NOTE!  The else is intended to fall through to default.
            default:
                if (*Command) {
                    strcat(Command, " ");
                }
                strcat(Command, argv[argi]);
            }
        }
    }
}                                               // ParseArgs()


static int GotoStartDir(void) {

    int BadInit = 0;
    if (SetDir[1] == ':') {
        char Drv = (char) toupper(SetDir[0]);
        int Err = _chdrive(Drv - 'A' + 1);
        if (Err) {
            fprintf(stderr, "Can't change to drive %c\n", Drv);
            BadInit = 1;
        }
    }
    if (!BadInit) {
        const char *Cp = SetDir;
        if (SetDir[1] == ':') {
            Cp = (SetDir + 2);
        }
        if (strlen(Cp)) {
            int Err = chdir(IBMCAST Cp);
            if (Err) {
                fprintf(stderr, "Can't change to %s\n", Cp);
                BadInit = 1;
            }
        }
    }
    if (Debug) {
        fprintf(stderr, "Changing from %s to %s\n", InitialDir, SetDir);
    }
    return(BadInit);
}                                               // GotoStartDir()


static int MainSetup(void) {

    int BadInit = 0;
#ifdef DO_SIGNAL_STUFF
    if (signal(SIGINT, Kill)) {
        perror("signal SIGINT");
    }
#endif
    char *cwd = getcwd(InitialDir, sizeof(InitialDir));
    if (cwd == NULL) {
        char ErrMsg[80];
        sprintf(ErrMsg, "initial getcwd err %d", errno);
        perror(ErrMsg);
        BadInit = 1;
    }
    if (!BadInit) {
        if (Debug) {
            printf("debug=%d\n", Debug);
        }
        if (SetDir != NULL) {
            BadInit = GotoStartDir();
        }
    }
    return(BadInit);
}                                               // MainSetup()


int main(int argc, char * const *argv) {

    argv0 = argv[0];
    ParseArgs(argc, argv);
    if (DoUsageOnly || (!(*Command))) {
        Usage();
    }
    if (DoTheDup) {
        dup2(1, 2);             // This makes stderr follow stdout.
    }
    int BadInit = MainSetup();
    if (!BadInit) {
        MainLoop();
    }
    MainCleanup();
    if (Debug) {
        printf("Total %d\n", TotalCnt);
    }
    fflush(stdout);
    return(0);
}                                               // main() treedo
