unit LicenseGenerator;

{ ==============================================================
  LicenseGenerator 1.0.0
{ ==============================================================

  SUMMARY

  Standard form for a software supplier to generate trial and
  registrations codes for their customers.

  Author:       1997, Andy Schmidt
  Email:       Andy_Schmidt@CompuServe.com
  Compiler:    Delphi 2.01
  Runtime:     Win32


{ --------------------------------------------------------------

  USAGE

  1. Copy this unit to your "MyObjRep" private object repository.
     (Follow Delphi documentation on how to add a form to your
      object repository)

  2. Create a new project. Then choose File|New to display your
     object repository.

  3. Select the License Generator form and choose "inherit".


 { --------------------------------------------------------------

  CHANGE HISTORY

  1.0.0 24-Jan-97 (AS)  Initial Development

  -------------------------------------------------------------- }

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, StdCtrls, ClipBrd,
  LicenseCode;

type
  TLicenseGenerator = class(TForm)
    LUserName: TLabel;
    LFeatures: TLabel;
    LTrialCode: TLabel;
    LRegisteredUser: TLabel;
    BtnRecalculate: TButton;
    FUserName: TEdit;
    FTrialCode: TLabel;
    FMessage: TLabel;
    FRegisteredCode: TLabel;
    LBevel1: TBevel;
    LBevel2: TBevel;
    FOption1: TCheckBox;
    FOption2: TCheckBox;
    FOption3: TCheckBox;
    FOption4: TCheckBox;
    FOption5: TCheckBox;
    FOption6: TCheckBox;
    FOption7: TCheckBox;
    FOption8: TCheckBox;
    LTrialCode2: TLabel;
    BtnCopy: TButton;
    FRegisteredOptions: TLabel;
    procedure FormCreate(Sender: TObject);
    procedure BtnRecalculateClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure BtnCopyClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure Changed(Sender: TObject);
  private
    { Private declarations }
    function GetOptions: byte;
    procedure SetOptions(Value: byte);
  protected
    { Protected declarations: visible only to derived objects }
    ObjLicense: TLicenseCode;
  public
    { Public declarations }
    property Options: byte read GetOptions write SetOptions;
  end;

implementation

{$R *.DFM}


{ Procedures to create the License Object and to initialize the display.
  We are keeping these two seperate, so that user can choose to override
  the OnCreate procedure and effect the License Object before the screen
  is initialized }
procedure TLicenseGenerator.FormCreate(Sender: TObject);
begin
    { Create non-visual objects }
    ObjLicense := TLicenseCode.Create;      // create without registry access
end;

procedure TLicenseGenerator.FormShow(Sender: TObject);
begin
    { Initialize display }
    FUserName.Text := ObjLicense.UserName;  // get default user name
    BtnRecalculateClick(Sender);            // act as if button was pressed
end;

procedure TLicenseGenerator.FormClose(Sender: TObject; var Action: TCloseAction);
begin
    ObjLicense.Free;
end;


{ Calculate License Codes and display on screen }
procedure TLicenseGenerator.BtnRecalculateClick(Sender: TObject);
begin
    ObjLicense.Store( FUserName.Text, ObjLicense.TrialOptions, date + ObjLicense.TrialDays, '');
                       // store trial license information in properties
    with ObjLicense do
        begin
        { Refresh display with generated information }
        FUserName.Text := UserName;      // user name may have been defaulted
        FTrialCode.Caption := Code;      // generated trial code
        FMessage.Caption := LicenseStatus;
                           // and the most recent license status information
        end;
    FRegisteredOptions.Caption := IntToStr(self.Options);
    FRegisteredCode.Caption := ObjLicense.Calculate( FUserName.Text, self.Options, 0);
                                                     // display fully registered code
    BtnCopy.Enabled := True;
end;


{ Copy User and Codes to Clipboard for use in word processors or email }
procedure TLicenseGenerator.BtnCopyClick(Sender: TObject);
begin
    ClipBoard.AsText := 'User Name ..:' + #09 + FUserName.Text + #13#10 +
                        'Trial Use ..:' + #09'Options = ' + IntToStr(ObjLicense.TrialOptions) + ',  Code = '+ FTrialCode.Caption + #13#10 +
                        'Registration:' + #09'Options = ' + IntToStr(self.Options) + ',  Code = ' + FRegisteredCode.Caption + #13#10;
end;


{ Propery Access Methods }
function TLicenseGenerator.GetOptions: byte;
begin
    with ObjLicense do
        begin
        Result := integer(FOption1.Checked);
        Result := (Result shl 1) or integer(FOption2.Checked);
        Result := (Result shl 1) or integer(FOption3.Checked);
        Result := (Result shl 1) or integer(FOption4.Checked);
        Result := (Result shl 1) or integer(FOption5.Checked);
        Result := (Result shl 1) or integer(FOption6.Checked);
        Result := (Result shl 1) or integer(FOption7.Checked);
        Result := (Result shl 1) or integer(FOption8.Checked);
        end;
end;

procedure TLicenseGenerator.SetOptions(Value: byte);
begin
    FOption1.Checked := boolean(Value and $80);
    FOption2.Checked := boolean(Value and $40);
    FOption3.Checked := boolean(Value and $20);
    FOption4.Checked := boolean(Value and $10);
    FOption5.Checked := boolean(Value and $08);
    FOption6.Checked := boolean(Value and $04);
    FOption7.Checked := boolean(Value and $02);
    FOption8.Checked := boolean(Value and $01);
end;


{ Invalidate old data on screen }
procedure TLicenseGenerator.Changed(Sender: TObject);
begin
    BtnCopy.Enabled := False;
    FTrialCode.Caption := '';
    FRegisteredCode.Caption := '';
    FRegisteredOptions.Caption := '';
    FMessage.Caption := '';
end;

end.
