unit LicenseCode;

{ ==============================================================
  LicenseCode 1.0
{ ==============================================================

  SUMMARY

  Maintains and validates license information on disk.

  Author:       1997, Andy Schmidt
  Email:       Andy_Schmidt@CompuServe.com
  Compiler:    Delphi 2.01
  Runtime:     Win32


{ --------------------------------------------------------------


  USAGE

  1. add LicenseCode to your project

  2. include LicenseCode in the "USES" clause of your object

  3. declare an object variable to hold the object reference, e.g.

     var ObjLicense: TLicenseCode;

  4. anywhere inside your implementation, create the object, e.g.

     ObjLicense := TLicenseCode.CreateFromName( MyCompany, MyProduct, MyVersion );
     -- or --
     ObjLicense := TLicenseCode.Create;         // won't access disk for info

  5. access the properties and methods:

     Status := ObjLicense.Prompt(Always, false);

{ --------------------------------------------------------------


  PROPERTIES  (Read-only, unless indicated)

  Company
  Product
  Version          Are used to create a unique key in the registry
                   following Microsoft conventions:
                   HKLM\'Software'\Company\Product\Version\
                   (These properties are set by CreateFromName
                    constructor)

  UserName         Name of registered user.
                   (Default: name of computer owner, if available)

  Options          Registered features or product options
                   (Default: 0)

  ExpiryDate       End of trial period
                   (Default: current date plus Trialdays)

  Code             Registration code
                   Remark: This code is use to validate the UserName,
                   Options and ExpiryDate stored on disk. User cannot
                   temper with any of these without obtaining a new
                   code.

  TrialOptions     (R/W) Options used when calculating a trial code.
                   (Default: 255 = all options are available for trial)

  TrialDays        (R/W) Length of trial period. Used when calculating a
                   trial code.
                   (Default: 30 days)

  FirstTime        Set to true if the registry key has been created in
                   this session.
                   Remark: Use to determine if a product setup has been
                   performed.

  LicenseStatus    Message text set by certain methods to reflect current
                   license status, e.g. 'Trial expired in xxx days'.

  RegistryKey      Key to the license data in the registry

  DecodeTable      (R/W) Array of 32 unique letters and numbers used to
                   create the license code. Set your DecodeTable property
                   to make your license codes unique to others.


  METHODS

  Calculate        Returns a license code.
                   (Does not update any properties.)

  Validate         Calculates a full license code and a trial code
                   and compares it with the provided code.
                   The trial code allows for a ten day grace period
                   to allow for some delay between us generating a
                   code and the user receiving it.
                   (Sets the license status property)

  Store            Calculates or Validates a license code and stores
                   the license information on disk.
                   (Refreshes the read-only properties)

  Load             Retrieves and Validates the license information.
                   (Refreshes the read-only properites)

  Prompt           Displays a registration screen and prompts user to
                   provide or update the license information.
                   New license information is stored to disk.
                   Remark: Programmer may choose to display the prompt
                   unconditionally (e.g. at setup time), only during
                   a trial period ('nag screen'), or only when the
                   license has expired or is missing.
                   (Refreshes the read-only properites)


  EVENTS

  - none -


{ --------------------------------------------------------------

  LICENSE

  The Author hereby grants to you a nonexclusive license to use
  this software and the accompanying Instructions, only as
  authorized in this License.

  You agree that you will not assign, sublicense, transfer,
  pledge, lease, rent, or share your rights under this License
  in return for compensation of any kind. Before you use this
  software for commercial purposes, you are required to
  pay a license fee of $20.00 to the Author.

  You acknowledge and agree that the Software and the
  accompanying Instructions are intellectual property of
  the Author, protected under U.S. copyright law. You further
  acknowledge and agree that all right, title and interest in
  and to the Software, are and shall remain with the Author.
  This License does not convey to you an interest in or to the
  Software, but only a limited and revocable right of use.

  THIS SOFTWARE IS LICENSED "AS IS," AND LICENSOR DISCLAIMS ANY
  AND ALL WARRANTIES, WHETHER EXPRESS OR IMPLIED, INCLUDING,
  WITHOUT LIMITATION, ANY IMPLIED WARRANTIES OF MERCHANTABILITY
  OR FITNESS FOR A PARTICULAR PURPOSE.

  Author's cumulative liability to you or any other party for
  any loss or damages resulting from any claims, demands, or
  actions arising out of or relating to this License shall not
  exceed the license fee paid (if any) to Author for the use of
  the Software. In no event shall Author be liable for any
  indirect, incidental, consequential, special, or exemplary
  damages or lost profits, even if Author has been advised of
  the possibility of such damages.

  This software and accompanying instructions are provided with
  restricted rights. Use, duplication or disclosure by the
  Government is subject to restrictions as set forth in
  subparagraph (c)(1)(ii) of The Rights in Technical Data and
  Computer Software clause at DFARS 252.227-7013 or
  subparagraphs (c)(1) and (2) of the Commercial Computer
  Software - Restricted Rights 48 CFR 52.227-19, as applicable.

{ --------------------------------------------------------------

  CHANGE HISTORY

  1.0.0 22-Jan-97 (AS)  Initial Development
  1.0.1 30-Jan-97 (AS)  Made RegistryKey available as a R/O property
                  (AS)  Use AS_Registry object for easy access.
                  (AS)  Eliminate LicenseRead/Write as they were only
                        used in the Load / Store methods.

  -------------------------------------------------------------- }


interface

uses
  Forms, Registry, Windows, SysUtils,
  AS_Registry;

type
  TLicense_Status = (License_OK, License_Trial, License_Expired, License_NoCode, License_BadCode);
  TSignificantChar = '0'..'Z';
  TDecodeTable = array [0..31] of TSignificantChar;
  TLicense_PromptWhen = (Always, DuringTrial, NoLicense);

  // License Code (Main object)
  TLicenseCode = class(TObject)
  private
  { Private declarations: visible only in this unit }
    // Fields to store property values
    FCompany: string;
    FProduct: string;
    FVersion: string;
    FUserName: string;
    FOwnersName: string;
    FExpiryDate: TDateTime;
    FCode: string;
    FLicenseStatus: TLicense_Status;
    FRegistryKey: string;
    FOptions: byte;
    FTrialOptions: byte;
    FTrialDays: byte;
    FFirstTime: boolean;
    FDecodeTable:  TDecodeTable;
    // Other instance variables
    ORegistry: TAS_Registry;
    // Property access methods and internal methods
    function GetLicenseStatus: string;
    function FmtCode(const VCode: string): string;
    procedure LicensePrompt(const EnforceSetup: boolean);
  protected
  { Protected declarations: visible only to derived objects }
    function DataOpen(const VRegistryKey: string; const Create: boolean): Tasr_OpenResult; virtual;
    procedure DataClose; virtual;
    function DataRead(const ValueName: string; const Value: variant): variant; virtual;
    procedure DataWrite(const ValueName: string; const Value: variant); virtual;
  public
  { Public declarations: visible only at run-time }
    property Company: string read FCompany;
    property Product: string read FProduct;
    property Version: string read FVersion;
    property UserName: string read FUserName;
    property Options: byte read FOptions;
    property ExpiryDate: TDateTime read FExpiryDate;
    property Code: string read FCode;
    property LicenseStatus: string read GetLicenseStatus;
    property RegistryKey: string read FRegistryKey;
    property FirstTime: boolean read FFirstTime;
    property TrialOptions: byte read FTrialOptions write FTrialOptions default High(byte);
    property TrialDays: byte read FTrialDays write FTrialDays default 30;
    property DecodeTable: TDecodeTable write FDecodeTable;
    constructor Create; virtual;
    constructor CreateFromName( const VCompany: string; const VProduct: string; const VVersion: string );
    destructor Destroy; override;
    function Calculate( const VUserName: string; const VOptions: byte; const VExpiryDate: TDateTime ): string;
    function Store( const VUserName: string; const VOptions: byte; const VExpiryDate: TDateTime; const VCode: string ): TLicense_Status;
    function Validate( const VUserName: string; const VOptions: byte; const VExpiryDate: TDateTime; const VCode: string ): TLicense_Status;
    function Load: TLicense_Status;
    function Prompt( const PromptWhen: TLicense_PromptWhen; const EnforceSetup: boolean ): TLicense_Status;

  end;

implementation

uses LicenseInput;

const
    { Tables to for license code generation}
    XLEncodeChars: array[TSignificantChar] of
                        byte = (1, 2, 3, 4, 5, 6, 7, 8, 9, 10,             {'0'..'9'}
                                0, 0, 0, 0, 0, 0, 0,                       {:;<=>?@}
                                11, 12, 13, 14, 15, 16, 17, 18, 19, 20,    {'A'..'J'}
                                21, 22, 23, 24, 25, 26, 27, 28, 29, 30,    {'K'..'T'}
                                31, 32, 33, 34, 35, 36);                   {'U'..'Z'}
    XLValidChars: set of '0'..'z' = ['0'..'9', 'A'..'Z', 'a'..'z'];
    XLDecodeTable: TDecodeTable  // ambiguous characters removed, e.g. B/8, O/0, S/5
                             = ('A', '1', 'E', 'R', 'J', '5', 'N', 'W',
                                'B', '2', 'F', 'T', 'K', '6', 'O', 'X',
                                'C', '3', 'G', 'U', 'L', '7', 'P', 'Y',
                                'D', '4', 'H', 'V', 'M', '9', 'Q', 'Z');

    { Location of Operating System user }
    DefaultNTOwner = '\SOFTWARE\Microsoft\Windows NT\CurrentVersion';
    DefaultOwner = '\Software\Microsoft\Windows\CurrentVersion';

    { Default Usernames }
    UserNotRegistered = '** Not Registered **';
    UserIncorrectCode = '** Not Licensed **';

    { License Status Messages }
    MsgLicense_Status: array [TLicense_Status] of string =
          ( 'Registered User',
            'Trial expires in %.1d days.',
            'Software trial has expired.',
            'No license code found.',
            'License information invalidated.' );

    GracePeriod = 10;

{ Initialize basic object properties }
constructor TLicenseCode.Create;
begin
    { Default to a trial user }
    FTrialOptions := High(FTrialOptions);
    FTrialDays := 30;
    FDecodeTable := XLDecodeTable;
    FUserName := UserNotRegistered;
    FExpiryDate := Date + FTrialDays;
end;

constructor TLicenseCode.CreateFromName(const VCompany: string; const VProduct: string; const VVersion: string);
begin
    self.Create;                              // call the default constructor

    { Try to determine the owner of this computer }
    if OK in DataOpen(DefaultOwner, False) then     // find windows owner, if any
        FOwnersName := DataRead('RegisteredOwner', FOwnersName);
    if OK in DataOpen(DefaultNTOwner, False) then   // find NT owner, if any
        FOwnersName := DataRead('RegisteredOwner', FOwnersName);

    { Construct the registry key then create or open it }
    FCompany := VCompany;
    FProduct := VProduct;
    FVersion := VVersion;
    FRegistryKey := '\Software';
    if FCompany <> '' then
        FRegistryKey := FRegistryKey + '\' + FCompany;
    if FProduct <> '' then
        FRegistryKey := FRegistryKey + '\' + FProduct;
    if FVersion <> '' then
        FRegistryKey := FRegistryKey + '\' + FVersion;
    if Created in DataOpen(FRegistryKey, True)
        then FFirstTime := True;

    { Extract existing properties from disk, if any }
    Load;
end;

destructor TLicenseCode.Destroy;
begin
    if ORegistry <> nil then
        begin
        DataClose;              // Current key to disk, or remove empty key
        ORegistry.Free;         // Free the registry object
        end;
    inherited destroy;          // Call the inherited destructor last
end;


{ Property access methods }
function TLicenseCode.GetLicenseStatus: string;
begin
    result := Format( MsgLicense_Status[FLicenseStatus], [round(FExpiryDate - date)] );
end;


{ I/O routines - can be overriden to store data elsewhere }
function TLicenseCode.DataOpen(const VRegistryKey: string; const Create: boolean): Tasr_OpenResult;
begin
    if ORegistry = nil then                 // In case we are re-opening with different key
        ORegistry := TAS_Registry.Create    // This object will use the registry to store license data
    else
        ORegistry.CloseKey;                 // Close any open key
    result := ORegistry.OpenKeyVerbose(HKEY_LOCAL_MACHINE, VRegistryKey, Create);
end;

procedure TLicenseCode.DataClose;
begin
    { Check if this is first time use }
    if FFirstTime then
        { remove any empty key we created }
        ORegistry.CleanUpKey(FRegistryKey);
    ORegistry.CloseKey;
end;

function TLicenseCode.DataRead(const ValueName: string; const Value: variant): variant;
begin
    Result := ORegistry.ReadValue(ValueName, Value);
end;

procedure TLicenseCode.DataWrite(const ValueName: string; const Value: variant);
begin
    ORegistry.WriteValue(ValueName, Value);
end;


{ Display the Registration Form }
procedure TLicenseCode.LicensePrompt(const EnforceSetup: boolean);

var
    WLicenseInput: TLicenseInput;

begin
    WLicenseInput := TLicenseInput.Create(Application);
    WLicenseInput.ObjLicense := self;
    WLicenseInput.EnforceSetup:= EnforceSetup;
    WLicenseInput.ShowModal;           // stops the application until form closed
    WLicenseInput.Release;
end;


{ Prompt for Registration, upon condition }
function TLicenseCode.Prompt(const PromptWhen: TLicense_PromptWhen; const EnforceSetup: boolean): TLicense_Status;
begin
    if FirstTime or (PromptWhen = Always) then
        begin
        LicensePrompt(EnforceSetup);
        result := Load;                   // refresh status after prompt
        exit;
        end;

    result := Load;                       // get current status from disk
    if result = License_OK then exit;     // good license, done.
    if (result = License_Trial) and (PromptWhen <> DuringTrial) then exit;

    { License is not OK, or caller wants prompts for every trial }
    LicensePrompt(EnforceSetup);
    result := Load;                       // refresh status after prompt
end;


{ Load the license code from disk and check for integrity }
function TLicenseCode.Load: TLicense_Status;
begin
    { if registry object exists, load properties from disk }
    if ORegistry <> nil then
        begin
        FUserName := DataRead('UserName', FUserName);         // default to computer owner
            if FUserName = '' then FUserName := FOwnersName;
        FExpiryDate := DataRead('ExpiryDate', FExpiryDate);   // default to trial period
            if FExpiryDate = 0 then FExpiryDate := Date + FTrialDays;
        FOptions := DataRead('LicenseOptions', FOptions);
        FCode := FmtCode(DataRead('LicenseCode', FCode));
        end;
    { check loaded code for integrity }
    result := Validate ( FUserName, FOptions, FExpiryDate, FCode );
    if (result = License_BadCode) then        // if this code does not check out
        FUserName := UserIncorrectCode;       // indicate tempering
end;


{ Store a new license code and all associated information }
function TLicenseCode.Store( const VUserName: string; const VOptions: byte; const VExpiryDate: TDateTime; const VCode: string ): TLicense_Status;
begin
    FExpiryDate := VExpiryDate;
    FOptions := VOptions;

    { If no user specified, create test password for test user }
    FUserName := VUserName;
    if FUserName = '' then
       begin
       FUserName := UserNotRegistered;
       FExpiryDate := Date + FTrialDays;
       FOptions := FTrialOptions;
       end;

    { Generate code as necessary }
    FCode := FmtCode(VCode);
    if FCode = '' then
        { attempt to calculate license code }
        FCode := Calculate( FUserName, FOptions, FExpiryDate );

    { Now determine license status, and save Full or Trial License code to disk }
    result := Validate ( FUserName, FOptions, FExpiryDate, FCode );
    if ( (result = License_OK) or (result = License_Trial) )
     and (ORegistry <> nil) then
        begin
        DataWrite('UserName', FUserName);
        DataWrite('ExpiryDate', FExpiryDate);
        DataWrite('LicenseCode', FCode);
        DataWrite('LicenseOptions', FOptions);
        end;
end;


{ Validate a license code }
function TLicenseCode.Validate( const VUserName: string; const VOptions: byte; const VExpiryDate: TDateTime; const VCode: string ): TLicense_Status;

var
    TempCode: string;

begin
    TempCode := FmtCode(VCode);                  // format the license code
    if TempCode <> '' then
        // let's see if the user is fully licensed (ignore the date)
        if TempCode <> Calculate( VUserName, VOptions, 0 ) then
            begin
            // can we match up with the expiry date for a trial code ?
            if (TempCode <> Calculate ( VUserName, FTrialOptions, VExpiryDate )) and
                // allow for up to 10 days between us generating the code, and the
                // user entering it!
               (TempCode <> Calculate ( VUserName, FTrialOptions, VExpiryDate - GracePeriod )) then
                // UserID, options or date were manipulated. Bad Boy!
                    FLicenseStatus := License_BadCode
            else
                // Code matches trial date, let's see if we're still within trial period,
                // and that expiry date is not outside the assigned trial period from today.
                if (Date <= VExpiryDate) and ((Date + FTrialDays) >= ExpiryDate) then
                    FLicenseStatus := License_Trial
                else
                    FLicenseStatus := License_Expired
            end
        else
            FLicenseStatus := License_OK
    else
        FLicenseStatus := License_NoCode;
    result := FLicenseStatus;
end;


{ Calculate a new license code }
function TLicenseCode.Calculate( const VUserName: string; const VOptions: byte; const VExpiryDate: TDateTime ): string;

type
    TEncodedHex = packed record
        Date: word;
        Options: byte;
        UserName: longint;
        end;

const
    ScrambledLength = Trunc( SizeOf(TEncodedHex) * 8 / 5 ) + 1;

var
    ExpiryDay, ExpiryMonth, ExpiryYear: word;
    TempUserName: string;
    TempChar: char;
    BitsPerChar: single;
    ShiftFactor, HaveShifted: byte;
    i, j, k: byte;
    XLUserName: longint;
    TempCode: string;

    { This stores our encoded fields in hex }
    EncodedHex: TEncodedHex;
    EncodedByte: packed array [1..SizeOf(EncodedHex)] of byte absolute EncodedHex;
    ScrambledByte: array [1..ScrambledLength] of byte;

begin
    { We require a user name }
    if VUserName = '' then
        begin
        result := '';
        exit;
        end;

    { Extract Month and Year, we're not concerned with the day }
    DecodeDate( Int(VExpiryDate), ExpiryYear, ExpiryMonth, ExpiryDay );
    EncodedHex.Date := (ExpiryMonth * 100) + ExpiryYear;

    { Create an uppercase string with only the significant letters and numbers }
    TempUserName := '';                                       // initialize local variable
    for i := 1 to length(VUserName) do
        begin
        TempChar := VUserName[i];
        if TempChar in XLValidChars then                      // restrict to letters and numbers
            TempUserName := TempUserName + TempChar;
        end;
    TempUserName := UpperCase(TempUserName);

    { Encode the string into an integer }
    BitsPerChar := SizeOf(XLUserName) * 8 / length(TempUserName);
    XLUserName := 0;
    HaveShifted := 0;
    for i := 1 to length(TempUserName) do
        begin
        ShiftFactor := Trunc( (BitsPerChar * i) - HaveShifted );
        XLUserName := XLUserName shl ShiftFactor;             // shift to accomodate characters
        XLUserName := XLUserName + XLEncodeChars[ TempUserName[i] ];
        Inc(HaveShifted, ShiftFactor);                        // keep track how much we shifted
        end; {for}
    EncodedHex.UserName := XLUserName;                        // combine the encoded values with date
    EncodedHex.Options := VOptions;

    { Now scatter each source byte across all destination bytes }
    fillchar( ScrambledByte, ScrambledLength, 0 );   // clear local variable
    j := 1;
    for k := 1 to Trunc( SizeOf(EncodedHex) * 8 / ScrambledLength ) + 1 do
                                                     // one pass for each bit in source
        for i := 1 to ScrambledLength do             // one pass for each byte in target
            begin
            ScrambledByte[i] := ((ScrambledByte[i]) shl 1) or ((EncodedByte[j]) and $01);
                             // shift target left to make space, then copy rightmost source bit
            EncodedByte[j] := (EncodedByte[j]) shr 1; // eliminate this source bit
            if j < SizeOf(EncodedHex) then inc(j)    // take same bit in next source byte
            else j := 1;                             // start over with next bit first source byte
            end; {for i}

    { Translate scrambled bytes into letters and numbers }
    TempCode := '';                                  // empty destination string
    for k := 1 to ScrambledLength do                 // one pass for each byte in scrambled code
        begin
        TempChar := FDecodeTable[ (ScrambledByte[k]) ];
        TempCode := TempCode + TempChar;             // translate into non-umbiguous characters
        end;

    result := FmtCode(TempCode);
end;


{ Format the license code - remove unwanted characters and insert seperators }
function TLicenseCode.FmtCode(const VCode: string): string;

var
    i, j: byte;

begin
    result := ''; 
    j := 0;
    for i := 1 to length(VCode) do
        begin                           // check for numbers and letters first
        if VCode[i] in XLValidChars then
            begin                       // convert to uppercase and copy to target
            if j >= 4 then               // count every four target strings
                begin
                j := 1;
                result := result + '-'; // insert a seperator in target
                end {if}
            else
                inc(j);
            result := result + UpCase(VCode[i]);
            end; {if}
        end; {do}
end;


end.
