/* $Revision: 1.12 $ */
/*
 EPSHeader

   File: log.c
   Author: J. Kercheval
   Created: Sat, 06/08/1991  09:24:14
*/
/*
 EPSRevision History

   J. Kercheval  Sat, 06/08/1991  09:24:27  move from fileio.c
   J. Kercheval  Wed, 06/26/1991  21:30:57  use standard fileio
   J. Kercheval  Fri, 07/12/1991  23:09:16  use stderr as default
   J. Kercheval  Sun, 09/03/1995 22:10:21   Add hooks for windows output
   J. Kercheval  Mon, 09/04/1995 15:37:01   Add flags output routine
*/

/* disable inline compiler warning */
#pragma warning(disable:4001 4201 4209 4214 4711)

#if defined(WIN16) || defined(WIN32)
#include <windows.h>
#include <windowsx.h>
#include "resource.h"
#endif

#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "flags.h"
#include "log.h"


static BOOLEAN log_is_open = FALSE;     /* if true then file has been opened */
static BOOLEAN log_verbose = TRUE;		/* if true log to status */
static BOOLEAN logging_error = FALSE;	/* if true, now logging an error message */

static FILE *log_file;          /* log_file is the output log file pointer */


/* external_cleanup() is used on exit(1) for internal error */
extern void external_cleanup(void);

#if defined(WIN16) || defined(WIN32)
extern HWND hTagsDlg;
#endif

/*----------------------------------------------------------------------------
 *
 * output helper macros
 *
 ---------------------------------------------------------------------------*/

#define logfile_output(message) \
{ \
    if (log_is_open) { \
        tfprintf(log_file, "%s\n", message); \
	} \
}

#define console_output(message) \
{ \
	logfile_output(message); \
	if (log_verbose) { \
		fprintf(stderr, "%s\n", message); \
	} \
}


/*----------------------------------------------------------------------------
 *
 * tpwrite(), tfprintf() and tfclose checks the output stream for
 * output errors caused by disk space full conditions.
 *
 ---------------------------------------------------------------------------*/

void __cdecl tfwrite(const void *buf, size_t size, size_t count, FILE *fp)
{
    if (!fwrite(buf, size, count, fp)) {
        log_error("# tfwrite() -- error writing file, disk may be full");
        external_cleanup();
        exit(1);
    }
}

void __cdecl tfprintf(FILE *fp, char *fmt, ...)
{
    va_list argptr;

    va_start(argptr, fmt);
    if (vfprintf(fp, fmt, argptr) == EOF) {
		if (!logging_error) {
			log_error("# tfprintf() -- error writing file, disk may be full");
			external_cleanup();
			exit(1);
		}
    }
}

void __cdecl tfclose(FILE *fp)
{
	if (fclose(fp) == EOF) {
       log_error("# fclose() -- error closing file, disk may be full");
       external_cleanup();
       exit(1);
	}
}


/*----------------------------------------------------------------------------
*
*  open the log file after closing any existing log file
*
----------------------------------------------------------------------------*/

BOOLEAN log_open(char *fname, BOOLEAN quiet, BOOLEAN overwrite)
{
    char open_type[3];          /* used to determine the open file type */

    /* assign the file open type */
    if (overwrite) {
        strcpy(open_type, "w");
    }
    else {
        strcpy(open_type, "a");
    }

    /* close the log if already open */
    if (log_is_open)
        log_close();

    log_verbose = !quiet;

    /* open file if non-zero file name */
    if (strlen(fname)) {
        /* try to open the log file */
        if ((log_file = fopen(fname, open_type)) == (FILE *) NULL) {
            return FALSE;
        }
        else {

            /* ah, sweet success */
            log_is_open = TRUE;
            return TRUE;
        }
    }
    else {

        /* no log file */
        log_is_open = FALSE;
        return TRUE;
    }
}


/*----------------------------------------------------------------------------
*
*  close the log file, if open
*
----------------------------------------------------------------------------*/

void log_close()
{
    if (log_is_open) {
        fclose(log_file);
        log_is_open = FALSE;
    }
}


/*----------------------------------------------------------------------------
*
*  log status message
*
----------------------------------------------------------------------------*/

void log_message(char *message)
{
#if defined(WIN16) || defined(WIN32)
	logfile_output(message);
	Static_SetText(GetDlgItem(hTagsDlg, IDC_STATUS), message);
#else
	console_output(message);
#endif
}


/*----------------------------------------------------------------------------
*
*  log error message
*
----------------------------------------------------------------------------*/

void log_error(char *message)
{
	logging_error = TRUE;

#if defined(WIN16) || defined(WIN32)
	logfile_output(message);
	MessageBox(hTagsDlg, message, "Tags Error", MB_OK | MB_ICONHAND);
#else
	console_output(message);
#endif

	logging_error = FALSE;
}


/*----------------------------------------------------------------------------
*
*  log information message
*
----------------------------------------------------------------------------*/

void log_info(char *message)
{

#if defined(WIN16) || defined(WIN32)
	logfile_output(message);
	ListBox_AddString(GetDlgItem(hTagsDlg, IDC_LIST), message);
#else
	console_output(message);
#endif
}


/*----------------------------------------------------------------------------
 *
 * log_flags logs flag information as information
 *
 ---------------------------------------------------------------------------*/

void log_flags(Flags * flags)
{
	char tmpstr[2 * MAXPATH];

	/* output format in use */
	if (flags->oe) {
		log_info("# Output Format:\tEpsilon (-oe)");
	}
	if (flags->o5) {
		log_info("# Output Format:\tEpsilon <= V5.0 (-o5)");
	}
	if (flags->og) {
		log_info("# Output Format:\tGNU etags (-og)");
	}
	if (flags->os) {
		log_info("# Output Format:\tSpace delimited (-os)");
	}
	if (flags->om) {
		log_info("# Output Format:\tMicrosoft error (-om)");
	}
	if (flags->oc) {
		log_info("# Output Format:\tCodewright (-oc)");
	}

	/* output output destinations */
	if (flags->convert_file) {
		sprintf(tmpstr, "# Compiled File:\t%s", flags->compile_filename);
		log_info(tmpstr);
	}
	if (flags->output_file) {
		sprintf(tmpstr, "# Output File:  \t%s", flags->output_filename);
		log_info(tmpstr);
	}
	if (log_is_open) {
		sprintf(tmpstr, "# Log File:     \t%s", flags->log_filename);
		log_info(tmpstr);
	}
	
	/* output sort option */
	if (flags->sort_tags) {
		if (flags->case_sensitive) {
			log_info("# Sort Method:  \tCase Sensitive");
		}
		else {
			log_info("# Sort Method:  \tCase Insensitive");
		}
	} 
	else {
		log_info("# Sort Method:  \tNone");
	}
	
	/* output filter option */
	if (flags->filter_junk) {
		log_info("# Output Filter:\tJunk Filter");
	} 
	else {
		log_info("# Output Filter:\tNone");
	}
	
	/* output path format option */
	if (flags->use_relative_pathnames) {
		log_info("# Path Output:  \tRelative");
	}
	else {
		log_info("# Path Output:  \tAbsolute");
	}
	
	/* output miscellaneous flags (shown only if non default) */
	if (flags->delete_response_files ||
		!flags->log_overwrite ||
		!flags->output_overwrite) {
		log_info("#");
		if (flags->delete_response_files) {
			log_info("# Deleting listfiles and exclude response files");
		}
		if (!flags->log_overwrite) {
			log_info("# Logfile output will be appended");
		}
		if (!flags->output_overwrite) {
			log_info("# Output file will be merged");
		}
	}
}


/*----------------------------------------------------------------------------
*
*  log stdout message
*
----------------------------------------------------------------------------*/

void log_stdout(char *message)
{
#if defined(WIN16) || defined(WIN32)
	log_info(message);
#else
	fprintf(stdout, "%s\n", message);
#endif
}


