unit zc32load;
{$A+,B-,D+,F+,G+,I+,K+,L+,N+,P-,Q-,R-,S-,T-,V-,W-,X+,Y+}
{$M 8192,1024}
interface

{ Written for BPW 7.0 }

{ This contains an object declartion (Tcode32man) that can be used to     }
{ load and execute proper 32 bit code (i.e. assembler code compiled with  }
{ the USE32 directive.                                                    }

{ Real 32 bit code (i.e. no code prefixes that empty the pipeline) can only }
{ be called 16 bit code over a call gate.                                   }

{ This code is hereby placed in the public domain.                            }
{ I am not responsible for any errors or damage that the program              }
{ may cause.                                                                  }
{ Author: Mike Wise, 20 July 1994, Luebeck, Germany                           }
{         CompuServe 100014,170                                               }
{                                                                             }
{                                                                             }
{ Possible Improvements:                                                      }
{    - We should test the winMem32 ver to make sure that it is not too old,   }
{      but I'm not sure what exactly too old is.                              }
{    - We could expand this to handle multiple callGates into a single code   }
{      segment, but then we would have to deal with the linking problem. This }
{      way the offset in makeCallGate is always zero.                         }
{    - We could write a real loader, that took OBJ files and linked them in.  }
{      I suspect this would be a lot of work, but then we wouldn't have to    }
{      TLINK and EXE2BIN anymore. We also could use multiple callGates easily.}
{                                                                             }
{ Problem:                                                                    }
{     I couldn't get the version of TDW that works with BP7 to understand     }
{     about 32-bit code segments. So if your going to develop code with this  }
{     thing you should probably get something like SoftIce. Ideally you       }
{     want a debugger that understands both BPW debugging info and 32-bit     }
{     code segments. If anybody has any ideas here please let me know         }


type Tcode32man=object selector32:word;
                       dataPtr:pointer;
                       codeAlias:word;
                       callGatePtr:pointer;
                       dataMarker:^byte;
                       allocErr:word;
                       winMem32ver:word;
                       constructor init;
                       procedure allocMem( size:longint );
                       function loadFile( fname:string ):integer;
                       function makeCallGate( offset:word; ring:byte; nParam:word):pointer;
                       procedure code1( b1:byte );
                       procedure code2( b1,b2:byte );
                       destructor done;
                       end;
     Pcode32man=^Tcode32man;

{ Test routines }
procedure test32_1;
procedure test32_2;{ This one needs the files s32.asm }

implementation

uses WinMem32,zldt,winprocs,windos,hexunit;

constructor Tcode32man.init;
begin
     selector32 := 0;
     dataPtr := nil;
     codeAlias := 0;
     callGatePtr := nil;
     dataMarker := nil;
end;

procedure Tcode32man.code1( b1:byte );
{ Code in 1 more byte }
begin
     dataMarker^ := b1; inc( dataMarker );
end;
procedure Tcode32man.code2( b1,b2:byte );
{ Code in 2 more byte }
begin
     dataMarker^ := b1; inc( dataMarker );
     dataMarker^ := b2; inc( dataMarker );
end;

procedure Tcode32man.allocMem( size:longint );
{ Allocate a chunk of 32 bit memory, along with 16-bit data and 32-bit code aliase selectors }
var maxsize:longint;
begin
     { Get the WinMem32 version number just in case we can think of anything to do with it }
        allocErr := $ffff;
        winMem32ver := GetWinMem32Version;

     { Allocate a chunk of of 32-bit memory  }
         maxsize := size;
         allocErr := Global32Alloc( size, @selector32, maxsize, 0 );
         if allocErr<>0 then exit;

     { Get a 16:16 aliased pointer on to it }
         dataPtr := nil;
         allocErr := Global16PointerAlloc( selector32, 0, @dataPtr, size, 0 );
         if allocErr<>0 then exit;
         fillchar( dataPtr^, size, 0 );{ zero it out, don't have to do this but it helps with debugging }
         dataMarker := dataPtr;

     { Get a 32-bit opcode code selector on the same memory }
         allocErr := Global32CodeAlias( selector32, @codeAlias, 0 );
end;

const FILE_NOT_FOUND=-1;

function filelength( fname:string ):longint;
{ Utility, returns -1 if file not found, else it returns the length of the file }
var srec:Tsearchrec;
const AnyFile=$3f;
begin
     fname := fname+#0;
     filelength := FILE_NOT_FOUND;
     findfirst( @fname[1], AnyFile,srec );
     if doserror<>0 then exit;
     filelength := srec.size;
end;

function Tcode32man.loadFile( fname:string ):integer;
{ Does an allocMem (see above) on a chunk of memory big enough to hold this file,  }
{ and then loads that file into this memory.                                       }
{ Presumably the file is an assembly file coded with the USE32 directive and then  }
{ linked and striped down to a BIN file (good old EXE2BIN).                        }

{ Note that this function only works on files less than 64k large.                 }
{                                                                                  }
{ loadFile errors 1 = file not found }
{                 2 = file too big (>64k) }
{                 3 = could alloc mem }
{                 4 = could not open file }
{                 5 = could not read all of file }
var size:longint;
    nread:word;
    io:word;
    f:file;
begin
     { Get the filesize and make sure it is not too big }
        loadFile := 1;
        size := filelength( fname );
        if size=FILE_NOT_FOUND then exit;

        loadFile := 2;
        if size>$ffff then exit;{ blockread won't handle files>64k }

     { ALlocate the memory }
        loadFile := 3;
        allocMem( size );
        if allocErr<>0 then exit;

     { Open the file }
        loadFile := 4;
        assign(f,fname);
       {$i-}reset(f,1);{$i+}
        io := IOresult;
        if io<>0 then exit;

     { Read the file }
        loadFile := 5;
        blockread(f,dataMarker^,size,nread );
        if nread=size then loadFile := 0;

     { Cleanup }
        close(f);
end;

function Tcode32man.makeCallGate( offset:word; ring:byte; nParam:word ):pointer;
{ Make a call gate that using a LDTmesser }
var CGp:PcallGateDescriptor;
    CSp:PselDescriptor;
    LDT:LDTmesser;
    codeptr:pointer;
begin
     LDT.init;

     codeptr := ptr( codeAlias,offset );
     callGatePtr := LDT.CreateCallGate32( codeptr, ring, nParam );

     { for testing only }
     (*
     CSp := LDT.GetSelectorDesP( hiword(longint(codeptr)) );
     CSp^.dump(output,'CS');
     CGp := LDT.GetCallGateDesP( hiword(longint(callGatePtr)) );
     CGp^.dump(output,'CG');
     *)

     LDT.done;

     makeCallGate := callGatePtr;
end;

destructor Tcode32man.done;
{ Freeup everything we allocated }
var LDT:LDTmesser;
begin
     if callGatePtr<>nil then begin
        LDT.init;
        LDT.freeCallGate( callGatePtr );
        LDT.done;
        callGatePtr := nil;
        end;
     if codeAlias<>0 then begin
        Global32CodeAliasFree( selector32, codeAlias,  0 );
        codeAlias := 0;
        end;
     if dataPtr<>nil then begin
        Global16PointerFree( selector32, longint(dataPtr), 0 );
        dataPtr := nil;
        end;
     if selector32<>0 then begin
        Global32Free( selector32, 0 );
        selector32 := 0;
        end;
end;


{-------------------------------- Test routines ---------------------------------------- }
var glbeax:longint;

procedure test32_1;
{ test the code with a simple example that just decrements a number }
var codeman:Tcode32man;
var doit:procedure;
    offset:word;
    ring:byte;
    nparam:word;
begin
     { Allocate memory }
        codeman.init;
        codeman.allocMem( 32 );

     { "Compile" our code into the memory }
        codeman.code2( $33,$c0 );{ xor ax,ax }
        codeman.code1( $48     );{ dec ax    }
        codeman.code1( $cb     );{ retf      }

     { Make the callgate }
        offset := 0;
        ring := 0;
        nparam := 0;
        @doit := codeman.makeCallGate( offset, ring, nparam );

     { Test it }
        inline( $66/$33/$c0 );{ xor eax,eax }
        doit;
        inline( $66/$a3/>glbeax );{ mov [glbeax],eax }
        writeln('glbeax:',glbeax);

     { Note if glbeax=65635 ($0000ffff) then the above was executed as 16 bit code }
     { and if glbeax=-1 ($ffffffff) then the above was executed as 32 bit code! }

     { Clean up }
        codeman.done;
end;

procedure test32_2;
{ For this example you need the file s32.asm           }
{ you then make s32.bin by the sequence: tasm s32;     }
{                                        tlink s32;    }
{                                        exe2bin s32;  }
var codeman:Tcode32man;
var add2longs:function (l1,l2:longint):longint;
    offset:word;
    ring:byte;
    nparam:word;
    err:integer;
    result:longint;
begin
     { Init, allocate, and load file }
        codeman.init;
        err := codeman.LoadFile( 's32.bin' );
        if err<>0 then exit;

     { Make the callgate }
        offset := 0;
        ring := 3;
        nparam := 0;{ since we are in ring 3 already, there will be no stack switch }
        @add2longs := codeman.makeCallGate( offset, ring, nparam );

        writeln('@add2longs:',hexad(@add2longs),' ss:sp:',hex(sseg),':',hex(sptr));

     { Test it }
        result := add2longs($10000, $12345 );
        writeln('result:',hexl(result));

     { Clean up }
        codeman.done;
end;

end.