/* pmbdskfe.c - Sysbench front end to Kai Uwe Rommel's Diskio code
 *
 * Author:  Trevor Hemsley <Trevor-Hemsley@dial.pipex.com?
 * Created: Mon Apr 14 1997
 */

#define INCL_DOS
#define INCL_DOSDEVICES
#define INCL_DOSDEVIOCTL
#define INCL_DOSERRORS

#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/* prototype definitions for functions */
double pmb_diskio_avseek(int disknr);
double pmb_buscache_xfer(int disknr);
double pmb_diskio_transfer(int disknr);
double pmb_diskio_cpupct(int disknr);
double pmb_diskio_disksize(int nDisk);
int    pmb_diskio_nrdisks(void);

int    pmb_cdio_nrcds(void);
double pmb_cdio_disksize(int nDrive);
double pmb_cdio_avseek(int nDrive);
double pmb_cdio_transfer(int nDrive);
double pmb_cdio_cpupct(int nDrive);

int nHandle;
unsigned nSides, nTracks, nSectors;
BOOL doneDhry = 0;

extern char *pBuffer;

extern int time_over;
extern double dhry_time, dhry_result;

extern int    bench_hd_seek(int nHandle, unsigned nSides, unsigned nSectors, unsigned nTracks);
extern int    bench_hd_bus(int nHandle, unsigned nSectors);
extern int    bench_hd_transfer(int nHandle, int nTrack, int nDirection, unsigned nSides, unsigned nSectors);
extern int    bench_hd_cpuusage(int nHandle, unsigned nSides, unsigned nSectors);
extern int    bench_cd_transfer(int nHandle);
extern int    bench_cd_cpuusage(int nHandle);
extern int    bench_cd_seek(int nHandle, unsigned nSectors);

extern int    bench_dhry(void);
//extern double bench_concurrent(void);

extern void   err(char* s);

#include "diskacc2.h"


double pmb_diskio_avseek(int disknr)
{
  double r;
  char szName[8];

  disknr++;
  sprintf(szName, "$%d:", disknr);

  if ((nHandle = DskOpen(szName, 0, 0, &nSides, &nTracks, &nSectors)) < 0)
    return printf("\nCannot access disk %d.\n", disknr), -1;

  if ((pBuffer = malloc(nSectors * 512)) == NULL)
    return printf("\nNot enough memory.\n"), -1;

  r = bench_hd_seek(nHandle, nSides, nSectors, nTracks);

  free(pBuffer);
  DskClose(nHandle);

  return r;
}

double pmb_buscache_xfer(int disknr)
{
  double r;
  char szName[8];

  disknr++;

  sprintf(szName, "$%d:", disknr);

  if ((nHandle = DskOpen(szName, 0, 0, &nSides, &nTracks, &nSectors)) < 0)
     {
      err("Cannot access disk");
      return -1;
      }

  if ((pBuffer = malloc(nSectors * 512)) == NULL)
     {
      err("Not enough memory - bus cache test.");
      return -1;
      }

  r = bench_hd_bus(nHandle, nSectors);

  free(pBuffer);
  DskClose(nHandle);
  return r;
}

double pmb_diskio_transfer(int disknr)
{
  int r;
  char szName[8];

  disknr++;
  sprintf(szName, "$%d:", disknr);

  if ((nHandle = DskOpen(szName, 0, 0, &nSides, &nTracks, &nSectors)) < 0)
    return printf("\nCannot access disk %d.\n", disknr), -1;

  if ((pBuffer = malloc(nSectors * 512)) == NULL)
    return printf("\nNot enough memory.\n"), -1;

  r = (bench_hd_transfer(nHandle, 0, 1, nSides, nSectors) +
      bench_hd_transfer(nHandle, nTracks/2, 1, nSides, nSectors) +
      bench_hd_transfer(nHandle, nTracks-1, -1, nSides, nSectors))/3;

  free(pBuffer);
  DskClose(nHandle);

  return r;
}

double pmb_diskio_cpupct(int disknr)
{
  double r;
  char szName[8];
  APIRET rc;

  disknr++;

  sprintf(szName, "$%d:", disknr);

  if ((nHandle = DskOpen(szName, 0, 0, &nSides, &nTracks, &nSectors)) < 0)
     {
      err("Cannot access disk - cpu percent");
      return -1;
      }

  if ((pBuffer = malloc(nSectors * 512)) == NULL)
     {
      err("Not enough memory - cpu percent test.");
      return -1;
      }

  if (!doneDhry)
     {
     rc = DosSetPriority(PRTYS_THREAD, PRTYC_TIMECRITICAL, PRTYD_MAXIMUM, 0);

     r = bench_dhry();      /* get base value for CPU when we're running at hi priority */

     rc = DosSetPriority(PRTYS_THREAD, PRTYC_REGULAR, 0, 0);

     doneDhry = 1;
     }

  r = bench_hd_cpuusage(nHandle, nSides, nSectors);

  free(pBuffer);
  DskClose(nHandle);
  return r;
}


// return number of bytes on disk nDisk
double pmb_diskio_disksize(int nDisk) {
  char szName[8];

  nDisk++;

  sprintf(szName, "$%d:", nDisk);

  if ((nHandle = DskOpen(szName, 0, 0, &nSides, &nTracks, &nSectors)) < 0) {
    err("Disk IO test error : Cannot access disk.");
    exit(1);
  }

  DskClose(nHandle);
  return (nSides * nTracks * nSectors) / 2;    /* divide by 2 to get Kb on disk */
}

int pmb_diskio_nrdisks(void) {
  USHORT nDisks;
  int nCount;

  if (DosPhysicalDisk(INFO_COUNT_PARTITIONABLE_DISKS, &nDisks, sizeof(nDisks), 0, 0)) {
    err("Disk IO test error : Cannot determine number of disks.");
    exit(1);
  }
  return nDisks;
}


// return number of bytes on disk nDisk
double pmb_cdio_disksize(int nDrive) {
  int nHandle, nDriveLetter;
  unsigned nSectors;
  char szDrive[3], szUPC[8];

  nDrive++;

  if ((nDriveLetter = CDFind(nDrive)) == -1)
    return printf("\nCannot access CD-ROM drive %d.\n", nDrive), -1;

  szDrive[0] = (char) nDriveLetter;
  szDrive[1] = ':';
  szDrive[2] = 0;

  if ((nHandle = CDOpen(szDrive, 1, szUPC, &nSectors)) == -1)
    return -1;

  CDClose(nHandle);
  return (nSectors * 2);    /* multiply by 2 to get Kb on disk */
}


int pmb_cdio_nrcds(void) {
  USHORT nCDROMs;

  nCDROMs = CDFind(0);

  return nCDROMs;
}


double pmb_cdio_avseek(int nDrive)
{
  int nHandle, nDriveLetter;
  unsigned nSectors;
  char szDrive[3], szUPC[8];
  double r;

  nDrive++;

  if ((nDriveLetter = CDFind(nDrive)) == -1)
    return printf("\nCannot access CD-ROM drive %d.\n", nDrive), -1;

  szDrive[0] = (char) nDriveLetter;
  szDrive[1] = ':';
  szDrive[2] = 0;

  if ((nHandle = CDOpen(szDrive, 1, szUPC, &nSectors)) == -1)
    return -1;

  if ((pBuffer = malloc(32 * 2048)) == NULL)
    return printf("\nNot enough memory.\n"), -1;

  /* spin up and seek to first sector */
  if (CDRead(nHandle, 0, 32, pBuffer) == -1)
    return printf("CD-ROM read error.\n"), -1;

  r = bench_cd_seek(nHandle, nSectors);

  free(pBuffer);
  CDClose(nHandle);

  return r;
}


double pmb_cdio_transfer(int nDrive)
{
  int nHandle, nDriveLetter;
  unsigned nSectors;
  char szDrive[3], szUPC[8];
  double r;

  nDrive++;

  if ((nDriveLetter = CDFind(nDrive)) == -1)
    return printf("\nCannot access CD-ROM drive %d.\n", nDrive), -1;

  szDrive[0] = (char) nDriveLetter;
  szDrive[1] = ':';
  szDrive[2] = 0;

  if ((nHandle = CDOpen(szDrive, 1, szUPC, &nSectors)) == -1)
    return -1;

  if ((pBuffer = malloc(32 * 2048)) == NULL)
    return printf("\nNot enough memory.\n"), -1;

  /* spin up and seek to first sector */
  if (CDRead(nHandle, 0, 32, pBuffer) == -1)
    return printf("CD-ROM read error.\n"), -1;

  r = bench_cd_transfer(nHandle);

  free(pBuffer);
  CDClose(nHandle);

  return r;
}


double pmb_cdio_cpupct(int nDrive)
{
  int nHandle, nDriveLetter;
  unsigned nSectors;
  char szDrive[3], szUPC[8];
  double r;
  APIRET rc = 0;

  nDrive++;

  if ((nDriveLetter = CDFind(nDrive)) == -1)
    return printf("\nCannot access CD-ROM drive %d.\n", nDrive), -1;

  szDrive[0] = (char) nDriveLetter;
  szDrive[1] = ':';
  szDrive[2] = 0;

  if ((nHandle = CDOpen(szDrive, 1, szUPC, &nSectors)) == -1)
    return -1;

  if ((pBuffer = malloc(32 * 2048)) == NULL)
    return printf("\nNot enough memory.\n"), -1;

  /* spin up and seek to first sector */
  if (CDRead(nHandle, 0, 32, pBuffer) == -1)
    return printf("CD-ROM read error.\n"), -1;

  if (!doneDhry)
     {
     rc = DosSetPriority(PRTYS_THREAD, PRTYC_TIMECRITICAL, PRTYD_MAXIMUM, 0);

     r = bench_dhry();      /* get base value for CPU when we're running at hi priority */

     rc = DosSetPriority(PRTYS_THREAD, PRTYC_REGULAR, 0, 0);

     doneDhry = 1;
     }

  r = bench_cd_cpuusage(nHandle);

  free(pBuffer);
  CDClose(nHandle);

  return r;
}


