/***
* STAWIN.PRG by Jim Schaffner
*
* Displays a status bar across the screen when performing a long
* running task, such as indexing or reporting.  This is a Clipper 5
* version of the Foxpro STATWIN function, originally written by
* Duane Keeling of Keeling Consulting.
*
* To use this, first call StatInit() with the desired top/left
* coordinates of the display "window".  Coordinates are optional.
* Then call StatWin() from within your processing loop, passing
* the total to be processed, and the current process count.  There
* are optional parameters for a display message, a logical for
* displaying estimated completion time, a title, and colors.
*
* Here's an example:
*
* StatInit()
* do while ! eof()
*    StatWin(lastrec(),recno(),"Percent processed:",.t.)
*    skip
* enddo
*
* The StatWin() function will automatically call the cleanup routine
* StatExit() when 100% has been reached, but a premature break from
* the processing loop will cause this not to happen.  For this reason,
* I have made the StatExit() function public rather than static, so
* you can call it from outside this PRG to handle exceptions.  If you
* fail to call StatExit(), you will get some strange display results
* the next time you run StatWin().
*
* Here's an exception example:
*
* StatInit()
* do while ! eof()
*    StatWin(lastrec(),recno())
*    if NAME = "SMITH"
*       StatExit()
*       exit
*    endif
*    skip
* enddo
*
* Compile with /a/m/n.
*
* Enjoy!
*/

// "borrowed" from Clipper 5.2 COMMON.CH (allows compile in 5.01)

#xcommand DEFAULT <v1> TO <x1> [, <vn> TO <xn> ]                        ;
          =>                                                            ;
          IF <v1> == NIL ; <v1> := <x1> ; END                           ;
          [; IF <vn> == NIL ; <vn> := <xn> ; END ]


#include "setcurs.ch"

// Static variables:

//  ?Sav???         -> save video attributes, screen, etc.
//  nTop ... nRight -> screen coordinates for display
//  lFirst          -> denotes first pass through routine

STATIC cSavWin,cSavClr,nSavCsr
STATIC nSavRow,nSavCol,lFirst := .t.
STATIC nTop,nLeft,nBottom,nRight

/***
* StatInit() - initialize the status display
*/

FUNCTION StatInit(nT,nL)

   nTop    := if(nT == NIL,0,nT)

   if nTop > maxrow()-6
      nTop := maxrow()-6
   endif

   nLeft   := if(nL == NIL,0,nL)

   if nLeft > maxcol()-53
      nLeft := maxcol()-53
   endif

   nBottom := nTop+5
   nRight  := nLeft+52

   cSavWin := savescreen(nTop,nLeft,nBottom+1,nRight+1)

   cSavClr := setcolor()
   nSavCsr := setcursor()
   nSavRow := row()
   nSavCol := col()

   setcursor(SC_NONE)

RETURN NIL

/***
* StatWin() - display the status bar
*/

FUNCTION StatWin(nTotal,nCurrent,cMsg,lEstimate,cTitle,cColor)

   LOCAL nWidth,cFull,cHalf,nPct,nBlock,nBlocks,nEst,nCol

   STATIC nStart

   cFull  = chr(219) // 
   cHalf  = chr(221) // 
   nWidth = 50                      // default bar length to 50 characters

   setcolor("W+/BG,W+/BG")

   default cMsg to ""               // Default to NULL display message

   default lEstimate to .f.         // Default to NO estimating

   default cTitle to ""             // Default to NO window title

   default cColor to "W+/BG"        // default color to high white on cyan

   nPct := if(nTotal > 0,round((nCurrent*100)/nTotal,0),100)

   nPct := if(nPct > 100,100,nPct)  // This will prevent screen overrun and percentages > 100

   if lFirst
      @ nTop,nLeft clear to nBottom,nRight
      @ nTop,nLeft to nBottom,nRight

     // I have my own shadow routine, so I'll comment this out,
     //  but take note if you want to include your own shadow.

     // Shadow(nTop,nLeft,nBottom,nRight)

      if lEstimate
         nStart := seconds()
         @ nTop+4,nLeft+01 say "Start: " + TimeAsString(nStart)
         @ nTop+4,nLeft+22 say "Estimated Completion: "
      endif
      if ! empty(cMsg)                        // If we have a message then
         @ nTop+1,nLeft+1 say padr(cMsg,48)   // display with padded spaces
      endif
      lFirst := .f.
   else
      if lEstimate
         nEst := seconds() - nStart
         nEst := (nEst*100)/nPct
         @ nTop+4,nLeft+44 say TimeAsString(nStart+nEst+2)
      endif
   endif

   nBlock  := int(100/nWidth)

   nBlocks := int(nPct/nBlock)        // How many blocks do I need?

   nCol    := if(empty(cMsg),26,47)   // Calculate percentage display column

   @ nTop+1,nCol say str(nPct,3)+"%"       // Display percentage

   if nPct/2 # int(nPct/2)        // Add a half block if needed
      @ nTop+02,nLeft+01 say replicate(cFull,nBlocks)+cHalf
   else                           // and display the bar...
      @ nTop+02,nLeft+01 say replicate(cFull,nBlocks)
   endif

   if nCurrent >= nTotal              // If we're done (100%) then
      inkey(1)                        // wait 1 second...
      StatExit()                      // ...and clean up
   endif

RETURN NIL


// "borrowed" this one from Clipper sample source

/***
*  TimeAsString( <nSeconds> ) --> cTime
*  Convert numeric seconds to a time string
*
*  NOTE: Same as TSTRING() in Examplep.prg
*/
STATIC FUNCTION TimeAsString( nSeconds )
   RETURN StrZero(INT(Mod(nSeconds / 3600, 24)), 2, 0) + ":" +;
		  StrZero(INT(Mod(nSeconds / 60, 60)), 2, 0) + ":" +;
		  StrZero(INT(Mod(nSeconds, 60)), 2, 0)


/***
* StatExit() - exit from the status display & restore attributes
*/

FUNCTION StatExit()

   restscreen(nTop,nLeft,nBottom+1,nRight+1,cSavWin)

   setcolor(cSavClr)
   setcursor(nSavCsr)
   setpos(nSavRow,nSavCol)

   cSavWin := ""
   lFirst  := .t.

RETURN NIL