/*===========================================================================
SOLAR v0.94 :: Module solar.c

This source code has been placed into the public domain.

History:  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
04-20-93 KJH  Started history.
05-03-93 KJH  Changed LOCAL packet processing to copy packet to temp
              directory rather than renaming.
06-12-93 KJH  Added command line /v for Waffle version.
              Added exit status LOCALREPLY and fixed packet copy bug.
07-15-94 KJH  Changed all printf() to fprintf()
===========================================================================*/

/* Header Files */
#include <dir.h>
#include <limits.h>
#include <conio.h>
#include <string.h>
#include <stdio.h>
#include <time.h>

/* Local Definitions */
#define VERSION              "0.94"   /* Version of Solar                */
#define SOLAR_JOIN     "NEWSRC.HPG"
#define SOLAR_JOIN_TMP "NEWSRC.TMP"
#define MAILBOX_FILE    "MAILBOX.F"
#define MAILBOX_IDX     "MAILBOX.I"
#define YES                      0
#define NO                       1

/* Supported Helldiver Packet Format Types */
enum Helldiver_Types { u, M, m, n, C, c };

/* Global Data */
char temp_path[MAXPATH];          /* Path to temporary directory      */
char user_path[MAXPATH];          /* Path to user's directory         */
char help_path[MAXPATH];          /* Full path to Solar help file     */
char log_path[MAXPATH];           /* Full path to Solar log file      */
char waffle_path[MAXPATH];        /* Full path to Waffle directory    */
char config_path[MAXPATH];        /* Full path to Solar config file   */
char solar_path[MAXPATH];         /* Full path to Solar directory     */
char describe_path[MAXPATH];      /* Full path to area descriptions   */

char uucp_name[10];               /* Host system UUCP name            */
char speed[6];                    /* Modem connect speed              */
char compress[10];                /* Compression method               */
char protocol[10];                /* Transfer protocol                */
char waffle_version[5];           /* Waffle version number            */

int time_left;                    /* User's time remaining on line    */
int news_message;                 /* Default to Usenet format         */
int mail_message;                 /* Default to MMDF format           */
int news_index;                   /* Default to full C-news overview  */
int delete_mail;                  /* Delete mailbox after transfer    */
int send_mail;                    /* Batch mailbox to packet          */
int send_list;                    /* Send LIST file in packet         */
int use_wafjoin;                  /* Use Waffle's join file           */

long news_total_bytes;            /* Total max. bytes to batch        */
long news_area_bytes;             /* Max. bytes per group to batch    */
long news_total_messages;         /* Total max. number of articles    */
long news_area_messages;          /* Max. articles per group          */

char logbuf[60];
char _slrerr[80];                 /* Solar error buffer               */

/* External Functions */
extern int getopt(int argc, char *argv[], const char *optionS);
extern int load_static(char *username);
extern int load_config();
extern int user_options();
extern int user_defaults();
extern int timechk(int min_left);
extern long batch_messages();
extern long batch_mail(char username[9]);
extern long compress_packet();

/* Local Functions */
void usage();
void cleanup();
int  user_defaults();
int  write_status(char *status);
int  logit();
int  prepare_xfer_status(char *username, long message_count, long mail_count);
int  prepare_reply_status();
int  copyfile(char *path1, char *path2);
void update_join(char *username);

int c_break(void)
{
  fprintf(stdout,"\nControl-break pressed, Solar aborting...\n");
  fcloseall();
  cleanup();
  write_status("ABORT");
  return 0;
}

int main(int argc, char *argv[])
{
	int c_break(void);
	extern char *optarg;
	unsigned char x;
  const char *optionS = "u:U:c:C:t:T:s:S:v:V:";
	char username[9];

	long message_count = 0L;
	long mail_count    = 0L;

	/********* Initialize global variables to standard defaults *********/

	strcpy(temp_path,"NONE");
	strcpy(user_path,"NONE");
	strcpy(help_path,"NONE");
	strcpy(log_path,"NONE");
	strcpy(config_path,"NONE");
  strcpy(solar_path,"NONE");
  strcpy(uucp_name,"SOLAR");
  strcpy(speed,"LOCAL");        /* Default to LOCAL       */
  strcpy(compress,"NONE");
  strcpy(protocol,"NONE");
  strcpy(username,"NONE");
	strcpy(describe_path,"NONE");
  strcpy(waffle_version,"NONE");

  time_left           = INT_MAX;

  send_mail           = NO;
  delete_mail         = NO;
  send_list           = NO;
  use_wafjoin         = NO;
  news_message        = u;
  mail_message        = M;
  news_index          = c;

	news_total_bytes    = LONG_MAX;
  news_area_bytes     = LONG_MAX;
	news_total_messages = LONG_MAX;
	news_area_messages  = LONG_MAX;

  logbuf[0] = '\0';

  /* Parse the command line and load file settings */

	while ((x = getopt(argc, argv, optionS)) != '\xFF')
	{
    switch (toupper(x)) {
			case '?'  : usage();
									goto UsageExit;
      case 'U'  : if ((strcmp(user_path,"NONE")) == 0)
                  {
										strcpy(username,optarg);
                    if (load_static(username) != 0) goto ErrorExit;
                    break;
                  }
                  else
                  {
                    strcpy(_slrerr,"Cannot specify username twice on command line");
                    goto ErrorExit;
                  }
      case 'C'  : if ((strcmp(config_path,"NONE")) == 0)
									{
                    strcpy(config_path,optarg);
                    break;
                  }
                  else
                  {
                    strcpy(_slrerr,"Cannot specify two configuration files on command line");
                    goto ErrorExit;
                  }
      case 'T'  : time_left = atoi(optarg); break;
      case 'S'  : strcpy(speed,optarg); break;
      case 'V'  : strcpy(waffle_version,optarg); break;
    }
	}

  /* The parameter /u must be on the command line */
  if ((strcmp(username,"NONE")) == 0)
	{
    usage();
    goto UsageExit;
	}
  if (load_config() != 0) goto ErrorExit;
  user_defaults();

  cleanup();

	switch (user_options()) {
    /* Error! */
    case -1 : goto ErrorExit;
    /* Exit Solar */
    case 0  : write_status("ABORT");
              break;
    /* Batch packet for download */
    case 1  : fprintf(stdout,"Press Ctrl-C to abort...\n");
              ctrlbrk(c_break);
              if (send_mail == YES)
              {
                if (timechk(time_left) != 0) goto ErrorExit;
                if ((mail_count = batch_mail(username)) < 0) goto ErrorExit;
							}
              if ((message_count = batch_messages()) < 0) goto ErrorExit;
              if (send_mail == YES) fprintf(stdout,"\nMail messages packed : %lu\n",mail_count);
              fprintf(stdout,"Messages packed : %lu\n",message_count);
              if ((message_count > 0) || (mail_count > 0))
              {
                if (prepare_xfer_status(username,message_count,mail_count) != 0) goto ErrorExit;
              }
              else
              {
                write_status("ABORT");
              }
              break;
		case 2  : if (timechk(time_left) != 0) goto ErrorExit;
              if (prepare_reply_status() != 0) goto ErrorExit;
              break;
  }
GoodExit:
  return 0;
UsageExit:
  return 1;
ErrorExit:
  write_status("ABORT");
  fprintf(stderr,"<solar> fatal: %s\n",_slrerr);
  return 1;
}

/*
Function: cleanup()
Purpose : Remove temporary directory and all files in it.
Return  : N/A
*/

void cleanup()
{
	struct ffblk ffblk;

  char work_path[MAXPATH];
  char del_path[MAXPATH];

  int done = 0;

  strcpy(work_path, temp_path);
  strcat(work_path, "\\*.*");
	done = findfirst(work_path,&ffblk,0);
	while (!done)
	{
    strcpy(del_path, temp_path);
		strcat(del_path, "\\");
		strcat(del_path, ffblk.ff_name);
		unlink(del_path);
		done = findnext(&ffblk);
	}

  strcpy(work_path, user_path);
  strcat(work_path, "\\NEWSRC.TMP");
  unlink(work_path);

  return;
}

/*
Function: write_status()
Purpose : Output a string to the Solar status file.
Return  : 0 on success, non-zero and set _slrerr on error.
*/

int write_status(char *status)
{
  char path[MAXPATH];
  FILE *status_file;

  strcpy(path,temp_path);
  strcat(path,"\\status.tmp");
  if ((status_file = fopen(path,"wt")) == NULL)
  {
    sprintf(_slrerr,"error opening %s for write");
    goto ErrorExit;
	}
  fprintf(status_file,"%s\n",status);
  fclose(status_file);
GoodExit:
  return 0;
ErrorExit:
  return 1;
}

int logit()
{
  FILE *log_file = fopen(log_path,"at");
  time_t t;

  char date_time[15];
  char temp[26];

  if (!log_file)
  {
    return -1;
  }
  t = time(NULL);
  sprintf(temp,"%s",ctime(&t));

  date_time[0]  = temp[8];
  date_time[1]  = temp[9];
	date_time[2]  = '-';
  date_time[3]  = temp[4];
  date_time[4]  = temp[5];
  date_time[5]  = temp[6];
  date_time[6]  = '-';
  date_time[7]  = temp[22];
  date_time[8]  = temp[23];
  date_time[9]  = ' ';
  date_time[10] = temp[11];
  date_time[11] = temp[12];
  date_time[12] = temp[13];
  date_time[13] = temp[14];
  date_time[14] = temp[15];
	date_time[15] = '\0';

	fprintf(log_file,"%s | %s",date_time,logbuf);
	fclose(log_file);
	return 0;
}

/*
Function: usage()
Purpose : Display Solar command line usage.
Return  : N/A
*/

void usage()
{
	fprintf(stdout,"Solar v%s\n\n",VERSION);
  fprintf(stdout,"Command Line Usage :\n\n");
  fprintf(stdout,"solar /u username [/c config] [/s speed] [/t time_left] [/v version]\n\n");
  fprintf(stdout,"\t/u username    User's account name (required)\n");
  fprintf(stdout,"\t/c config      Solar configuration file\n");
  fprintf(stdout,"\t/s speed       Modem speed. Default is LOCAL\n");
  fprintf(stdout,"\t/t time_left   User's time remaining\n");
  fprintf(stdout,"\t/v version     Waffle version (i.e. 1.64)\n\n");
  fprintf(stdout,"See documentation for further detail\n");
	return;
}

/*
Function: prepare_xfer_status()
Purpose : A packet was generated, do the rest.
Return  : 0 on success, non-zero on error.
*/

int prepare_xfer_status(char *username, long message_count, long mail_count)
{
	long packet_size = 0L;
	char now;
  int down_time;
  int xfer_speed, flag = 0;

  if ((packet_size = compress_packet()) < 0L) goto ErrorExit;
  if (packet_size == 0L)
  {
    fprintf(stdout,"Zero byte packet found. Aborting.\n");
    fprintf(stdout,"Subscription file will not be updated\n");
    write_status("ABORT");
  }
  else
  {
    fprintf(stdout,"Packet size : %lu\n",packet_size);
    sprintf(logbuf," %-8s | B  %4lu news %3lu mail  %8lu bytes\n",username,message_count,mail_count,packet_size);
    logit();
    if ((stricmp(speed,"LOCAL")) != 0)
    {
      xfer_speed = atol(speed);
      if ((packet_size != 0) && (xfer_speed != 0))
      {
        down_time = ((packet_size / (xfer_speed / 10)) / 60);
        fprintf(stdout,"Approx. transfer time : %u minute(s)\n",down_time);
      }
      if (timechk(time_left + down_time) != 0)
      {
				fprintf(stdout,"<solar> - not enough time to transfer!\n");
        write_status("ABORT");
        goto GoodExit;
      }
      flag = 0;
      while (flag == 0)
      {
        fprintf(stdout,"Transfer packet, or Abort [T,a] ? ");
        now = getche();
        fprintf(stdout,"\n\n");
				if (now == 'T' || now == 't' || now == '\x0D')
				{
					fprintf(stdout,"If for some reason the transfer is aborted, your\n");
					fprintf(stdout,"packet will be moved into %s\n",user_path);
					fprintf(stdout,"Sending with %s, start receiving now...\n",protocol);
					write_status("TRANSFER");
					flag = 1;
				}
				else
				{
					if (now == 'A' || now == 'a')
					{
						flag = 1;
						write_status("ABORT");
            fprintf(stderr,"Aborting.. subscription file will not be updated.");
					}
					else
					{
            fprintf(stdout,"%c ?\n",now);
					}
				}
			}
		}
		else
		{
      update_join(username);
			write_status("LOCAL");
		}
	}
GoodExit:
	return 0;
ErrorExit:
	return 1;
}

/*
Function: prepare_reply_status()
Purpose : User wants to upload replies. Prepare for it.
Return  : 0 on success, non-zero on error or if speed=LOCAL.
*/

int prepare_reply_status()
{
	char now;
	char packetpath[MAXPATH];
	char newpath[MAXPATH];
	int flag = 0;
	struct ffblk ffblk;

	if (stricmp(speed,"LOCAL") == 0)
	{
    fprintf(stdout,"Enter full path to packet file: ");
    gets(packetpath);
    fprintf(stdout,"\n");
    if (strcmp(packetpath,"\x0D") == 0)
    {
			write_status("ABORT");
      goto GoodExit;
		}
		if (findfirst(packetpath,&ffblk,0) == 0)
		{
			strcpy(newpath,temp_path);
			strcat(newpath,"\\");
			strcat(newpath,ffblk.ff_name);
			fprintf(stdout,"copying %s --> ",packetpath);
      if (copyfile(packetpath,newpath) != 0)
      {
        write_status("ABORT");
        fprintf(stdout,"\n");
				goto ErrorExit;
      }
			fprintf(stdout,"%s\n",newpath);
      write_status("LOCALREPLY");
			goto GoodExit;
    }
    else
    {
			fprintf(stdout,"Packet %s not found\n",packetpath);
      write_status("ABORT");
      goto GoodExit;
    }
  }

  while (flag == 0)
	{
    fprintf(stdout,"Upload reply packet [Y,n] ? ");
		now = getche();
    fprintf(stdout,"\n");
		if (now == 'Y' || now == 'y' || now == '\x0D')
		{
      fprintf(stdout,"Receiving with %s, start sending now...\n",protocol);
			write_status("REPLY");
			flag = 1;
		}
		else
		{
			if (now == 'N' || now == 'n')
			{
				flag = 1;
				write_status("ABORT");
			}
			else
			{
        fprintf(stdout,"%c ?\n",now);
			}
		}
	}
GoodExit:
	return 0;
ErrorExit:
	return 1;
}

/*
Function: int view_file(char filepath[MAXPATH])
Purpose : Display a text file.
Return  : 0 on success, non-zero on error.
*/

int view_file(char filepath[MAXPATH])
{
  FILE *viewfile = NULL;
  char buffer[80];
  int  lines = 0;

  if ((viewfile = fopen(filepath,"rt")) == NULL)
	{
    goto ErrorExit;
	}
	while ((fgets(buffer,80,viewfile)) != NULL)
	{
    fprintf(stdout,"%s",buffer);
		lines++;
		if (lines > 23)
		{
      fprintf(stdout,"[Any key for more, Q to quit]");
      lines = getch();
      fprintf(stdout,"\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b");
      fprintf(stdout,"                             ");
      fprintf(stdout,"\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b");
			if ((lines == 81) || (lines == 113)) break;
      lines = 0;
		}
	}
  fclose(viewfile);
GoodExit:
  return 0;
ErrorExit:
  return 1;
}

/*
Function: update_join(char *username)
Purpose : Delete old join file and rename temp. join.
Return  : N/A
*/

void update_join(char *username)
{
	extern int newsrc2join();

	char old_path[MAXPATH];
	char new_path[MAXPATH];
	int done;
	struct ffblk ffblk;

  strcpy(old_path,user_path);
  strcat(old_path,"\\");
  strcat(old_path,SOLAR_JOIN);
  strcpy(new_path,user_path);
  strcat(new_path,"\\");
  strcat(new_path,SOLAR_JOIN_TMP);
  unlink(old_path);
	rename(new_path,old_path);
	if (use_wafjoin == YES)
  {
    if (newsrc2join() != 0)
    {
      fprintf(stderr,"<solar> error: %s\n",_slrerr);
    }
  }
  if ((delete_mail == YES) && (send_mail == YES))
  {
    strcpy(old_path, user_path);
    strcat(old_path, "\\");
    strcpy(new_path,old_path);
		if (strcmp(waffle_version,"1.64") != 0)
    {
      strcat(old_path, MAILBOX_FILE);
      unlink(old_path);
      strcpy(old_path, user_path);
      strcat(old_path, "\\");
      strcat(old_path, MAILBOX_IDX);
      unlink(old_path);
    }
    else
    {
      strcat(old_path,username);
      strcat(old_path,".*");
			done = findfirst(old_path,&ffblk,0);
      while (!done)
      {
        strcpy(new_path,ffblk.ff_name);
        unlink(new_path);
        done = findnext(&ffblk);
      }
    }
  }
  return;
}

/*
Function: int copyfile(char *path1, char *path2)
Purpose: Copies file path1 to path2.
Return: zero on success, non-zero on error and set _slrerr.
*/

int copyfile(char *path1, char *path2)
{
  FILE *file1 = NULL;
  FILE *file2 = NULL;
  char copybuf;

  file1 = fopen(path1,"rb");
  file2 = fopen(path2,"wb");

  if ((!file1) || (!file2))
  {
    strcpy(_slrerr,"local packet copy failed");
    goto ErrorExit;
  }
  copybuf = fgetc(file1);
  while (!feof(file1))
  {
    fprintf(file2,"%c",copybuf);
    copybuf = fgetc(file1);
  }
  fclose(file1);
  fclose(file2);

GoodExit:
  return 0;
ErrorExit:
  if (file1) fclose(file1);
  if (file2) fclose(file2);
  return 1;
}

