/*===========================================================================
SOLAR v0.94 :: Module msg2hldr.c

This source code has been placed into the public domain.

History:  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
07-01-93 KJH  Started history.
07-15-94 KJH  Changed all printf() to fprintf()
===========================================================================*/

#include <stdio.h>
#include <string.h>
#include <dir.h>
#include <limits.h>

#define AREAS_NAME       "AREAS"    /* Complies with v1.1 of Helldiver */
#define BATCH_EXT          "MSG"    /* packet format.                  */
#define INDEX_EXT          "IDX"

extern enum Helldiver_Types { u, M, m, n, C, c };  /* Supported types */

extern char temp_path[MAXPATH];          /* Path to temporary directory      */
extern char user_path[MAXPATH];          /* Path to user's directory         */
extern char log_path[MAXPATH];           /* Full path to Solar log file      */
extern char describe_path[MAXPATH];      /* Full path to newsgroup desc file */

extern int time_left;                    /* User's time remaining on line    */
extern int news_message;                 /* HD type for news messages        */
extern int news_index;                   /* HD type for news index           */

extern long news_area_bytes;             /* Max. bytes per group to batch    */
extern long news_area_messages;
extern long totbyte_count;

extern char join_name[80];
extern long last_read;                    /* The last article actually read   */
extern char _slrerr[80];

char batch_prefix[10];

FILE *open_new_join_file();
FILE *open_batch_file();
FILE *open_index_file();
FILE *open_areas_file();
long high_fileno();
long low_fileno();
long filesize(FILE *filename);
int  get_batch_prefix();
int  partial_cnews(FILE *message_file, FILE *index_file, long message_size, long offset);
int  full_cnews(FILE *message_file, FILE *index_file, long message_size, long offset);
int  usenet_message(FILE *message_file, FILE *batch_file);
int  kill_msg(FILE *message_file);
void dump_join_to_new();
void write_description(FILE *areas_file);
void clean_zero_byte_file();

long messages_to_helldiver()
{
  FILE *new_join_file;
	FILE *batch_file;
  FILE *index_file;
	FILE *message_file;
  FILE *areas_file;

	long high_msg = high_fileno();
	long low_msg  = low_fileno();
  long message_size = 0L;
  long byte_count = 0L;
  long offset = 0L;
  long area_msg_count = 0L;
	int  msg_index = 0;
	int  x;
  int  killcount = 0;
  char temp[10];
  char file_name[MAXFILE];

	struct ffblk ffblk;

  if (last_read >= high_msg)
	{
		if (high_msg == 0)
    {
      fprintf(stdout," - No messages... skipping.\n");
    }
    else
    {
      fprintf(stdout,"- No new messages... skipping.\n");
		}
    dump_join_to_new();
    goto GoodExit;
	}

  if (low_msg > last_read) last_read = low_msg - 1L;
  area_msg_count = high_msg - last_read;

  if (area_msg_count > news_area_messages)
  {
    fprintf(stdout," - found %lu, batching last %lu :  ",area_msg_count,news_area_messages);
    msg_index = (high_msg - news_area_messages) + 1;
    area_msg_count = news_area_messages;
  }
  else
  {
    fprintf(stdout," - %lu new messages, batching :  ",area_msg_count);
    msg_index = last_read + 1L;
  }

  if (get_batch_prefix() != 0) goto ErrorExit;
  if ((batch_file = open_batch_file()) == NULL) goto ErrorExit;
  if ((news_index == C) || (news_index == c))
  {
    if ((index_file = open_index_file()) == NULL) goto ErrorExit;
	}

  killcount = 0;
  area_msg_count = 0;
  ltoa(area_msg_count,temp,10);
  while (!(msg_index > high_msg))
	{
		itoa(msg_index,file_name,10);
    if ((message_file = fopen(file_name, "rt")) == NULL)
		{
      msg_index++;    /* Expected message isn't there anymore! */
		}
		else
		{
      if (kill_msg(message_file) != 0)
      {
        message_size = filesize(message_file);
        byte_count += message_size;
        if (byte_count > news_area_bytes)
        {
          fprintf(stdout,"\nExceed area maximum of %lu bytes.",news_area_bytes);
          fclose(message_file);
          break;
        }
        if (news_message == u)
        {
          fprintf(batch_file, "#! rnews %lu\n", message_size);
        }
        offset = ftell(batch_file);
        switch (news_index) {
          case C  : partial_cnews(message_file, index_file, message_size, offset);
                    break;
          case c  : full_cnews(message_file, index_file, message_size, offset);
                    break;
          case n  : break;
        }
        switch (news_message) {
          case u  : usenet_message(message_file, batch_file);
                    break;
        }
        area_msg_count++;
        totbyte_count += message_size;
        for (x = 0; x < strlen(temp); x++)
        {
          fprintf(stdout,"\b");
        }
        ltoa(area_msg_count,temp,10);
        fprintf(stdout,"%s",temp);
      }
      else
      {
        killcount++;
      }
      fclose(message_file);
      msg_index++;
    }
  }
  if (killcount > 0) fprintf(stdout," killed %u",killcount);
  fprintf(stdout,"\n");
  fclose(batch_file);
  if (index_file) fclose(index_file);

  if (area_msg_count > 0L)
  {
    if ((areas_file = open_areas_file()) == NULL) goto ErrorExit;
    fprintf(areas_file,"%s\t%s\t",batch_prefix,join_name);
    switch (news_message) {
      case u  : fprintf(areas_file, "u"); break;
    }
    switch (news_index) {
      case c  : fprintf(areas_file,"c"); break;
      case C  : fprintf(areas_file,"C"); break;
      case n  : fprintf(areas_file,"n"); break;
    }
    if (strcmp(describe_path,"NONE") != 0)
    {
      write_description(areas_file);
    }
    fprintf(areas_file,"\n");
    fclose(areas_file);
  }
  else
  {
    clean_zero_byte_file();
  }

  /* When writing to new join file, use 1 as the beginning value in
     the number range. */

  if ((new_join_file = open_new_join_file()) == NULL) goto ErrorExit;
  fprintf(new_join_file,"%s: 1-%u\n",join_name,msg_index - 1);
  fclose(new_join_file);

GoodExit:
  return area_msg_count;
ErrorExit:
  if (new_join_file) fclose(new_join_file);
  if (batch_file) fclose(batch_file);
  if (index_file) fclose(index_file);
  if (message_file) fclose(message_file);
  if (areas_file) fclose(areas_file);
  return -1L;
}

/*
Function: dump_join_to_new()
Purpose : Copy info taken from join file to new join file.
Return  : N/A
*/

void dump_join_to_new()
{
  FILE *new_join_file = open_new_join_file();
  fprintf(new_join_file,"%s: 1-%lu\n",join_name,last_read);
  fclose(new_join_file);
  return;
}

/*
Function: high_fileno()
Purpose : Scan directory of numeric filenames and return the highest number.
Return  : Highest numbered filename, or 0 if no files found.
*/

long high_fileno()
{
  long msg_number = 0L;
  long high_msg   = 0L;
  int  done;
  struct ffblk ffblk;

  done = findfirst("*",&ffblk,0);
	while (!done)
	{
    msg_number = atoi(ffblk.ff_name);
    if (msg_number > high_msg)
		{
      high_msg = msg_number;
		}
		done = findnext(&ffblk);
	}
  return high_msg;
}

/*
Function: low_fileno()
Purpose : Scan directory of numeric filenames and return the lowest number.
Return  : Lowest numbered filename, or 0 if no files found.
*/

long low_fileno()
{
  long msg_number = 0L;
	long low_msg   = LONG_MAX;
	int  done;
  struct ffblk ffblk;

  done = findfirst("*",&ffblk,0);
	while (!done)
	{
    msg_number = atoi(ffblk.ff_name);
    if (msg_number < low_msg)
		{
      low_msg = msg_number;
		}
		done = findnext(&ffblk);
	}
  return low_msg;
}

/*
Function: open_areas_file()
Purpose : Open Helldiver packet index file for append in binary mode.
Return  : A pointer to the index file, NULL on error and set _slrerr.
*/
FILE *open_areas_file()
{
  FILE *areas_file;

  char path[MAXPATH];

  strcpy(path,temp_path);
  strcat(path,"\\");
  strcat(path,AREAS_NAME);

  if ((areas_file = fopen(path,"ab")) == NULL)
  {
    sprintf(_slrerr,"error opening %s file %s", AREAS_NAME,path);
  }
  return areas_file;
}

/*
Function: open_new_join_file()
Purpose : Open a temporary file in user's directory.
Return  : Pointer to temporary file, NULL on error and set _slrerr.
*/

FILE *open_new_join_file()
{
  FILE *join_file = NULL;
  char path[MAXPATH];

  strcpy(path,user_path);
  strcat(path,"\\NEWSRC.TMP");
  if ((join_file = fopen(path,"at")) == NULL)
  {
    sprintf(_slrerr,"error opening %s", path);
  }
  return join_file;
}

/*
Function: open_batch_file()
Purpose : Open a batch file in binary mode in temp directory. Use unique
          filename prefix on a per user basis.
Return  : Pointer to batch file, exit on error.
*/

FILE *open_batch_file()
{
  FILE  *batch_file   = NULL;

  char  batch_path[MAXPATH];

  strcpy(batch_path, temp_path);
	strcat(batch_path, "\\");
  strcat(batch_path, batch_prefix);
  strcat(batch_path, ".");
  strcat(batch_path, BATCH_EXT);
  if ((batch_file = fopen(batch_path, "wb")) == NULL)
	{
    sprintf(_slrerr,"error opening batch file %s batch_path");
	}
  return batch_file;
}

/*
Function: open_index_file()
Purpose : Open an index file in binary mode in temp directory. Use unique
          filename prefix on a per user basis.
Return  : Pointer to index file, NULL on error and set _slrerr.
*/

FILE *open_index_file()
{
  FILE  *index_file   = NULL;

  char  index_path[MAXPATH];

  strcpy(index_path, temp_path);
  strcat(index_path, "\\");
  strcat(index_path, batch_prefix);
  strcat(index_path, ".");
  strcat(index_path, INDEX_EXT);
  if ((index_file = fopen(index_path, "wb")) == NULL)
	{
    sprintf(_slrerr,"error opening index file %s",index_path);
	}
  return index_file;
}

/*
Function: get_batch_prefix()
Purpose : Read seqno.slr file in user's directory and place value in
          file scope storage. Also, re-write an incremented seqno in
          seqno.slr for use next time.
Return  : Always return 0, exit on error.
*/

int get_batch_prefix()
{
  FILE  *seqno_file   = NULL;
  char  temp;
  char  seqno_path[MAXPATH];
  int   x     = 0;
  int   seqno = 1;

  strcpy(seqno_path, user_path);
  strcat(seqno_path, "\\SEQNO.HPG");
	if ((seqno_file = fopen(seqno_path, "rt")) == NULL)
	{
    strcpy(batch_prefix,"1");   /* Reset the prefix counter to 1 */
	}
	else
	{
		temp = fgetc(seqno_file);
		while (temp != '\n')
		{
      batch_prefix[x++] = temp;
			temp = fgetc(seqno_file);
		}
    batch_prefix[x] = '\0';
    seqno = atoi(batch_prefix);
    if (seqno > 32000) seqno = 0;
    fclose(seqno_file);
	}
	seqno++;
  if ((seqno_file = fopen(seqno_path, "wt")) == NULL)
  {
    strcpy(_slrerr,"error opening seqno file");
    goto ErrorExit;
  }
	fprintf(seqno_file, "%i\n", seqno);
	fclose(seqno_file);
GoodExit:
  return 0;
ErrorExit:
  return 1;
}

/*
Function: filesize()
Purpose : Determine the number of bytes in a file.
Return  : The number of bytes in the file, zero if none.
*/

long filesize(FILE *filename)
{
  long bytes  = 0L;
  long curpos = 0L;
	char temp;

	curpos = ftell(filename);
	fseek(filename, 0L, SEEK_SET);
	temp = fgetc(filename);
	while (!feof(filename))
	{
    if (temp != '\r') bytes++;
		temp = fgetc(filename);
	}
	fseek(filename, curpos, SEEK_SET);
  return bytes;
}

/*
Function: partial_cnews()
Purpose : Scan message file and create a partial C-news format index record.
Return  : 0 on success, non-zero on error.
*/

int partial_cnews(FILE *message_file, FILE *index_file, long message_size, long offset)
{
  char *extract_header(char *buf);

  char buf[1024];

  int  lines = 0;
  char subject[1024];
  char author[1024];
  char date[1024];

  subject[0] = '\0';
  author[0]  = '\0';
  date[0]    = '\0';

  while (fgets(buf,1024,message_file) != NULL)
	{
		if ((strnicmp(buf,"Subject:",8)) == 0)
		{
      strcpy(subject,extract_header(buf));
		}
		if ((strnicmp(buf,"From:",5)) == 0)
		{
      strcpy(author,extract_header(buf));
		}
		if ((strnicmp(buf,"Date:",5)) == 0)
		{
      strcpy(date,extract_header(buf));
		}
		if (strlen(buf) < 4)
		{
			while (fgets(buf,1024,message_file) != NULL)
			{
        lines++;
			}
      break;
		}
	}
  fseek(message_file,0L,SEEK_SET);    /* Reset message to beginning */
  fprintf(index_file,"%lu\t%s\t%s\t%s\t%lu\t%u\n",offset,subject,author, \
          date,message_size,lines);
  return 0;
}

int full_cnews(FILE *message_file, FILE *index_file, long message_size, long offset)
{
  char *extract_header(char *buf);

  char buf[1024];

  int  lines = 0;
  char subject[1024];
  char author[1024];
  char date[1024];
  char msgid[1024];
  char refs[1024];

  subject[0] = '\0';
  author[0]  = '\0';
  date[0]    = '\0';
  msgid[0]   = '\0';
  refs[0]    = '\0';

  while (fgets(buf,1024,message_file) != NULL)
	{
		if ((strnicmp(buf,"Subject:",8)) == 0)
		{
      strcpy(subject,extract_header(buf));
		}
		if ((strnicmp(buf,"From:",5)) == 0)
		{
      strcpy(author,extract_header(buf));
		}
    if ((strnicmp(buf,"Date:",5)) == 0)
		{
      strcpy(date,extract_header(buf));
		}
    if ((strnicmp(buf,"Message-ID:",11)) == 0)
		{
      strcpy(msgid,extract_header(buf));
		}
    if ((strnicmp(buf,"References:",11)) == 0)
		{
      strcpy(refs,extract_header(buf));
		}
    if (strlen(buf) < 4)
		{
			while (fgets(buf,1024,message_file) != NULL)
			{
        lines++;
			}
      break;
		}
	}
  fseek(message_file,0L,SEEK_SET);    /* Reset message to beginning */
  fprintf(index_file,"%lu\t%s\t%s\t%s\t%s\t%s\t%lu\t%u\n",offset,subject, \
          author,date,msgid,refs,message_size,lines);
  return 0;
}

int usenet_message(FILE *message_file, FILE *batch_file)
{
  char buffer[1024];

  while (fgets(buffer,1024,message_file) != NULL)
  {
    fprintf(batch_file,"%s",buffer);
  }
  return 0;
}

char *extract_header(char *buf)
{
  int x = 0;
  int y = 0;
  static char new_header[1024];

  for (x = strlen(buf) - 2; buf[x] == ' '; x--);
  buf[++x] = '\0';

  x = 0;
  while (buf[x++] != ':');
  while (buf[x++] != ' ');
  while (buf[x] != '\0')
  {
    if (buf[x] == '\t')
    {
      new_header[y++] = ' ';
      x++;
    }
    else
    {
      new_header[y++] = buf[x++];
    }
  }
  new_header[y] = '\0';
  return new_header;
}

/*
Function: void write_description(FILE *areas_file)
Purpose : Pull newsgroup description from file and add to areas file.
Return  : N/A
*/

void write_description(FILE *areas_file)
{
  FILE *desc_file = fopen(describe_path,"rt");
  char descbuf[128];
  char description[128];
	int x = 0, y = 0;

  if (!desc_file) goto ExitFunct;
  strcpy(description,"No Description");
  while (fgets(descbuf,128,desc_file) != NULL)
  {
    if (strnicmp(join_name,descbuf,strlen(join_name)) == 0)
    {
      x = 0;
      y = 0;
      while ((descbuf[x] != ' ') && (descbuf[x] != '\0') && (descbuf[x] != '\n')) x++;
      while ((descbuf[x] == ' ') && (descbuf[x] != '\0') && (descbuf[x] != '\n')) x++;
      while ((descbuf[x] != '\0') && (descbuf[x] != '\n'))
      {
        if (descbuf[x] == '\t')
        {
          description[y++] = ' ';
          x++;
        }
        else
        {
          description[y++] = descbuf[x++];
        }
      }
      break;
    }
  }
  description[y] = '\0';
  fprintf(areas_file,"\t%s",description);
  fclose(desc_file);
ExitFunct:
  return;
}

/*
Function: int kill_msg(FILE *message_file);
Purpose : If user has a KILL.HPG file, check the message's From:
          and Subject: headers against the entries in the kill file.
Return  : 0 if a string from kill file matches something in either
          header. non-zero if there is no match found, or no kill file.
*/

int kill_msg(FILE *message_file)
{
  char killpath[MAXPATH];
  FILE *killfile = NULL;
  char killstr[128];
  char header[1024];
  int flag = 0;

  strcpy(killpath, user_path);
  strcat(killpath, "\\KILL.HPG");

  if ((killfile = fopen(killpath,"rt")) == NULL)
  {
    goto NoKill;
  }
  while (fgets(header,1024,message_file) != NULL)
  {
    strlwr(header);
    if (strnicmp(header,"From:",5) == 0)
    {
      flag++;
      while (fgets(killstr,128,killfile) != NULL)
      {
        strlwr(killstr);
        killstr[strlen(killstr) - 1] = '\0';
        if (strstr(header,killstr) != NULL)
        {
          goto Kill;
        }
      }
      fseek(killfile, 0L, SEEK_SET);
    }
    if (strnicmp(header,"Subject:",8) == 0)
    {
      while (fgets(killstr,128,killfile) != NULL)
      {
        strlwr(killstr);
        killstr[strlen(killstr) - 1] = '\0';
        if (strstr(header,killstr) != NULL)
        {
          goto Kill;
        }
      }
      fseek(killfile, 0L, SEEK_SET);
    }
    if (flag == 2) break;
  }

NoKill:
  fseek(message_file, 0L, SEEK_SET);
  if (killfile) fclose(killfile);
  return 1;
Kill:
  fseek(message_file, 0L, SEEK_SET);
  if (killfile) fclose(killfile);
  return 0;
}

/*
Function: clean_zero_byte_file()
Purpose : Remove zero byte *.MSG and *.IDX files.
Return  : N/A
*/

void clean_zero_byte_file()
{
	char fname[MAXPATH];

  strcpy(fname,temp_path);
	strcat(fname,"\\");
	strcat(fname,batch_prefix);
	strcat(fname,".");
	strcat(fname,BATCH_EXT);
	unlink(fname);
  strcpy(fname,temp_path);
	strcat(fname,"\\");
	strcat(fname,batch_prefix);
	strcat(fname,".");
  strcat(fname,INDEX_EXT);
	unlink(fname);

  return;
}

