/*===========================================================================
SOLAR slrxfer v0.94 :: Module slrxfer.c

This source code has been placed into the public domain.

History:  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
04-05-93 KJH  Started history.
              Changed error handling scheme.
              Added forced mode operations.
04-17-93 KJH  Added option to use Waffle's JOIN file.
04-20-93 KJH  Change so aborted download transfers get copied to user's
              directory and subscription file gets updated.
05-23-93 KJH  Modified logging
06-12-93 KJH  Added status LOCALREPLY.
07-08-93 KJH  Simple version number upgrade. No changes.
07-15-94 KJH  Changed all printf() to fprintf()
===========================================================================*/

/* Header Files */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dir.h>
#include <process.h>
#include <time.h>
#include <spawno.h>                     /* A product of Ralph Brown */

/* Local Definations */
#define VERSION              "0.94"     /* Version of SLRXFER */
#define SOLAR_JOIN     "NEWSRC.HPG"
#define SOLAR_JOIN_TMP "NEWSRC.TMP"
#define SOLAR_STATUS   "STATUS.TMP"
#define OPTIONS_FILE  "OPTIONS.HPG"
#define MAILBOX_FILE    "MAILBOX.F"
#define MAILBOX_IDX     "MAILBOX.I"
#define YES                    0
#define NO                     1

/* Local Data */
char temp_path[MAXPATH];          /* Path to temporary directory      */
char user_path[MAXPATH];          /* Path to user's directory         */
char log_path[MAXPATH];           /* Full path to Solar log file      */
char config_path[MAXPATH];        /* Full path to Solar config file   */
char static_path[MAXPATH];        /* Full path to Waffle static file  */
char solar_path[MAXPATH];         /* Full path to Solar directory     */

char uucp_name[10];               /* Host system UUCP name            */
char protocol[10];                /* Transfer protocol                */

enum Helldiver_Types { u, M, m, n, C, c };  /* Supported types */

int delete_mail;                  /* Delete mailbox after transfer    */
int send_mail;                    /* Was mail sent in packet?         */
int use_wafjoin;                  /* Use Waffle's JOIN file           */

char force_mode[10];              /* Force past status check          */
char program[MAXPATH];
char options[80];
char logbuf[60];
char _slrerr[80];

/* Local Functions */
int load_static(char username[10]);
int load_config();
int user_defaults();
int status();
int xfer_packet(char *function, char username[10]);
int load_transfer_parms(char *function);
int logit();
int set_temp_dir();
int write_status(char *status);
long get_packet_size();
void clean_up();
void update_join();
void usage();
int  copy_packet();
char *extract_parm(char string[128], char delimiter);
int  convert_to_number(char *argument);

/* External Functions */
extern int getopt(int argc, char *argv[], const char *optionS);

int main(int argc, char *argv[])
{
	extern char *optarg;
	unsigned char x;
  const char *optionS = "u:U:c:C:Ff:";
	char username[10];
	long packet_size;
  clock_t end_ticks, start_ticks;
  long   cps;

  strcpy(temp_path,"NONE");
  strcpy(user_path,"NONE");
  strcpy(log_path,"NONE");
  strcpy(uucp_name,"NONE");
  strcpy(config_path,"NONE");
  strcpy(static_path,"NONE");
  strcpy(protocol,"NONE");
  strcpy(force_mode,"NONE");

  delete_mail   = NO;
  send_mail     = NO;
  use_wafjoin   = NO;

  while ((x = getopt(argc, argv, optionS)) != '\xFF')
	{
    switch (toupper(x)) {
      case '?'  : usage();
                  goto GoodExit;
      case 'U'  : if ((strcmp(user_path,"NONE")) == 0)
                  {
										strcpy(username,optarg);
                    if (load_static(username) != 0) goto ErrorExit;
                    break;
                  }
                  else
                  {
                    strcpy(_slrerr,"cannot specify username twice on command line");
                    goto ErrorExit;
                  }
      case 'C'  : if ((strcmp(config_path,"NONE")) == 0)
                  {
                    strcpy(config_path,optarg);
                    break;
                  }
                  else
                  {
                    strcpy(_slrerr,"cannot specify two configuration files on command line");
                    goto ErrorExit;
                  }
      case 'F'  : strcpy(force_mode,optarg);
                  break;
    }
  }

  if (strcmp(user_path,"NONE") == 0)
	{
    usage();
    goto GoodExit;
	}
  if (load_config() != 0) goto ErrorExit;
  user_defaults();

  switch (status()) {
    /* ABORT code = 0 */
    case 0  : write_status("ABORT");
              break;
    /* TRANSFER code = 1 */
    case 1  : start_ticks = clock();
              if (xfer_packet("SEND",username) == 0)
              {
                end_ticks = clock();
                packet_size = get_packet_size();
                if ((end_ticks - start_ticks) > 0)
                {
                  cps = packet_size / ((end_ticks - start_ticks) / CLK_TCK);
                }
                else
                {
                  cps = packet_size;
                }
                sprintf(logbuf," %-8s | XS  %4.1f min %4lu cps  %8lu bytes",username,((end_ticks - start_ticks) / CLK_TCK) / 60,cps,packet_size);
                logit();
                update_join();
                clean_up();
                write_status("ABORT");
              }
              else
              {
                if (copy_packet() == 0) update_join();
                clean_up();
                write_status("ABORT");
                goto ErrorExit;
              }
              break;
    /* LOCAL code = 2 */
    case 2  : clean_up();
              write_status("ABORT");
              break;
    /* No status file, SOLAR has not run or was exited without status */
    case 3  : write_status("ABORT");
    /* Invalid status code found */
    case 4  : write_status("ABORT");
              goto ErrorExit;
    /* REPLY code = 5 */
    case 5  : if (set_temp_dir() != 0)
              {
                write_status("ABORT");
                goto ErrorExit;
              }
              start_ticks = clock();
              if (xfer_packet("RECEIVE",username) == 0)
              {
                packet_size = get_packet_size();
                if (packet_size > 0L)
                {
                  end_ticks = clock();
                  if ((end_ticks - start_ticks) > 0)
                  {
                    cps = packet_size / ((end_ticks - start_ticks) / CLK_TCK);
                  }
                  else
                  {
                    cps = packet_size;
                  }
                  sprintf(logbuf," %-8s | XR  %4.1f min %4lu cps  %8lu bytes",username,((end_ticks - start_ticks) / CLK_TCK)/60,cps,packet_size);
                  logit();
                  write_status("PROCESS");
                }
                else
                {
									write_status("ABORT");
								}
              }
              else
              {
                write_status("ABORT");
              }
              break;
    /* Code 6, LOCALREPLY */
    case 6  : write_status("PROCESS");
              break;
  }
GoodExit:
  return 0;
ErrorExit:
  fprintf(stderr,"<slrxfer> fatal: %s\n",_slrerr);
  return 1;
}

/*
Function: load_static()
Purpose : Load uucpname, user, and temporary parameters from Waffle's
          static file. Load temporary only if it is not already set.
          This allows the configuration file SOLARWORK parameter
          override to work if the configuration file is processed before
					the static file.
Return  : 0 on success, non-zero on error and set _slrerr
*/

int load_static(char username[10])
{
  FILE *static_file = NULL;

	char buffer[128];
	char delimiter = ':';
  char *path = NULL;

	if ((path = getenv("WAFFLE")) == NULL)
	{
    strcpy(_slrerr,"environment variable WAFFLE not set");
    goto ErrorExit;
  }
	strcpy(static_path,path);
  if ((static_file = fopen(static_path,"rt")) == NULL)
  {
    sprintf(_slrerr,"error opening static file %s",static_path);
    goto ErrorExit;
  }

  while (fgets(buffer,128,static_file) != NULL)
	{
		if (strnicmp(buffer,"UUCPNAME",8) == 0)
			strcpy(uucp_name,extract_parm(buffer,delimiter));
		if (strnicmp(buffer,"USER",4) == 0)
		{
			strcpy(user_path,extract_parm(buffer,delimiter));
			strcat(user_path, "\\");
			strcat(user_path, username);
		}
		if (strnicmp(buffer,"WAFFLE",6) == 0)
		{
      strcpy(log_path,extract_parm(buffer,delimiter));
      strcat(log_path,"\\admin\\solar");
		}
		if (strnicmp(buffer,"TEMPORARY",9) == 0)
		{
			if (strcmp(temp_path,"NONE") == 0)
				strcpy(temp_path,extract_parm(buffer,delimiter));
		}
	}
	fclose(static_file);

	if (strcmp(uucp_name,"NONE") == 0)
	{
    strcpy(_slrerr,"uucpname: parameter not found in static file");
    goto ErrorExit;
  }
	if (strcmp(temp_path,"NONE") == 0)
	{
    strcpy(_slrerr,"temporary: parameter not found in static file");
    goto ErrorExit;
	}
	if (strcmp(user_path,"NONE") == 0)
	{
    strcpy(_slrerr,"user: parameter not found in static file");
    goto ErrorExit;
	}
GoodExit:
  return 0;
ErrorExit:
  if (static_file) fclose(static_file);
  return 1;
}

/*
Function: extract_parm()
Purpose : Extracts one word found after a delimiter in a string.
Return  : Returns the word found.
*/

char *extract_parm(char string[128], char delimiter)
{
	int  idx  = 0;
	int  idx2 = 0;
	char path[MAXPATH];

	while (string[idx++] != delimiter);
	while (string[idx] == ' ') idx++;
	while ((string[idx] != ' ') && (string[idx] != '\0') && (string[idx] != '\n'))
	{
		if (string[idx] == '/')
		{
			path[idx2++] = '\\';
		}
		else
		{
			path[idx2++] = string[idx];
		}
		idx++;
	}
	path[idx2] = '\0';
	return path;
}

/*
Function: load_config()
Purpose : Load Solar's configuration file.
Return  : 0 on success, non-zero on error and set _slrerr.
*/

int load_config()
{
  FILE *config_file = NULL;
	char buf[128];
	char delimiter = '=';
  char *path = NULL;

  if (strcmp(config_path,"NONE") == 0)
  {
    if ((path = getenv("SOLAR")) == NULL)
    {
      strcpy(_slrerr,"environment variable SOLAR not set");
      goto ErrorExit;
    }
    strcpy(config_path,path);
  }

  if ((config_file = fopen(config_path,"rt")) == NULL)
	{
    sprintf(_slrerr,"error opening config file %s",config_path);
    goto ErrorExit;
  }
	while (fgets(buf,128,config_file) != NULL)
	{
    if (strnicmp(buf,"SOLARWORK",9) == 0)
		{
			strcpy(temp_path,extract_parm(buf,delimiter));
		}
    if (strnicmp(buf,"SOLARPATH",9) == 0)
		{
      strcpy(solar_path,extract_parm(buf,delimiter));
		}
    if (strnicmp(buf,"LOGFILE",7) == 0)
		{
			strcpy(log_path,extract_parm(buf,delimiter));
		}
    if (strnicmp(buf,"subscribe",9) == 0)
		{
      use_wafjoin = convert_to_number(extract_parm(buf,delimiter));
		}
    if ((strnicmp(buf,"receive",8)) == 0)
    {
      get_protocol_type(buf);
    }
  }
	fclose(config_file);

  if (strcmp(solar_path,"NONE") == 0)
  {
    strcpy(_slrerr,"solarpath: not found in config file");
    goto ErrorExit;
  }
GoodExit:
  return 0;
ErrorExit:
  if (config_file) fclose(config_file);
  return 1;
}

/*
Function: user_defaults()
Purpose : Check for a user config file and load values if present.
Return  : zero on success, non-zero on error and set _slrerr.
*/

int user_defaults()
{
	FILE *user_config = NULL;

	char buf[128];
	char path[MAXPATH];
	char delimiter = '=';

	strcpy(path,user_path);
	strcat(path,"\\");
	strcat(path,OPTIONS_FILE);

	if ((user_config = fopen(path,"rt")) == NULL)
	{
    goto GoodExit;    /* For some reason, user has no defaults */
	}
	while (fgets(buf,128,user_config) != NULL)
	{
    if ((strnicmp(buf,"delete-mail",11)) == 0)
		{
      delete_mail = convert_to_number(extract_parm(buf,delimiter));
		}
    if ((strnicmp(buf,"send-mail",9)) == 0)
		{
      send_mail = convert_to_number(extract_parm(buf,delimiter));
		}
    if ((strnicmp(buf,"protocol",8)) == 0)
    {
      strcpy(protocol,extract_parm(buf,delimiter));
    }
    if (strnicmp(buf,"subscribe",9) == 0)
		{
      use_wafjoin = convert_to_number(extract_parm(buf,delimiter));
		}
  }
	fclose(user_config);
GoodExit:
  return 0;
ErrorExit:
  if (user_config) fclose(user_config);
  return 1;
}

/*
Function: status()
Purpose : Read Solar status file and return status.
Return  : 0 - ABORT
          1 - TRANSFER
          2 - LOCAL
          3 - File not found
          4 - Invalid status
          5 - REPLY
          6 - LOCALREPLY
*/

int status()
{
  FILE *status_file;
  char status_path[MAXPATH];
  char buffer[20];

  /* Test for a command line forced mode and
     branch accordingly. Always do LOCAL after
     LOCALREPLY to avoid getting them confused. */

  if (strcmp(force_mode,"NONE") != 0)
  {
		if (strnicmp(force_mode,"ABORT",5) == 0) goto ExitAbort;
		if (strnicmp(force_mode,"REPLY",5) == 0) goto ExitReply;
		if (strnicmp(force_mode,"TRANSFER",8) == 0) goto ExitTransfer;
    if (strnicmp(force_mode,"LOCALREPLY",10) == 0) goto LocalReplyExit;
    if (strnicmp(force_mode,"LOCAL",5) == 0) goto ExitLocal;
    goto ExitInvalid;
  }

  /* No command line forced mode. Open the
     STATUS.TMP file for read in text mode. */

  strcpy(status_path,temp_path);
  strcat(status_path,"\\");
  strcat(status_path,SOLAR_STATUS);
  if ((status_file = fopen(status_path,"rt")) == NULL)
  {
    sprintf(_slrerr,"status file %s not found", status_path);
    goto ExitFNF;
  }

  /* Read the first/only record in the file
     and close the file */

  fgets(buffer,20,status_file);
  fclose(status_file);
  unlink(status_path);

  /* Check the status code from the STATUS.TMP
     file and branch accordingly. Again, make
     sure LOCAL is after LOCALREPLY. */

  if (strnicmp(buffer,"ABORT",5) == 0) goto ExitAbort;
  if (strnicmp(buffer,"TRANSFER",8) == 0) goto ExitTransfer;
  if (strnicmp(buffer,"REPLY",5) == 0) goto ExitReply;
	if (strnicmp(buffer,"LOCALREPLY",10) == 0) goto LocalReplyExit;
  if (strnicmp(buffer,"LOCAL",5) == 0) goto ExitLocal;

/* Return based on the status. */

ExitBadFile:
	sprintf(_slrerr,"status mode missing or invalid");
	return 4;
ExitInvalid:
	sprintf(_slrerr,"invalid force mode: %s",force_mode);
	return 4;
ExitAbort:
  return 0;
ExitTransfer:
  return 1;
ExitLocal:
  return 2;
ExitFNF:
  return 3;
ExitReply:
  return 5;
LocalReplyExit:
  return 6;
}

/*
Function: xfer_packet()
Purpose : Call transfer protocol to send/receive packet.
Return  : zero on success, non-zero on error and set _slrerr.
*/

int xfer_packet(char *function, char username[10])
{
  char packet_name[MAXPATH];

  if (load_transfer_parms(function) != 0) goto ErrorExit;

	init_SPAWNO(".",SWAP_DISK);

  if (stricmp(function,"SEND") == 0)
	{
    strcpy(packet_name,temp_path);
    strcat(packet_name,"\\");
    strcat(packet_name,uucp_name);
		strcat(packet_name,".*");

		if (spawnlp(P_WAIT,program,program,options,packet_name,NULL) != 0)
		{
      sprintf(_slrerr,"%s transfer unsuccessful",protocol);
      goto ErrorExit;
		}
	}
	else
	{
    if (spawnlp(P_WAIT,program,program,options,NULL) != 0)
		{
      sprintf(_slrerr,"%s transfer unsuccessful!",program);
      goto ErrorExit;
    }
	}
GoodExit:
  return 0;
ErrorExit:
	return 1;
}

/*
Function: update_join()
Purpose : Delete old join file and rename temp. join.
Return  : N/A
*/

void update_join()
{
  char old_path[MAXPATH];
	char new_path[MAXPATH];

  strcpy(old_path,user_path);
  strcat(old_path,"\\");
  strcat(old_path,SOLAR_JOIN);
  strcpy(new_path,user_path);
  strcat(new_path,"\\");
  strcat(new_path,SOLAR_JOIN_TMP);
  unlink(old_path);
  rename(new_path,old_path);
	if (use_wafjoin == YES)
  {
    if (newsrc2join() != 0)
    {
      fprintf(stderr,"<slrxfer> error: %s\n",_slrerr);
    }
  }
  if ((delete_mail == YES) && (send_mail == YES))
  {
    strcpy(old_path, user_path);
    strcat(old_path, "\\");
    strcat(old_path, MAILBOX_FILE);
    unlink(old_path);
    strcpy(old_path, user_path);
    strcat(old_path, "\\");
    strcat(old_path, MAILBOX_IDX);
    unlink(old_path);
  }
  return;
}

/*
Function: clean_up()
Purpose : Remove temporary directory and NEWSRC.TMP file
Return  : N/A
*/

void clean_up()
{
	struct ffblk ffblk;

  char work_path[MAXPATH];
  char del_path[MAXPATH];

  int done = 0;

  strcpy(work_path, temp_path);
  strcat(work_path, "\\*.*");
	done = findfirst(work_path,&ffblk,0);
	while (!done)
	{
    strcpy(del_path, temp_path);
		strcat(del_path, "\\");
		strcat(del_path, ffblk.ff_name);
		unlink(del_path);
		done = findnext(&ffblk);
	}

  strcpy(work_path, user_path);
  strcat(work_path, "\\");
  strcat(work_path, SOLAR_JOIN_TMP);
  unlink(work_path);

  return;
}

/*
Function: get_protocol_type()
Purpose : Extract protocol type from string and store in global
          variable protocol.
Return  : 0 on success, non-zero on error and set _slrerr.
*/

int get_protocol_type(char buf[128])
{
  char *p = NULL;

  if ((p = strtok(buf,"-")) == NULL)
  {
    strcpy(_slrerr,"error in configuration file receive: parameter");
    goto ErrorExit;
  }
  if ((p = strtok(NULL,"=")) == NULL)
  {
    strcpy(_slrerr,"error in configuration file receive: parameter");
    goto ErrorExit;
  }
  strcpy(protocol,p);

GoodExit:
  return 0;
ErrorExit:
  return 1;
}

/*
Function: load_transfer_parms()
Purpose : Load the appropriate transfer parameters from the Solar
          configuration file.
Return  : zero on success, non-zero on error and set _slrerr.
*/

int load_transfer_parms(char *function)
{
	FILE *config_file = fopen(config_path,"rt");

	int idx  = 0;

	char buf[128];
	char search_string[20];
	char *p = NULL;

  if (!config_file)
	{
    sprintf(_slrerr,"error opening config file %s",config_path);
    goto ErrorExit;
  }

	strcpy(program,"NONE");
	options[0] = '\0';
	if (stricmp(function,"SEND") == 0)
	{
		strcpy(search_string,"send-");
	}
	if (stricmp(function,"RECEIVE") == 0)
	{
		strcpy(search_string,"receive-");
	}
  strcat(search_string,protocol);

	while (fgets(buf,128,config_file) != NULL)
	{
		if ((strnicmp(buf,"send",4) == 0) || (strnicmp(buf,"receive",7) == 0))
		{
			if ((p = strtok(buf,"=")) == NULL)
			{
        strcpy(_slrerr,"error in configuration file protocol parameter");
        goto ErrorExit;
			}
			idx = 0;
			while (p[idx] != '\n') /* strip the LF and add a null character */
      {
				p[idx] = p[idx++];
      }
			p[idx] = '\0';
      if (stricmp(p,search_string) == 0)
			{
				if ((p = strtok(NULL,"=")) == NULL)
        {
          strcpy(_slrerr,"error in configuration file protocol parameter");
          goto ErrorExit;
        }
				idx = 0;
				while (p[idx] != '\n') /* copy back into buf, drop LF */
        {
					buf[idx] = p[idx++];
        }
				buf[idx] = '\0';
        if ((p = strtok(buf," ")) != NULL)
				{
					idx = 0;
					while (p[idx] != '\n') /* strip the LF and add a null character */
          {
						program[idx] = p[idx++];
          }
					program[idx] = '\0';
          if ((p = strtok(NULL,"=")) != NULL)
          {
						idx = 0;
						while (p[idx] != '\n') /* strip the LF and add a null character */
            {
							options[idx] = p[idx++];
						}
						options[idx] = '\0';
          }
          else
          {
            options[0] = '\0';
          }
        }
        else
        {
          strcpy(_slrerr,"error in configuration file protocol parameter");
          goto ErrorExit;
        }
				break;
			}
    }
  }
  fclose(config_file);

  if (strcmp(program,"NONE") == 0)
	{
    strcpy(_slrerr,"selected protocol not configured");
    goto ErrorExit;
  }
GoodExit:
  return 0;
ErrorExit:
  if (config_file) fclose(config_file);
  return 1;
}

/*
Function: logit()
Purpose : Write contents of log buffer to log file with time.
Return  : 0 on success, non-zero on error.
*/

int logit()
{
  FILE *log_file = fopen(log_path,"at");
  time_t t;

  char date_time[15];
  char temp[26];

  if (!log_file)
  {
    return -1;
  }
  t = time(NULL);
  sprintf(temp,"%s",ctime(&t));

  date_time[0]  = temp[8];
  date_time[1]  = temp[9];
  date_time[2]  = '-';
  date_time[3]  = temp[4];
  date_time[4]  = temp[5];
  date_time[5]  = temp[6];
  date_time[6]  = '-';
  date_time[7]  = temp[22];
  date_time[8]  = temp[23];
  date_time[9]  = ' ';
  date_time[10] = temp[11];
  date_time[11] = temp[12];
  date_time[12] = temp[13];
  date_time[13] = temp[14];
  date_time[14] = temp[15];
  date_time[15] = '\0';

  fprintf(log_file,"%s | %s\n",date_time,logbuf);
  fclose(log_file);
  return 0;
}

/*
Function: write_status()
Purpose : Output a string to the Solar status file.
Return  : 0 on success, non-zero on error.
*/

int write_status(char *status)
{
  char path[MAXPATH];
  FILE *status_file;

  strcpy(path,temp_path);
  strcat(path,"\\");
  strcat(path,SOLAR_STATUS);
  if ((status_file = fopen(path,"wt")) == NULL)
  {
    return 1;
  }
  fprintf(status_file,"%s\n",status);
  fclose(status_file);
  return 0;
}

/*
Function: set_temp_dir()
Purpose : Change disk and directory to temp directory.
Return  : 0 on success, non-zero on error and set _slrerr.
*/

int set_temp_dir()
{
	char disk_drive = temp_path[0];
	char new_path[MAXPATH];
	int  disknum = 0;
	int  x = 0;
  struct ffblk ffblk;
  int done;

	disk_drive = toupper(disk_drive);
	disknum = disk_drive - 65;
	if ((disknum >= 0) && (disknum <=25))
	{
		setdisk(disknum);
	}
	else
	{
    sprintf(_slrerr,"invalid disk for temp directory: %c",disk_drive);
    goto ErrorExit;
	}

  new_path[0] = '\0';
  for (x = 2; x < strlen(temp_path); x++)
	{
    new_path[x - 2] = temp_path[x];
	}
  new_path[x - 2] = '\0';
  if (chdir(new_path) != 0)
  {
    sprintf(_slrerr,"error changing to directory %s",new_path);
    goto ErrorExit;
  }
  done = findfirst("*.*",&ffblk,0);
	while (!done)
	{
    unlink(ffblk.ff_name);
		done = findnext(&ffblk);
	}
GoodExit:
  return 0;
ErrorExit:
  return 1;
}

/*
Function: get_packet_size()
Purpose : Find size of packet in temp directory. Assumes the
          packet is the only file in the directory.
Return  : The size of the packet in bytes.
*/

long get_packet_size()
{
  char path[MAXPATH];
  struct ffblk ffblk;

  strcpy(path,temp_path);
  strcat(path,"\\*.*");

  if (findfirst(path,&ffblk,0) != 0)
	{
		return 0L;
	}
  return ffblk.ff_fsize;
}

/*
Function: void usage()
Purpose : Display slrxfer command line usage.
Return : N/A
*/

void usage()
{
	fprintf(stdout,"\nSOLAR v%s <> slrxfer\n",VERSION);
	fprintf(stdout,"\nUsage: slrxfer /u username [/c d:/path/config.file] [/f MODE]\n");
}

/*
Function: int copy_packet();
Purpose : After an unsuccessfull download transfer, copy packet
          to user's directory so download may be continued another time.
          This assumes the packet is the only file in the temp directory.
Return  : 0 on success, non-zero on error.
*/

int copy_packet()
{
	struct ffblk ffblk;
	FILE *orig_file = NULL;
	FILE *dest_file = NULL;
	char orig_path[MAXPATH];
	char dest_path[MAXPATH];
	unsigned char movebuf;

  if (findfirst("*.*",&ffblk,0) == 0)
	{
		strcpy(orig_path,temp_path);
		strcat(orig_path,"\\");
		strcat(orig_path,ffblk.ff_name);
		strcpy(dest_path,user_path);
		strcat(dest_path,"\\");
		strcat(dest_path,ffblk.ff_name);

    fprintf(stdout,"moving %s to %s\n",orig_path,dest_path);

		if ((orig_file = fopen(orig_path,"rb")) == NULL) goto ErrorExit;
		if ((dest_file = fopen(dest_path,"wb")) == NULL) goto ErrorExit;
    movebuf = getc(orig_file);
    while (feof(orig_file) == 0)
		{
			putc(movebuf,dest_file);
      movebuf = getc(orig_file);
    }
	}
ExitFunct:
	if (orig_file) fclose(orig_file);
	if (dest_file) fclose(dest_file);
	return 0;
ErrorExit:
	if (orig_file) fclose(orig_file);
	if (dest_file) fclose(dest_file);
	return 1;
}

/*
Function: int convert_to_number(char *argument)
Purpose : Convert alpha strings to internal numberical representations.
Return  : The number corresponding to the alpha string.
*/

int convert_to_number(char *argument)
{
  /* Helldiver Packet Format Types */
  if (strcmp(argument,"u") == 0)
    return 0;
  if (strcmp(argument,"M") == 0)
    return 1;
  if (strcmp(argument,"m") == 0)
    return 2;
  if (strcmp(argument,"n") == 0)
    return 3;
  if (strcmp(argument,"C") == 0)
    return 4;
  if (strcmp(argument,"c") == 0)
    return 5;

  /* Boolean Values */
  if (stricmp(argument,"YES") == 0)
    return 0;
  if (stricmp(argument,"NO") == 0)
    return 1;
  fprintf(stderr,"<warning> obsolete configuration value accepted: %s\n",argument);
  return atoi(argument);
}




