/*===========================================================================
SOLAR slrreply v0.94 :: Module slrreply.c

This source code has been placed into the public domain.

History:  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
03-30-93 KJH  Started history.
              Changed tons of code to accomidate the new extract.c module.
              Added extensive error checking system to packet handling.
04-18-93 KJH  Removed call to clean_up() if status = ABORT, so that local
							packets do not get erased.
06-29-93 KJH  Changed YES to 0 and NO to 1 to match solar.exe
							Added code to load 'subscribe' parm from OPTIONS.HPG for
							LIST file processing.
09-23-93 KJH  Added optional Paths: header to news messages.
07-15-94 KJH  Changed all printf() to fprintf()
===========================================================================*/

/* Header Files */
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <dir.h>
#include <time.h>
#include <values.h>

/* Local Definitions */
#define VERSION                  "0.94"
#define SOLAR_STATUS       "STATUS.TMP"
#define NO                           1
#define YES                          0

/* Global Data */
char temp_path[MAXPATH];        /* Full path to temporary directory.  */
char user_path[MAXPATH];        /* Full path to user directory.       */
char log_path[MAXPATH];         /* Full path to log file.             */
char static_path[MAXPATH];      /* Full path to Waffle's static file  */
char system_path[MAXPATH];      /* Full path to ~/system directory    */
char config_path[MAXPATH];      /* Full path to configuration file    */
char lost_path[MAXPATH];        /* Full path to lost+found directory  */
char solar_path[MAXPATH];       /* Full path to main Solar directory  */
char logbuf[60];                /* Buffer for Solar log file          */
char _slrerr[80];               /* Error handling message buffer      */
long _maxmailsize;              /* Maximum number of bytes for mail   */
long _maxnewssize;              /* Maximum number of bytes for news   */
char backbone[40];              /* Backbone for moderated posts       */
char domain_name[40];           /* Host system registered domain      */
char uucp_name[10];             /* Host system registered UUCP name   */
char username[10];              /* User's username                    */
char real_name[20];             /* User's real name                   */
char organization[128];         /* Host system organization           */
int  force_mode = NO;           /* Forced reply mode                  */
int  use_wafjoin;               /* Boolean: use Waffle's join file    */

/* External Functions */
extern int getopt(int argc, char *argv[], const char *optionS);
extern int process_packet();

/* Local Functions */
char *extract_parm(char string[128], char delimiter);
int  status(int force_mode);
int  load_static();
int  load_config();
int  user_defaults();
int  logit();
int  setconfig();
int  convert_to_number(char *argument);
void clean_up();
void usage();

/*
Function: int main(int argc, char *argv[])
Purpose : The Solar System slreply Helldiver reply packet processor
          for DOS systems.
Return  : 0 on success, non-zero on error.
*/

int main(int argc, char *argv[])
{
	extern char *optarg;
  extern int  optind;
  unsigned char x;
  const char *optionS = "u:U:c:C:Rr:Ff";
	int message_count = 0;

	strcpy(temp_path,"NONE");
  strcpy(user_path,"NONE");
  strcpy(log_path,"NONE");
	strcpy(domain_name,"NONE");
	strcpy(uucp_name,"NONE");
	strcpy(config_path,"NONE");
	strcpy(static_path,"NONE");
  strcpy(system_path,"NONE");
  strcpy(solar_path,"NONE");
	strcpy(lost_path,"NONE");
  strcpy(backbone,"NONE");
  strcpy(username,"NONE");
  strcpy(real_name,"NONE");
  strcpy(organization,"NONE");
  _maxnewssize = MAXLONG;
	_maxmailsize = MAXLONG;

	while ((x = getopt(argc, argv, optionS)) != '\xFF')
	{
		switch (toupper(x)) {
			case '?'  : usage();
									goto GoodExit;
			case 'U'  : if ((strcmp(static_path,"NONE")) == 0)
									{
										strcpy(username,optarg);
										if (load_static() != 0) goto ErrorExit;
										break;
									}
									else
									{
										strcpy(_slrerr,"cannot specify username twice on command line");
										goto ErrorExit;
									}
			case 'C'  : if ((strcmp(config_path,"NONE")) == 0)
									{
                    strcpy(config_path,optarg);
                    break;
                  }
                  else
                  {
                    strcpy(_slrerr,"cannot specify two configuration files on command line");
                    goto ErrorExit;
                  }
      case 'F'  : force_mode = YES;
                  break;
      case 'R'  : strcpy(real_name,optarg);
                  while (argc > optind)
									{
                    strcat(real_name," ");
                    strcat(real_name,argv[optind++]);
                  }
                  break;
    }
	}
  /* The /u parameter must appear on the command line. */
  if (strcmp(username,"NONE") == 0)
	{
    usage();
    goto GoodExit;
	}

  /* If no configuration file was given on command line, check environment */
  /* and then load the configuration file parameters.                      */

  if (strcmp(config_path,"NONE") == 0)
  {
    if (setconfig() != 0) goto ErrorExit;
	}
  if (load_config() != 0) goto ErrorExit;

  /* Build path to user directory and load user default file */

  strcat(user_path,"\\");
  strcat(user_path,username);
  if (user_defaults() != 0) goto ErrorExit;

  /* Check for status */
  switch (status(force_mode)) {
		/* ABORT code = 0 */
    case 0  : break;
		/* PROCESS code = 1 */
    case 1  : if ((message_count = process_packet()) > 0)
                clean_up();
							break;
    /* Status file not found = 2 */
    case 2  : goto ErrorExit;
    /* Invalid status code */
    case 3  : goto ErrorExit;
  }
GoodExit:
	return 0;
ErrorExit:
  fprintf(stderr,"slrreply: %s\n",_slrerr);
  return 1;
}

/*
Function: load_static()
Purpose : Load parameters from Waffle's static file.
Return  : 0 on success, non-zero on error and set _slrerr.
*/

int load_static()
{
	char *full_line(char line[128]);
  FILE *static_file = NULL;

	char buffer[128];
	char delimiter = ':';
  char *path = NULL;

	if ((path = getenv("WAFFLE")) == NULL)
	{
    strcpy(_slrerr,"environment variable WAFFLE not defined");
    goto ErrorExit;
	}
	strcpy(static_path,path);
	if ((static_file = fopen(static_path,"rt")) == NULL)
  {
    sprintf(_slrerr,"cannot open Waffle static file %s",static_path);
    goto ErrorExit;
  }

  while (fgets(buffer,128,static_file) != NULL)
	{
    if (strnicmp(buffer,"USER",4) == 0)
      strcpy(user_path,extract_parm(buffer,delimiter));
    if (strnicmp(buffer,"NODE",4) == 0)
      strcpy(domain_name,extract_parm(buffer,delimiter));
    if (strnicmp(buffer,"UUCPNAME",8) == 0)
      strcpy(uucp_name,extract_parm(buffer,delimiter));
    if (strnicmp(buffer,"organ",5) == 0)
      strcpy(organization,full_line(buffer));
    if (strnicmp(buffer,"backbone",8) == 0)
			strcpy(backbone,extract_parm(buffer,delimiter));
    if (strnicmp(buffer,"WAFFLE",6) == 0)
		{
      if (strcmp(log_path,"NONE") == 0)
      {
        strcpy(log_path,extract_parm(buffer,delimiter));
        strcat(log_path,"\\admin\\solar");
      }
      strcpy(system_path,extract_parm(buffer,delimiter));
      strcat(system_path,"\\system");
		}
		if (strnicmp(buffer,"TEMPORARY",9) == 0)
		{
			if (strcmp(temp_path,"NONE") == 0)
				strcpy(temp_path,extract_parm(buffer,delimiter));
		}
	}
	fclose(static_file);

	if (strcmp(domain_name,"NONE") == 0)
	{
    strcpy(_slrerr,"node: parameter not found in static file");
    goto ErrorExit;
	}
  if (strcmp(uucp_name,"NONE") == 0)
	{
    strcpy(_slrerr,"uucpname: parameter not found in static file");
		goto ErrorExit;
	}
  if (strcmp(temp_path,"NONE") == 0)
	{
    strcpy(_slrerr,"temporary: parameter not found in static file");
    goto ErrorExit;
	}
  if (strcmp(log_path,"NONE") == 0)
	{
    strcpy(_slrerr,"waffle: parameter not found in static file");
    goto ErrorExit;
	}
GoodExit:
	return 0;
ErrorExit:
  return 1;
}

/*
Function: extract_parm()
Purpose : Extracts one word found after a delimiter in a string.
Return  : Returns the word found.
*/

char *extract_parm(char string[128], char delimiter)
{
	int  idx  = 0;
	int  idx2 = 0;
  static char path[MAXPATH];

	while (string[idx++] != delimiter);
	while (string[idx] == ' ') idx++;
	while ((string[idx] != ' ') && (string[idx] != '\0') && (string[idx] != '\n'))
	{
		if (string[idx] == '/')
		{
			path[idx2++] = '\\';
		}
		else
		{
			path[idx2++] = string[idx];
		}
		idx++;
	}
	path[idx2] = '\0';
	return path;
}

/*
Function: char full_line(char line[128])
Purpose : Extracts entire line found after a space.
Return  : Returns the line found.
*/

char *full_line(char line[128])
{
	int  idx  = 0;
	int  idx2 = 0;
  static char path[MAXPATH];

  while (line[idx++] != ':');
  while (line[idx] == ' ') idx++;
	while ((line[idx] != '\0') && (line[idx] != '\n'))
	{
    path[idx2++] = line[idx];
		idx++;
	}
	path[idx2] = '\0';
	return path;
}

/*
Function: load_config()
Purpose : Load Solar's configuration file.
Return  : 0 on success, non-zero on error and set _slrerr.
*/

int load_config()
{
	FILE *config_file = fopen(config_path,"rt");

	char buf[128];
	char delimiter = '=';

	if (config_file == NULL)
	{
    sprintf(_slrerr,"cannot open Solar configuration file %s",config_path);
    goto ErrorExit;
	}
	while (fgets(buf,128,config_file) != NULL)
	{
		if (strnicmp(buf,"LOST+FOUND",10) == 0)
			strcpy(lost_path,extract_parm(buf,delimiter));
		if (strnicmp(buf,"SOLARPATH",9) == 0)
			strcpy(solar_path,extract_parm(buf,delimiter));
    if (strnicmp(buf,"MAX-MAIL-SIZE",13) == 0)
			_maxmailsize = atol(extract_parm(buf,delimiter));
    if (strnicmp(buf,"MAX-NEWS-SIZE",13) == 0)
			_maxnewssize = atol(extract_parm(buf,delimiter));
    if ((strnicmp(buf,"subscribe",9)) == 0)
      use_wafjoin = convert_to_number(extract_parm(buf,delimiter));

		/* Override static settings from config file */
    if (strnicmp(buf,"SOLARWORK",9) == 0)
			strcpy(temp_path,extract_parm(buf,delimiter));
		if (strnicmp(buf,"LOGFILE",7) == 0)
			strcpy(log_path,extract_parm(buf,delimiter));
    if (strnicmp(buf,"host-domain",11) == 0)
      strcpy(domain_name,extract_parm(buf,delimiter));
  }
	fclose(config_file);

GoodExit:
  return 0;
ErrorExit:
  return 1;
}

/*
Function: int status(int force_mode)
Purpose : Read Solar status file and return status.
Return  : 0 = ABORT
          1 = PROCESS
          2 = File not found
          3 = invalid code, set _slrerr
*/

int status(int force_mode)
{
  FILE *status_file;
  char status_path[MAXPATH];
  char buffer[20];

  if (force_mode == YES) goto ProcessExit;

  strcpy(status_path,temp_path);
  strcat(status_path,"\\");
  strcat(status_path,SOLAR_STATUS);
  if ((status_file = fopen(status_path,"rt")) == NULL)
  {
    goto FNFExit;
  }

  fgets(buffer,20,status_file);
  fclose(status_file);
  unlink(status_path);

  if (strnicmp(buffer,"ABORT",5) == 0) goto AbortExit;
  if (strnicmp(buffer,"PROCESS",7) == 0) goto ProcessExit;

InvalidExit:
  sprintf(_slrerr,"invalid status code : %s",buffer);
  return 3;
ProcessExit:
  return 1;
FNFExit:
  sprintf(_slrerr,"status file not found");
  return 2;
AbortExit:
  return 0;
}

/*
Function: clean_up()
Purpose : Remove files in temporary directory.
Return  : N/A
*/

void clean_up()
{
	struct ffblk ffblk;

  char work_path[MAXPATH];
  char del_path[MAXPATH];

  int done = 0;

  strcpy(work_path, temp_path);
  strcat(work_path, "\\*.*");
	done = findfirst(work_path,&ffblk,0);
	while (!done)
	{
    strcpy(del_path, temp_path);
		strcat(del_path, "\\");
		strcat(del_path, ffblk.ff_name);
		unlink(del_path);
		done = findnext(&ffblk);
	}
  return;
}

/*
Function: logit()
Purpose : Write contents of log buffer to log file with time.
Return  : 0 on success, non-zero on error.
*/

int logit()
{
  FILE *log_file = fopen(log_path,"at");
  time_t t;

  char date_time[15];
  char temp[26];

  if (!log_file)
  {
    return -1;
  }
  t = time(NULL);
  sprintf(temp,"%s",ctime(&t));

  date_time[0]  = temp[8];
  date_time[1]  = temp[9];
  date_time[2]  = '-';
  date_time[3]  = temp[4];
  date_time[4]  = temp[5];
  date_time[5]  = temp[6];
  date_time[6]  = '-';
  date_time[7]  = temp[22];
  date_time[8]  = temp[23];
  date_time[9]  = ' ';
  date_time[10] = temp[11];
  date_time[11] = temp[12];
  date_time[12] = temp[13];
  date_time[13] = temp[14];
  date_time[14] = temp[15];
  date_time[15] = '\0';

  fprintf(log_file,"%s | %s\n",date_time,logbuf);
  fclose(log_file);
  return 0;
}

/*
Function: void usage()
Purpose : Display command line usage.
Return  : N/A
*/

void usage()
{
	fprintf(stdout,"\nSOLAR v%s <> slrreply \n",VERSION);
  fprintf(stdout,"\nUsage: slrreply /u username [[/c d:/path/config.file] [/f]] [/rUser Name]\n");
  return;
}

/*
Function: int setconfig()
Purpose : Set config_path from environment variable SOLAR.
Return  : 0 on success, non-zero on error and set _slrerr.
*/

int setconfig()
{
  char *path = NULL;

  if ((path = getenv("SOLAR")) == NULL)
	{
    strcpy(_slrerr,"Environment variable SOLAR not defined");
    goto ErrorExit;
	}
  strcpy(config_path,path);

GoodExit:
  return 0;
ErrorExit:
  return 1;
}

/*
Function: int convert_to_number(char *argument)
Purpose : Convert alpha strings to internal numberical representations.
Return  : The number corresponding to the alpha string.
*/

int convert_to_number(char *argument)
{
  /* Helldiver Packet Format Types */
  if (strcmp(argument,"u") == 0)
    return 0;
  if (strcmp(argument,"M") == 0)
    return 1;
  if (strcmp(argument,"m") == 0)
    return 2;
  if (strcmp(argument,"n") == 0)
    return 3;
  if (strcmp(argument,"C") == 0)
    return 4;
  if (strcmp(argument,"c") == 0)
    return 5;

  /* Boolean Values */
  if (stricmp(argument,"YES") == 0)
    return 0;
  if (stricmp(argument,"NO") == 0)
    return 1;
  fprintf(stderr,"<warning> obsolete OPTIONS.HPG value accepted: %s\n",argument);
  return atoi(argument);
}

/*
Function: user_defaults()
Purpose : Check for a user config file and load values if present.
Return  : 0 on success, non-zero if not loaded or error.
*/

int user_defaults()
{
	FILE *user_config = NULL;

  char buf[128];
  char path[MAXPATH];
  char newpath[MAXPATH];
	long temp = 0L;
	char delimiter = '=';

  strcpy(path,user_path);
  strcat(path,"\\OPTIONS.HPG");

  if ((user_config = fopen(path,"rt")) == NULL)
    goto GoodExit; /* Assume there is no user config file */

  while (fgets(buf,128,user_config) != NULL)
	{
    if ((strnicmp(buf,"subscribe",9)) == 0)
      use_wafjoin = convert_to_number(extract_parm(buf,delimiter));
  }
  fclose(user_config);
GoodExit:
  return 0;
}

