/*
Snmp Library
Copyright (C) 1997 Alex Kowalenko Associates Pty Ltd. All rights reserved.

This software maybe be free distributed, any any form, without fee, 
but may not be modified in any way without express permission of 
the directors of Alex Kowalenko Associates Pty Ltd. 

Alex Kowalenko Associates Pty Ltd makes no representations or
warranties about the suitabililty of the software, not even the
implied warranty of merchantability or fitness for any particular
purpose.    
*/

package aka.snmp;

import java.io.*;
import java.util.*;

/**
 * Base class for SNMP type system
 * @version     $Id: Type.java,v 1.5 1997/05/21 10:09:56 alex Exp $
 * @author      Alex Kowalenko
 */

abstract public class Type implements BERSerializable, Serializable {

  private static Hashtable _objectFactory = new Hashtable(); // <String, Class>
  
    static {
      // Construct a HashTable for factory of objects
      _objectFactory.put(TypeInt.name, (new TypeInt()).getClass());
      _objectFactory.put(TypeString.name, (new TypeString()).getClass());
      _objectFactory.put(TypeGauge.name, (new TypeGauge()).getClass());
      _objectFactory.put(TypeCounter.name, (new TypeCounter()).getClass());
      _objectFactory.put(TypeTimeTick.name, (new TypeTimeTick()).getClass());
      _objectFactory.put(TypeIPAddress.name, (new TypeIPAddress()).getClass());
      _objectFactory.put(ObjectId.name, (new ObjectId()).getClass());

      // Add common names
      _objectFactory.put("DisplayString", (new TypeString()).getClass());
      _objectFactory.put("OctetString", (new TypeString()).getClass());
      _objectFactory.put("Opaque", (new TypeString()).getClass());
      _objectFactory.put("PhysAddress", (new TypeString()).getClass());
      _objectFactory.put("NetworkAddress", (new TypeIPAddress()).getClass());
    };

/**
 * Makes a type specified by typeName.
 */

    static Type make(String typeName) {
	try {
	    return (Type) ((Class) _objectFactory.get(typeName)).newInstance();
	} catch(Exception e) {
	    return new TypeNull();
	};
    };
  
/**
 * Parses a type from raw character data. Expects the type 
 * identifier to be the first character in the buffer pointed to.
 * Will remove all characters refering to that type from the buffer
 * and make the type specified.  
 */

    static Type parse(ByteBuffer buf) throws SnmpPDUException {
	byte type = buf.byteAt(0);
	buf.removeBeginning(1);
	switch(type) {
	  case ASN.INTEGER :
	    return new TypeInt(buf);
	  case ASN.OCTET_STR :
	    return new TypeString(buf);
	  case ASN.OBJECT_ID :
	    return new ObjectId(buf);
	  case ASN.NULL :
	    return new TypeNull(buf);
	  case ASN.TIMETICKS :
	    return new TypeTimeTick(buf);
	  case ASN.COUNTER :
	    return new TypeCounter(buf);
	  case ASN.GAUGE :
	    return new TypeGauge(buf);
	  case ASN.IPADDRESS :
	    return new TypeIPAddress(buf);
	  default :
	    throw new SnmpPDUException("SnmpType : Unknown type : " + type);
	}
    };

  protected Type() {
  };

/**
 * Will return type''s name.
 */
    
    abstract String typeName();
    
/**
 * Will read a value for the varaible from <str>.
 */
    
    abstract boolean read(String str);

/**
 * Convert the variable to a sequence of character, according to SNMP 
 * Protocol rules.
 */
    
    abstract public ByteBuffer BERSerialize();


/**
 * Test harness
 */

    static void testType(Type t) {
	System.out.println(t.typeName() + " value is " + t );
	StringUtil.Hexdump(t.BERSerialize());
	System.out.println("");
	try {
	    Type newType = Type.parse(t.BERSerialize());
	    System.out.println("Reserialise type is " + newType);
	} catch(SnmpPDUException e) {
	    System.out.println("Exception" + e);
	    e.printStackTrace();
	};
	return;
    }

/**
 * Test harness
 */

  public static void main(String args[]) {
  
      System.out.println("=====================================================");
      Type t = new TypeInt(172);
      testType(t);
      t = new TypeString("hello");
      testType(t);
      t = new TypeTimeTick(2345673);
      testType(t);
      t = new TypeIPAddress("1.23.255.33");
      testType(t); 
      t = new TypeCounter(255);
      testType(t); 
      t = new TypeGauge(255);
      testType(t); 
      t = new ObjectId("1.3.6.1.2.1.1.5");
      testType(t);
      t = new TypeNull();
      testType(t);

      try {
	  FileOutputStream file = new FileOutputStream("test.out");
	  byte array[] = new byte[1024];
	  String string = t.BERSerialize().toString();
	  System.out.println("lenght is " +  string.length());
	  array = string.getBytes();
	  file.write(array, 0, string.length());
	  file.close();
      } catch(Exception e) {
	  System.out.println("Exception");
      };

  };
  
};
