/*
Snmp Library
Copyright (C) 1997 Alex Kowalenko Associates Pty Ltd. All rights reserved.

This software maybe be free distributed, any any form, without fee, 
but may not be modified in any way without express permission of 
the directors of Alex Kowalenko Associates Pty Ltd. 

Alex Kowalenko Associates Pty Ltd makes no representations or
warranties about the suitabililty of the software, not even the
implied warranty of merchantability or fitness for any particular
purpose.    
*/

package aka.snmp;

/**
 * SNMP Object, as defined in the MIB
 * @version     $Id: SnmpObject.java,v 1.7 1997/05/21 10:09:56 alex Exp $
 * @author      Alex Kowalenko
 */

import java.io.*;

public class  SnmpObject implements BERSerializable, Serializable {
  
  private String _name;
  private ObjectId _id;
  private boolean _node;
  private Type _type;
  private Access _access;
  private Status _status;

    static SnmpObject Void = new SnmpObject("Void", "XXX");
  
/**
 * Constructor, creates SNMP Object with name, and an id (in the form of a
 * string).
 */
    
  public SnmpObject(String name, String id) {
      _name = name;
      _id = new ObjectId(id);
      _node = true;
      _type = Type.make("NULL");
      _access = new Access(Access.VOID);
      _status = new Status(Status.VOID);
  };

/**
 * Constructor, creates SNMP Object with name, and an id.
 */

  public SnmpObject(String name, ObjectId id) {
      _name = name;
      _id = id;
      _node = true;
      _type = Type.make("NULL");
      _access = new Access(Access.VOID);
      _status = new Status(Status.VOID);
  };

/**
 * Creates Snmp Object with name, ID (in the form of ObjectId), type, SNMP 
 * ACCESS, and SNMP Status.  This creates a complete Snmp Object.
 */

  public SnmpObject(String name, 
		    ObjectId id,
		    String typeName, 
		    Access access, 
		    Status status) {
      _name = name;
      _id = id;
      _node = false;
      _type = Type.make(typeName);
      _access = access;
      _status = status;
  };

/** 
 * This creates a SnmpObject from a byte representation, Used in
 * BER serialization of the objects.
 */

    SnmpObject(ByteBuffer buffer) throws SnmpPDUException {
	if(buffer.size() == 0)
	    throw new SnmpPDUException("No objects left");
    
	// SEQUENCE | CONSTRUCTOR
	if(!(buffer.byteAt(0) 
	     == (byte) (ASN.SEQUENCE|ASN.CONSTRUCTOR))) {
	    throw new SnmpPDUException("Incorrect coding");
	};
	buffer.removeBeginning(1);

	// Get length
	int length = ASN.getLength(buffer);// Total length of the packet.
		
        // Parse Name
	Type t = Type.parse(buffer);
	_name = ((TypeString)t).value();
		
	// Parse Id
	t = Type.parse(buffer);
	_id = (ObjectId) t;

	// Parse _node
	t = Type.parse(buffer);
	_node = ((TypeInt)t).value() == 1;
	
	// Parse _type
	t = Type.parse(buffer);
	_type = Type.make(((TypeString)t).value());

	// Parse _access
	t = Type.parse(buffer);
	_access = new Access(((TypeInt)t).value());

	// Parse _status
	t = Type.parse(buffer);
	_status = new Status(((TypeInt)t).value());
	return;
    };

/**
 * Serialize
 */

    public ByteBuffer BERSerialize() {
	ByteBuffer block = new ByteBuffer();
	Type t = new TypeString(_name);
	block.append(t.BERSerialize());
	block.append(_id.BERSerialize());
	t = new TypeInt( _node ? 1 : 0);
	block.append(t.BERSerialize());
	t = new TypeString( _type.typeName());
	block.append(t.BERSerialize());
	t = new TypeInt( _access.value());
	block.append(t.BERSerialize());
	t = new TypeInt( _status.value());
	block.append(t.BERSerialize());

	ByteBuffer result = new ByteBuffer();
	result.append((byte) (ASN.SEQUENCE | ASN.CONSTRUCTOR));
	result.append(ASN.buildLength(block.size()));
	result.append(block);
	return result;
    }; 
  
    
  public String name() {
	return _name;
    }

  public ObjectId id() {
	return _id;
    }

  public Access access() {
	return _access;
    }

  public Status status() {
	return _status;
    }

  public Type type() {
	return _type;
    };

  public boolean isNode() {
	return _node;
    }

  public String toString() {
      String result = "{" + _name + ", " + _id;
      if(_node) {
	  result += "}";
      } else {
	  result += ", " + _type.typeName() + ", " + 
	      _access + ", " + _status + "}";
      }
      return result;
  }

};
