/*
Snmp Library
Copyright (C) 1997 Alex Kowalenko Associates Pty Ltd. All rights reserved.

This software maybe be free distributed, any any form, without fee, 
but may not be modified in any way without express permission of 
the directors of Alex Kowalenko Associates Pty Ltd. 

Alex Kowalenko Associates Pty Ltd makes no representations or
warranties about the suitabililty of the software, not even the
implied warranty of merchantability or fitness for any particular
purpose.    
*/

package aka.snmp;

import java.util.*;
import java.io.*;
import java.net.*;

/**
 * This class implements a SNMP client.  The operations supported 
 * are SNMP Get, GetNext, and Set.  Also a Walk operation is implemented
 * 
 * @version     $Id: Snmp.java,v 1.10 1997/05/29 12:54:32 alex Exp $
 * @author      Alex Kowalenko
 */

public class Snmp {  

  private MibManager _mibs;
  private ConnectManager _conMan;
  private int _requestID;
  private String _communityName;
  
/**
 * Creation of client
 * @parameter cacheSize - sets the size of the cache of 
 * connections to multiple machines.
 */

  public Snmp(short cacheSize) {
      _mibs = new MibManagerSerializer();
      _conMan = new ConnectManager(cacheSize);
      _requestID = 0;
  };
 
/**
 * Creation of client, default size of cache of 5.
 */ 
 
  public Snmp() {
      this((short)5);
  };

  
/**
 * Read the SNMP objects from this file into memory.  The file
 * is a serializer version of the Managed Object Definitions.
 */

  public void readFile(String file) 
      throws IOException, SnmpMibException, FileNotFoundException {
	  ((MibManagerSerializer)_mibs).SerializeFromFile(file);
	  return;
  };

/**
 * Read the SNMP objects from this file into memory.  The file
 * is a MIB file that contains all the  managed objects.
 */

  public void readMibFile(String file) 
      throws SnmpMibException, FileNotFoundException {
	  ((MibManagerParser)_mibs).addFromMibDefFile(file);
	  return;
  };
    
/**
 * Set the community name that is going to be used.
 */

  public void setCommunityName(String name) {
      _communityName = name;
  };

/**
 * Set timeout for SNMP response to seconds.
 */

  public void setTimeOut(int seconds) {
      _conMan.setTimeOut(seconds);
  };

/** 
 * Dump out MIB for debugging
 */

  public void dumpMib() {
      _mibs.dump();
      return;
  };
  
/**
 * SNMP Get Operation.  Get the value of the variable and instance
 * from the SNMP agent at address.  It's value is returned as a Type.
 * Exceptions are thrown if the operation is unsuccessful.
 */
  
  public Type get(String address, String variable, int instance)
      throws SnmpException, UnknownHostException, 
	  IOException, SocketException {
	      PDU pdu = new PDU(address, 
				Operation.GET, 
				_communityName,  
				_requestID++);
	      ObjectId oid = _mibs.containsName(variable);
	      if(oid.equals(ObjectId.Void)) 
		  throw new SnmpException("Can't find variable name " + 
					  variable);
	      oid.addInstance(instance);
	      pdu.addVar(oid);
	      PDU resultPdu;

	      resultPdu = _conMan.send(address, pdu);
	      if(resultPdu.errorStatus().value() == SnmpError.NO_ERROR) {
		  return resultPdu.var(0).value();
	      }
	      else {
		  throw new SnmpException("Snmp error occured : " + 
					  resultPdu.errorStatus() 
					  + " in index : " + 
					  resultPdu.errorIndex());
	      }
  };
  
/**
 * SNMP GetNext Operation.  Get the value of the variable and instance from 
 * the SNMP agent at address.  It's value is returned in result, the next
 * variable in the agents MIB hierachy is returned in nextVariable.
 * A exception is generated if the operation is unsuccessful.
 */

  public Type getNext(String address, 
		      String variable, 
		      int instance, 
		      StringBuffer newVariable)
      throws SnmpException, UnknownHostException, 
	  IOException, SocketException  {
	      PDU pdu = new PDU(address, 
				Operation.GET_NEXT, 
				_communityName, 
				_requestID++);
	      ObjectId oid = _mibs.baseForm(variable);
	      if(oid.equals(ObjectId.Void))
		  throw new SnmpException("Can't find variable name '" + 
					  variable + "'");
	      if(instance >= 0)
		  oid.addInstance(instance);
	      pdu.addVar(oid);
	      PDU resultPdu;
	      resultPdu = _conMan.send(address, pdu);
  
	      if(resultPdu.errorStatus().value() == SnmpError.NO_ERROR) {
		  String result = resultPdu.var(0).value().toString();
		  ObjectId id = resultPdu.var(0).oid();
		  String nextVariable = _mibs.reduceToName(id);
		  if(nextVariable.length() == 0)
		      nextVariable = id.toString();
		  newVariable.append(nextVariable);
		  return resultPdu.var(0).value();
	      }
	      else
		  throw new SnmpException("Snmp error occured : " + 
					  resultPdu.errorStatus() +  
					  "in index : " +
					  resultPdu.errorIndex());
  };
  
/**
 * SNMP GetNext Operation.  Get the value of the variable from the SNMP 
 * agent at address.  It's value is returned in result, the next
 * variable in the agents MIB hierachy is returned in nextVariable.
 * A exception is generated if a problem occurs.  This same SNMP operation 
 * as the one above, but it differs that only an ID is given which could 
 * or could not have an instance attached to it
 */

  public Type getNext(String address, String variable, StringBuffer result) 
      throws SnmpException, UnknownHostException, 
	  IOException, SocketException  {    
	      return getNext(address, variable, -1, result);
  };

/**
 * Performs a multiple SNMP GetNext starting at variable
 * in the MIB hierachy to the SNMP agent at address.
 * Results are returned in the Vector result, which has elements of arrays
 * of two elements, the first element in the array is a String which
 * has the object name, the second element is a Type object with the
 * value of the variable.
 */

  public void walk(String address, String variable, Vector result)
      throws SnmpException, UnknownHostException, 
	  IOException, SocketException  {
	      ObjectId lastID; 
	      ObjectId thisID = _mibs.containsName(variable);
	      ObjectId topID = thisID; 
	      if(thisID.equals(ObjectId.Void))
		  throw new SnmpException("Can't find variable name '" + 
					  variable + "'");
	      do {
		  PDU pdu = new PDU(address, 
				    Operation.GET_NEXT, 
				    _communityName, 
				    _requestID++);
		  pdu.addVar(thisID);
		  PDU resultPdu;
		  resultPdu = _conMan.send(address, pdu);
	
		  if(resultPdu.errorStatus().value() == SnmpError.NO_ERROR) {
		      Object nextVariable[] = new Object[2];
		      ObjectId id = resultPdu.var(0).oid();
		      lastID = thisID;
		      thisID = id;
		      nextVariable[0] = _mibs.reduceToName(id);
		      if(((String)nextVariable[0]).length() == 0)
			  nextVariable[0] = id.toString();
		      nextVariable[1] = resultPdu.var(0).value();
		      result.addElement(nextVariable);
		      continue;
		  }
		  else
		      throw new SnmpException("Snmp error occured : " + 
					      resultPdu.errorStatus() + 
					      " in index : " + 
					      resultPdu.errorIndex());  
	      } while(thisID.isUnder(topID) && !(lastID.equals(thisID)));
	      result.removeElementAt(result.size() - 1);
	      return;
  };
  
/**
 * Performs the SNMP Set operation. Set the value of the 
 * variable of instance to value at SNMP agent at address
 * A exception is generated when the operation was unsuccessful.
 */

  public boolean setValue(String address, 
			  String variable, 
			  int instance, 
			  String value) 
      throws SnmpException, UnknownHostException, 
			      IOException, SocketException  {    
          PDU pdu = new PDU(address, 
			    Operation.SET, 
			    _communityName, 
			    _requestID++);
	  ObjectId oid = _mibs.containsName(variable);
	  if(oid.equals(ObjectId.Void)) 
	      throw new SnmpException("Can't find variable name");
	  
	  oid.addInstance(instance);
	  // Convert string into type
	  Type t = Type.make((_mibs.contains(oid)).type().typeName());
	  t.read(value);
	  pdu.addVar(oid,t);
	  PDU resultPdu;
	  resultPdu = _conMan.send(address, pdu);
	  if(resultPdu.errorStatus().value() == SnmpError.NO_ERROR) {
	      return true;
	  }
	  else
	      throw new SnmpException("Snmp error occured : " + 
				      resultPdu.errorStatus() +
				      ", Snmp index : " + 
				      resultPdu.errorIndex());
  }

/**
 * Test Harness
 */
  
  public static void main(String args[]) {

      if(args.length < 3) {
	  System.out.println("Snmp command host communityName oid [ value ] ");
	  return;
      };

      Snmp snmp = new Snmp();
      try {
	  snmp.readFile("mibs.ser");
      } catch (Exception e) {
	  System.err.println("Problem reading mib file " + e.getMessage());
	  return;
      }
      snmp.setTimeOut(3);

      snmp.setCommunityName(args[2]);
  
      try {
	  if(args[0].equals("get")) {
	      Type resultVal = snmp.get(args[1], args[3], 0);
	      System.out.println(args[3] + " = " + resultVal);
	      return;
	  }
	  else if(args[0].equals("getnext")) {
	      StringBuffer nextVar = new StringBuffer();
	      Type resultVal = snmp.getNext(args[1], args[3], 0, nextVar);
	      System.out.println(nextVar + " = " + resultVal);
	      return;
	  }
	  else if(args[0].equals("set")) {
	      boolean resultBoolean = snmp.setValue(args[1], args[3], 0, args[4]);
	      System.out.println(args[3] + " = " + resultBoolean);
	      return;
	  }
	  else if(args[0].equals("walk")) {
	      Vector resultVector = new Vector();
	      long current = 0;
	      try {
		  snmp.walk(args[1], args[3], resultVector);
	      } catch(Exception e) {
		  System.err.println("Problem with " + args[0] + ": " + e.getMessage());
		  e.printStackTrace();
	      }
 	      for(Enumeration e = resultVector.elements(); 
 		  e.hasMoreElements(); ) {
 		  Object element[] = (Object[]) e.nextElement();
 		  System.out.println((String) element[0] + " = " + 
 				     (Type) element[1]);
 	      }	  
	      return;
	  } 
	  else if(args[0].equals("test")) {
	      Vector resultVector = new Vector();
	      long current = 0;
	      try {
		  System.gc(); 
		  current = System.currentTimeMillis();
		  snmp.walk(args[1], args[3], resultVector);
		  current = System.currentTimeMillis() - current;
	      } catch(Exception e) {
		  System.err.println("Problem with " + args[0] + ": " + e.getMessage());
		  e.printStackTrace();
	      }
	      System.out.println("Time : " + current + "ms");
	      return;
	  }
	  else {
	      System.out.println("Unknown command : " + args[0]);
	      return;
	  };
      } catch(Exception e) {
	  System.err.println("Problem with " + args[0] + ": " + e.getMessage());
      }
  };

}
