/*
Snmp Library
Copyright (C) 1997 Alex Kowalenko Associates Pty Ltd. All rights reserved.

This software maybe be free distributed, any any form, without fee, 
but may not be modified in any way without express permission of 
the directors of Alex Kowalenko Associates Pty Ltd. 

Alex Kowalenko Associates Pty Ltd makes no representations or
warranties about the suitabililty of the software, not even the
implied warranty of merchantability or fitness for any particular
purpose.    
*/

package aka.snmp;

import java.util.*;

/**
 * This class implements a SNMP Protocol Data Unit (PDU).  These are 
 * the packets of information that are sent back and forth between 
 * SNMP agents and clients.
 * This class models these packets and will convert to and from,
 * sequences of characters according to the rules of the SNMP
 * protocol and BER encoding.
 * @see PDUVar
 * @version     $Id: PDU.java,v 1.4 1997/05/18 08:11:53 alex Exp $
 * @author      Alex Kowalenko
 */

public class PDU implements BERSerializable {
  
  private static int thisVersion = 0; // SNMP Version 1 Protocol

  private boolean _valid;
  private int _version;
  private String _address;
  private Operation _command;
  private String _communityName;
  private int _requestId;
  private SnmpError _errorStatus;
  private int _errorIndex;
  private Vector _varList; // Vector<PDUVariable>

/**
 * Construct a PDU packet with the address addr, SNMP operation
 * command, Community Name cn, and Request Identification
 * number redId.
 */

    PDU(String addr, byte command, String cn, int reqId) {
	_valid = true;
	_version = thisVersion;
	_address = addr;
	_command = new Operation(command);
	_communityName = cn;
	_requestId = reqId;
	_errorStatus = new SnmpError(SnmpError.NO_ERROR);
	_errorIndex = 0;
	_varList = new Vector();
    };
  
/** 
 * Construct a PDU packet from the sequence of characters in buffer. 
 */

    PDU(ByteBuffer buffer) throws SnmpPDUException {
	_valid = false;
	if(buffer.size() == 0)
	    return;
    
	// SEQUENCE | CONSTRUCTOR
	if(!(buffer.byteAt(0) == (byte) (ASN.SEQUENCE|ASN.CONSTRUCTOR))) {
	    throw new SnmpPDUException("Packet is not PDU");
	};
	buffer.removeBeginning(1);

	// Get length
	int length = ASN.getLength(buffer);// Total length of the packet.
	if(length >  buffer.size()) {
	    throw new SnmpPDUException("Packet length " + length + 
				       " doesn't match as specified " 
+ buffer.size());
	}
	else if(length < buffer.size())
	    buffer.truncate((int)length);

	// Parse Version number
	Type version = Type.parse(buffer);
	if(!version.typeName().equals(TypeInt.name)) {
	    throw new SnmpPDUException("Snmp version type wrong.");
	}
	_version = ((TypeInt)version).value();
    
	// Parse Community Name
	Type communityName = Type.parse(buffer);
	if(!communityName.typeName().equals(TypeString.name)) {
	    throw new SnmpPDUException("Snmp Community Name not found");
	}
	_communityName = ((TypeString)communityName).value();

	// Parse Snmp Operation
	byte snmpAction = buffer.byteAt(0);
	_command = new Operation((byte) (snmpAction & ~(ASN.CONTEXT) & ~(ASN.CONSTRUCTOR)));
	buffer.removeBeginning(1);
	length = ASN.getLength(buffer);
    
	// Parse Request id
	Type reqID = Type.parse(buffer);
	if(!reqID.typeName().equals(TypeInt.name)) {
	    throw new SnmpPDUException("Expecting Request type Integer");
	};
	_requestId = (int)((TypeInt)reqID).value();

	// Parse error-status
	Type errStat = Type.parse(buffer);
	if(!errStat.typeName().equals(TypeInt.name)) {
	    throw new SnmpPDUException("Expecting Error Status");
	}
	_errorStatus = new SnmpError(((TypeInt)errStat).value());
    
	// Parse error index
	Type errIdx = Type.parse(buffer);
	if(!errIdx.typeName().equals(TypeInt.name)) {
	    throw new SnmpPDUException("Expecting Error Index");
	};
	_errorIndex = (int) ((TypeInt)errIdx).value();
    
	// Parse Pdu Variables
	if(!(buffer.byteAt(0) == (byte) (ASN.SEQUENCE | ASN.CONSTRUCTOR))) {
	    throw new SnmpPDUException("Expecting variable-values");
	};
	buffer.removeBeginning(1);

	// Get length
	int subLength = ASN.getLength(buffer);
    
	_varList = new Vector(2);
	while(buffer.size() != 0)
	{
	    PDUVariable pduVar = new PDUVariable(buffer);
	    _varList.addElement(pduVar);
	}
	_valid = true;
    };

/**
 *  Construct an empty PDU packet
 */

    PDU() {
	_valid = false;
    };
  
/**
 * Add this variable <oid> with this value <value> to the
 * list of variables.  Used for SNMP Set operations.
 */

    void addVar(ObjectId oid, Type value) {
	PDUVariable var = new PDUVariable(oid, value);
	_varList.addElement(var);
    };
   
/**
 * Add this variable <oid> to the list of variables.  Used for
 * SNMP Get and GetNext operations.
 */

    void addVar(ObjectId oid) {
	Type t = new TypeNull();
	PDUVariable pv = new PDUVariable(oid, t);
	_varList.addElement(pv);
    }

/**
 * set Adress of PDU
 */

    void setAddress(String addr) {
	_address = addr;
    }

/**
 * Is this PDU valid, was a sequence of characters decoded into
 * a valid PDU.
 */

    boolean isValid() {
	return _valid;
    }

/**
 * Return the request identifier number of this request.
 */
    int requestID() {
	return _requestId;
    }

/**
 * What is the error status of this PDU.
 */

    SnmpError errorStatus() {
	return _errorStatus;
    };

/**
 * The error index points to which varaible.
 */

    int errorIndex() {
	return _errorIndex;
    };
  
/**
 * The number of variable-value pairs in this PDU.
 */

    int numOfVariables() {
	return _varList.size();
    };

/**
 * Return the Variable-value pair pointed to by the index.
 */
    PDUVariable var(int index) {
	return (PDUVariable) _varList.elementAt(index);
    };
      
/**
 * SNMP PROTOCOL
 * Transform this PDU to a sequence of characters, according to
 * rules of the SNMP protocol.
 */

    // Structure of PDU packet as I know it
    // 
    // Sequence {
    //     Integer version = 0
    //     String CommmunityName
    //     PDU Type | SEQUENCE {
    //         Integer request-id
    //         Integer error-status
    //         Integer error-index
    //         SEQUENCE {
    //             SEQUENCE {
    //                 ObjectID name;
    //                 Objecttype value;
    //             }
    //             ....
    //         }
    //     }
    // }
    
    public ByteBuffer BERSerialize() {
	_requestId++;
    
	ByteBuffer top = new ByteBuffer();
	TypeInt vers = new TypeInt(thisVersion);
	top.append(vers.BERSerialize());
	TypeString comm = new TypeString(_communityName);
	top.append(comm.BERSerialize());

	ByteBuffer packet = new ByteBuffer();
    
	// Request number
	TypeInt req = new TypeInt(_requestId);
	packet.append(req.BERSerialize());
	// Error code
	TypeInt err = new TypeInt(_errorStatus.value());
	packet.append(err.BERSerialize());
	// Error index
	TypeInt errind = new TypeInt(_errorIndex);
	packet.append(errind.BERSerialize());

	// List all variables.

	ByteBuffer varbuf = new ByteBuffer();
	for(Enumeration iter = _varList.elements(); iter.hasMoreElements(); ) {
	    varbuf.append(((PDUVariable)iter.nextElement()).BERSerialize());
	};
	ByteBuffer newvarbuf = new ByteBuffer();
	newvarbuf.append((byte) (ASN.SEQUENCE | ASN.CONSTRUCTOR));
	newvarbuf.append(ASN.buildLength(varbuf.size()));
	newvarbuf.append(varbuf);

	packet.append(newvarbuf);
	ByteBuffer newpacket = new ByteBuffer();
	newpacket.append((byte) (ASN.CONTEXT +  ASN.CONSTRUCTOR +  (byte) _command.value()));
	newpacket.append(ASN.buildLength(packet.size()));
	newpacket.append(packet);

	top.append(newpacket);

	ByteBuffer newtop = new ByteBuffer();
	newtop.append((byte) (ASN.SEQUENCE | ASN.CONSTRUCTOR));
	newtop.append(ASN.buildLength(top.size()));
	newtop.append(top);

	return newtop;
    };
  
/**
 * String representation of a PDU
 */

  public String toString() {
      String result = "{ " + _version + ", " + 
	  _address + ", " +
	  _command + ", " +
	  _requestId + ", " +
	  _errorStatus + ", " +
	  _errorIndex + ", {";
      for(Enumeration e = _varList.elements(); e.hasMoreElements(); )
	  result += e.nextElement().toString() + ", ";
      result += "}}";
      return result;
  };

};
