/*
Snmp Library
Copyright (C) 1997 Alex Kowalenko Associates Pty Ltd. All rights reserved.

$Id: ObjectId.java,v 1.9 1997/05/21 10:09:56 alex Exp $

This software maybe be free distributed, any any form, without fee, 
but may not be modified in any way without express permission of 
the directors of Alex Kowalenko Associates Pty Ltd. 

Alex Kowalenko Associates Pty Ltd makes no representations or
warranties about the suitabililty of the software, not even the
implied warranty of merchantability or fitness for any particular
purpose.    
*/

package aka.snmp;

import java.util.*;

/**
 * This is the Object Identifier, it is a subclass of Type.
 * @see Type
 * @version     $Id: ObjectId.java,v 1.9 1997/05/21 10:09:56 alex Exp $
 * @author      Alex Kowalenko
 */

public class ObjectId extends Type implements Cloneable {
  
    static final ObjectId Void = new ObjectId("XXX");
    
    static final byte asnValue = ASN.OBJECT_ID;

  public static String name = new String("ObjectID");
  private Vector _id;
  
/**
 * Constructs a ObjectId from a string
 */

  public ObjectId(String idStr) {
      int index = 0;
      _id = new Vector(15);
      String newString = idStr; 
  
      while(newString.length() > 0) {
	  index = newString.indexOf('.');
	  if(index == -1) {
	      try {
		  _id.addElement(new Integer(newString)); 
	      } catch (NumberFormatException e) {
		  // Incorrect format - invalid
		  return;
	      };
	      break;
	  }

	  String subString = newString.substring(0, index);
	  try {
	      _id.addElement(new Integer(subString));
	  } catch (NumberFormatException e) {
	      // Incorrect format - invalid
	      return;
	  };
	  newString = newString.substring(index + 1);
      };
  
  };

/**
 * Constucts a ObjectId from a sequence of bytes
 */
  
  public ObjectId(ByteBuffer buf) throws SnmpPDUException {
      _id = new Vector(15);
      int length = ASN.getLength(buf);
     
      int first = (int) buf.byteAt(0) / 40;
      int second = (int) buf.byteAt(0) % 40;
  
      _id.addElement(new Integer(first));
      _id.addElement(new Integer(second));
      for(int i = 1; i <= length - 1; i++) {
	  int b = (int) buf.byteAt(i);
	  b = (b >= 0 ? b : b + 256);
	  int t = b & 0x007F;
	  while (b >= 0x80) {
	      t = t << 7;
	      b = buf.byteAt(++i);
	      b = (int) (b >= 0 ? b : b + 256);
	      t += (b & 0x007F);
	  };
	  _id.addElement(new Integer((int) t));
      };
      buf.removeBeginning(length);
      return;
  }

/**
 * Default constructor
 */

  public ObjectId() {
      _id = new Vector(15);
  };

/**
 * comparison
 */

  public boolean equals(ObjectId id) {
      if(this == id)
	  return true;
      
      if(_id.size() != id._id.size())
	  return false;
  
      Enumeration e1 = _id.elements();
      Enumeration e2 = id._id.elements();
      try {
	  while(true) {
	      if(!e1.nextElement().equals(e2.nextElement()))
		  return false;
	  }
      } 
      catch (NoSuchElementException e) {  
	  // Both lists should be empty
	  if(! e1.hasMoreElements() && ! e2.hasMoreElements()) 
	      return true;
	  else
	      return false;
      }
  };
  
/**
 * comparison
 */

  public boolean equals(Object obj) {
      try {
	  ObjectId id = (ObjectId) obj;
	  return this.equals(id);
      } catch (Exception e) {
	  return false;
      };
  };

/**
 * creates a new copy of the object
 */

  protected Object clone() {
      ObjectId result = new ObjectId();
      for(Enumeration e = _id.elements(); e.hasMoreElements(); )
	  result._id.addElement(e.nextElement());
      return result;
  };

/**
 * hash code to store object
 */

  public int hashCode() {
      int code = 0;
      try { 
	  Enumeration e = _id.elements();
	  while(true) {
	      code +=  ((Integer)e.nextElement()).intValue();
	  }
      }
      finally {
	  return code;
      }
  }

/**
 * Is this a valid Object Identifier?
 */

  public boolean isValid() {
      return _id.size() > 0;
  }

/**
 * Is the current ID of an object under the SNMP Object ID topId.
 * In otherwords, is the current ID have the same begining as topId.
 */

  public boolean isUnder(ObjectId topId) { 
  
      if(topId._id.size() >= _id.size())
	  return false;
      Enumeration topE = topId._id.elements();
      Enumeration e = _id.elements();
      while(topE.hasMoreElements()) {
	  if(!topE.nextElement().equals(e.nextElement()))
	      return false;
      }
      return true;
  }

/**
 * The ID in dot notation
 */
    
  public String toString() {  
      if(!isValid())
	  return "Void";
      else {
	  String buffer = new String();
	  for(Enumeration e = _id.elements(); e.hasMoreElements();)
	      buffer = buffer + e.nextElement() + ".";
	  if(buffer.length() > 0)
	      buffer = buffer.substring(0,buffer.length() - 1);
	  return buffer;
      };
  }
 
/**
 * Add a integer to the ID, representing which instance of the
 * SNMP object you are interested in
 */

  public void addInstance(int i) {
      _id.addElement(new Integer(i));
  }
  
/**
 * Remove the last value in the ID as an Intance value, and
 * return it as an int.
 */

  public int removeInstance() {
      try {
	  int element = ((Integer)_id.elementAt(_id.size() - 1)).intValue();
	  _id.removeElementAt(_id.size() - 1);
	  return element;
      } catch(ArrayIndexOutOfBoundsException e) {
	  return -1;
      };
  
  }

/**
 * Returns the type of the object "ObjectID", when it is 
 * referenced as a Type object.
 */
  
  public String typeName() { 
      return name;
  };

/**
 * Reads a value from the string value
 */

  public boolean read(String value) {
      return false;
  };
    
/**
 * SNMP Protocol Conversion.
 * Converts this object into a sequence of characters
 * according to the rules of the SNMP protcol.
 */

    public ByteBuffer BERSerialize() {
	ByteBuffer buffer = new ByteBuffer();
	Enumeration e = _id.elements();
	if(!e.hasMoreElements())
	    return buffer;
	int first = ((Integer)e.nextElement()).intValue();
	int second = 0;
	if(e.hasMoreElements())
	    second = ((Integer)e.nextElement()).intValue();
	buffer.append((byte) (first*40 + second));
    
	int mask, testmask, testbits, bits;
	for( ; e.hasMoreElements(); ) {
	    mask = 0x7F;
	    bits = 0;
       
	    testmask = 0x7F; 
	    testbits = 0;
	    int item = ((Integer)e.nextElement()).intValue();
	    while(testmask != 0) {
		if((item & testmask) != 0) {
		    mask = testmask;
		    bits = testbits;
		}
		testmask <<= 7;
		testbits += 7;
	    }
	    while(mask != 0x7F) {
		if(mask == 0x1E00000)
		    mask = 0xFE00000;
		buffer.append((byte) (((item & mask) >> bits) | ASN.BIT8));
		mask >>= 7;
		bits -= 7;
	    }
	    buffer.append((byte) (item & mask));
	}
	ByteBuffer newbuffer = new ByteBuffer();
	newbuffer.append(asnValue);
	newbuffer.append(ASN.buildLength(buffer.size()));
	newbuffer.append(buffer);
	return newbuffer;
    };


/**
 * Test harness
 */

  public static void main(String args[]) {

      System.out.println("Snmp ID");
      ObjectId i = new ObjectId("1");
      System.out.println( "i : " + i);
      ObjectId j = new ObjectId("1.23.3.4444.5");
      System.out.println("j : " + j);
      ObjectId k = new ObjectId("..");
      System.out.println( "k : " + k);
      String str2 = "Garabage";
      ObjectId n = new ObjectId(str2);
      System.out.println("n : " + n);
      // n = i; copy

      System.out.println( "n : " + n);
      System.out.println("Is " + j + " under " + i + " : " + j.isUnder(i));
      System.out.println("Is " + i + " under " + j + " : " + i.isUnder(j));
      System.out.println("Is " + n + " under " + i + " : " + n.isUnder(i));
      System.out.println("Is " + n + " under " + j + " : " + n.isUnder(j));
      System.out.println("Is " + n + " under " + k + " : " + n.isUnder(k));
   
      ObjectId a = new ObjectId("1.3.6.1.2.1.1.1");
      ObjectId b = new ObjectId("1.3.6.1.2.1.1.1");
      System.out.println(a + " equals " + b + " = " + a.equals(b));
      
      ObjectId c = new ObjectId("1.3.6.1.2.100.1000.10000.100000.1000000.1");
      ByteBuffer buf = c.BERSerialize();
      StringUtil.Hexdump(buf);
      System.out.println(c);
      try {
	  Type t = Type.parse(buf);
	  System.out.println(t);
      } catch(Exception e) {};
  };
    
};
