/*
Snmp Library
Copyright (C) 1997 Alex Kowalenko Associates Pty Ltd. All rights reserved.

$Id: MibManager.java,v 1.10 1997/05/29 12:54:31 alex Exp $

This software maybe be free distributed, any any form, without fee, 
but may not be modified in any way without express permission of 
the directors of Alex Kowalenko Associates Pty Ltd. 

Alex Kowalenko Associates Pty Ltd makes no representations or
warranties about the suitabililty of the software, not even the
implied warranty of merchantability or fitness for any particular
purpose.    
*/

package aka.snmp;

import java.util.*;
import java.io.*;

/**
 * Responsible for holding the heirachy of managed objects.  Reads them
 * in from a number of different formats
 * @see SnmpObject
 * @version     $Id: MibManager.java,v 1.10 1997/05/29 12:54:31 alex Exp $
 * @author      Alex Kowalenko
 */

public class MibManager {

  protected Hashtable _tableName; // Map of String(names) -> SnmpObject
  protected Hashtable _tableID;   // Map of ObjectID -> SnmpObject

/**
 * Constructor
 */

  public MibManager() {
      _tableName = new Hashtable();
      _tableID = new Hashtable();
  };


/**
 * Insert a new object into the MIB
 */

  public boolean insertObject(SnmpObject obj) {
      if(!_tableName.contains(obj.name())) {
	  _tableName.put(obj.name(), obj);
	  _tableID.put(obj.id(), obj);  
	  return true;
      }
      return false;
  };

/**
 * Removes all SNMP Objects from this MIB
 */

    void deleteAll() {
	_tableName.clear();
	_tableID.clear();
    };
   
/**
 * Dumps all SNMP Objects to standard output.
 */
    
  public void dump() {
      PrintStream e = System.err;
      e.println("-----------------------------------------------------------------");
    
      for(Enumeration enum = _tableName.elements(); enum.hasMoreElements(); ) {
	  SnmpObject obj = (SnmpObject) enum.nextElement();
	  if(obj.isNode())
	      e.println(obj.name() + "\t" + obj.id());
	  else
	      e.println(obj.name() + "\t" + obj.id()
			+ "\t" + obj.type().typeName() 
			+ "\t" + obj.access() 
			+ "\t" + obj.status());
      }
    
  }
  
/**
 * Dumps all SNMP Objects to standard output. Different means
 * of access
 */

  public void dump2() {
      PrintStream e = System.err;
      e.println("-----------------------------------------------------------------");
      e.println("Size : " + _tableID.size());
      for(Enumeration enum = _tableID.elements(); enum.hasMoreElements(); ) {
	  SnmpObject obj = (SnmpObject) enum.nextElement();
	  if(obj.isNode())
	      e.println(obj.name() + "\t" + obj.id());
	  else
	      e.println(obj.name() + "\t" + obj.id()
			+ "\t" + obj.type().typeName() 
			+ "\t" + obj.access() 
			+ "\t" + obj.status());
      }
      e.println("-keys------------------------------------------------------------");
      e.println("Size : " + _tableID.size());
      for(Enumeration enum = _tableID.keys(); enum.hasMoreElements(); )
	  e.println((ObjectId) enum.nextElement());
      e.println("-----------------------------------------------------------------");
  }

/**
 * If SNMP Object is present with this name, returns SNMP ID of
 * object.
 */
    
  public ObjectId containsName(String name) {
      SnmpObject obj = (SnmpObject) _tableName.get(name);
      if(obj != null)
	  return (ObjectId) obj.id().clone();
      else
	  return ObjectId.Void;
  }

  
/**
 * Returns SNMP Object with this ID
 */

  public SnmpObject contains(ObjectId id) {
      SnmpObject obj = (SnmpObject) _tableID.get(id);
      if(obj != null)
	  return obj;
      else
	  return SnmpObject.Void;
  };
    
/**
 * Reduces the SNMP ID to the name, with instance variables 
 * attached
 */
  
  public String reduceToName(ObjectId id) {
      SnmpObject obj = contains(id);
      if(obj != SnmpObject.Void)
	  return obj.name();

      ObjectId testId = (ObjectId) id.clone();
      int instance = testId.removeInstance();
      obj = contains(testId);
      if(obj != SnmpObject.Void) {
	  String name = obj.name();
	  name += "." +  instance;
	  return name;
      }
    
      // Else try from the end of the string and search for a possible name
      Stack instances = new Stack();
      instances.push(new Integer(instance)); // push on first one striped off.
      while(testId.isValid() ) {
	  instance = testId.removeInstance();
	  instances.push(new Integer(instance));
	  obj = contains(testId);
	  if(obj != SnmpObject.Void) {
	      String name = obj.name();
	      while(!instances.isEmpty()) {
		  name += "." + ((Integer)instances.pop()).toString();
	      }
	      return name;
	  }
      }
      return null;
  };
    
/**
 * Converts a SNMP ID represented in a dot notation that contains 
 * SNMP Object names and numbers, to one containing only numbers.
 * It does this using SNMP Objects currently stored.
 */

  public ObjectId baseForm(String id) {
      StringBuffer decodedID = new StringBuffer();
      StringTokenizer idList = new StringTokenizer(id, ".");
      for(; idList.hasMoreElements();) {
	  String item = idList.nextToken();
	  if(StringUtil.isNumeric(item)) {
	      decodedID.append(item);
	      decodedID.append(".");
	  }
	  else {
	      ObjectId replace = containsName(item);
	      if(replace.isValid()) {
		  decodedID.append(replace.toString());
		  decodedID.append('.');
	      }
	      else {
		  return ObjectId.Void;
	      }
	  }
      }
      decodedID.setLength(decodedID.length() - 1);
      ObjectId returnId = new ObjectId(decodedID.toString());
      return returnId;
  };

/**
 * Test harness
 */
    
  public static void main(String args[]) {
      PrintStream out = System.out;

      out.println("Creating SNMP MIB MANAGER");
      MibManagerParser mibs = new MibManagerParser();

      out.println("Reading SNMP mib");
      try {
	  mibs.addFromMibDefFile("mib1.txt");
      } catch(FileNotFoundException e) {
	  out.println("File not found");
	  return;
      } catch(Exception e) {
	  out.println("Exception");
	  e.printStackTrace();
	  mibs.dump();
      };

      //  mibs.dump();
      //  mibs.dump2();
  
      out.println("1.1.3.6.1.2.1.1.3 - " + mibs.containsName("sysUpTime")
		  + ", sysUpTime " + mibs.contains(mibs.containsName("sysUpTime")).name()
		  + ", TimeTicks "  + mibs.contains(mibs.containsName("sysUpTime")).type().typeName());
	      
      out.println("");
      out.println(mibs.containsName("sysUpTime"));
      out.println(new ObjectId(mibs.containsName("sysUpTime").toString()));
      ObjectId id = new ObjectId(mibs.containsName("sysUpTime").toString());
  
      if(id.isValid())
	  out.println("sysUpTime " + mibs.contains(id).name() + "\n");
  
      out.println("system.sysName : " + mibs.baseForm("system.1"));

      out.println(mibs.containsName("sysName"));
      out.println(mibs.containsName("sysContact"));
    
      ObjectId a = new ObjectId("1.3.6.1.2.1.1.7");
      ObjectId b = new ObjectId("1.3.6.1.2.1.1.6");

      out.println(mibs.contains(a).name());
      out.println(mibs.contains(b).name());

      out.println("Reduce SNMP ID to names ");
      out.println(a + " reduces to " + mibs.reduceToName(a));
      ObjectId c = new ObjectId("1.3.6.1.2.1.1.10.0");
      ObjectId d = new ObjectId("2");
      ObjectId c1 = new ObjectId("1.3.6.1.2.1.1.3.0");
      out.println(c + " reduces to " + mibs.reduceToName(c));
      out.println(c1 + " reduces to " + mibs.reduceToName(c1));
      out.println(d + " reduces to " + mibs.reduceToName(d));

  };

};
