/*
Snmp Library
Copyright (C) 1997 Alex Kowalenko Associates Pty Ltd. All rights reserved.

This software maybe be free distributed, any any form, without fee, 
but may not be modified in any way without express permission of 
the directors of Alex Kowalenko Associates Pty Ltd. 

Alex Kowalenko Associates Pty Ltd makes no representations or
warranties about the suitabililty of the software, not even the
implied warranty of merchantability or fitness for any particular
purpose.    
*/

package aka.snmp;

import java.net.*;
import java.io.*;

/**
 *
 * This class implements a SNMP Unix/UDP Connection to an SNMP agent at
 * specific address.
 * <P>
 * Connect is managed by ConnectManager.  One difference
 * between Connect and ConnectManager is that 
 * Connect deals only with arrays of bytes of characters and knows
 * nothing about their contents.  ConnectManager deals with
 * Pdu objects.
 *
 * @version     $Id: Connect.java,v 1.7 1997/05/18 07:34:11 alex Exp $
 * @author      Alex Kowalenko
 * @see ConnectManager
 * @see Pdu
 */

class Connect implements Runnable {

  public static final short SNMP_PORT = 161;
  private static final short MAX_SNMP_LENGTH = 2048;
  private static final byte sentinal[] = { 0x00};

  private String _address;
  private int _timeOut;
  private int _port;
  private boolean _isOpen;
  private DatagramSocket _socket;
  private InetAddress _inetAddress;
  private int _noActions;
  
/**
 * Constructor
 */

    Connect(String address, int timeOut, int port) 
	throws UnknownHostException {
	    _address = address;
	    _inetAddress = InetAddress.getByName(address);
	    _timeOut = timeOut;
	    _port = port;
    };

/**
 * finalize
 */

  protected void finalize() {
      close();
  };

/**
 * Opens a connection
 */
    
    boolean open() throws SnmpNetException, SocketException {
	if(_isOpen)
	    throw new SnmpNetException("SnmpConnect.open already open");
    
	_socket = new DatagramSocket();
	_isOpen = true;
	return _isOpen;
    };
  
/**
 * Closes the connection 
 */

    void close() {
	_socket.close();
    };
  
/**
 * Is the connection currently open.
 */
    boolean isOpen() {
	return _isOpen;
    };
  
/**
 * Send this sequence of characters to the other side, return 
 * the response as a sequence of characters.
 */

    ByteBuffer send(ByteBuffer msg) throws IOException, SnmpTimeOutException {
	_noActions++;
	// System.out.println("Send this: ");
	// StringUtil.Hexdump(msg);
	DatagramPacket packet 
	    = new DatagramPacket(msg.array(), msg.size(), _inetAddress, _port);
    
	_socket.send(packet);

	Thread thread = new Thread(this);
	thread.start();
	byte recvBuffer[] = new byte[MAX_SNMP_LENGTH];
	DatagramPacket recvPacket 
	    = new DatagramPacket(recvBuffer, recvBuffer.length);
    
	try {
	    _socket.receive(recvPacket);
	}
	finally {
	    thread.stop();
	};

	ByteBuffer result = new ByteBuffer(recvBuffer, recvPacket.getLength());
	if(result.byteAt(0) == (byte) 0x00)
	    throw new SnmpTimeOutException("TimeOut to " + _address);
	// StringUtil.Hexdump(result);
	return result;
    };

/** 
 * This unblocks the socket when a packet is sent.  It writes a dummy
 * message to the thread
 */

  public void run() {
      try {
	  Thread.currentThread().sleep(_timeOut * 1000);

	  // Send a message to myself to wake the other thread
	  DatagramPacket packet 
	      = new DatagramPacket(sentinal, 
				   sentinal.length, 
				   InetAddress.getLocalHost(), 
				   _socket.getLocalPort());
	  DatagramSocket socket = new DatagramSocket();
	  socket.send(packet);
      } catch(Exception e) { }
      finally {
	  Thread.currentThread().stop();
	  return;
      }
  };
  
/** 
 * Cache information
 */

    int count() {
	return _noActions;
    };

};
