/*
Snmp Library
Copyright (C) 1997 Alex Kowalenko Associates Pty Ltd. All rights reserved.

This software maybe be free distributed, any any form, without fee, 
but may not be modified in any way without express permission of 
the directors of Alex Kowalenko Associates Pty Ltd. 

Alex Kowalenko Associates Pty Ltd makes no representations or
warranties about the suitabililty of the software, not even the
implied warranty of merchantability or fitness for any particular
purpose.    
*/

package aka.snmp;

/**
 * This class is similar to StringBuffer, but handles arrays of 8-bit
 * characters.
 * @version     $Id: ByteBuffer.java,v 1.6 1997/05/18 07:34:11 alex Exp $
 * @author      Alex Kowalenko
 */

public class ByteBuffer {

  private int _start; // Start of array  
  private int _end;   // Next avaliable position 
  private byte _array[];
    
  public static int STARTSIZE = 32;
    
/** 
 * Default constructor 
 */
    
  public ByteBuffer() {
      _array = new byte[STARTSIZE];
      _start = _end = 0;
  };
    
/**
 * Constructor from a byte[], copys a subarray into the object
 * @param array the array
 * @param length length of the subarray
 */  
  
  public ByteBuffer(byte array[], int length) {
      _array = new byte[ length > STARTSIZE ? length : STARTSIZE];
      System.arraycopy(array, 0, _array, 0, length); 
      _start = 0;
      _end = length;
  };

/**
 * append byte b to the end of the buffer.
 */

  public ByteBuffer append(byte b) {
      if(_end >= _array.length)
	  selfExtend(1);
      _array[_end] = b;
      _end++;
      return this;
  };

/**
 * append the array to the end of the buffer.
 */

  public ByteBuffer append(byte array[]) {
      if(_end + array.length >= _array.length)
	  selfExtend(array.length);
      System.arraycopy(array, 0, _array, _end, array.length); 
      _end += array.length;
      return this;
  };

/**
 * append the subarray to the end of the buffer.
 */

  public ByteBuffer append(byte array[], int size) {
      if(_end + size >= _array.length)
	  selfExtend(size);
      System.arraycopy(array, 0, _array, _end, size); 
      _end += size;
      return this;
  };

/**
 * append a ByteBuffer to the end of the buffer.
 */

  public ByteBuffer append(ByteBuffer buffer) {
      if(_end + buffer._end - buffer._start >= _array.length)
	  selfExtend(buffer._end - buffer._start);
      System.arraycopy(buffer._array, buffer._start, 
		       _array, _end, buffer._end - buffer._start); 
      _end += buffer._end - buffer._start ;
      return this;
  };

/**
 * Append a string to the end of the array, 16-bit characters truncated
 * to 8-bit characters
 */

  public ByteBuffer append(String str) {
      if(_end + str.length() >= _array.length)
	  selfExtend(str.length());
      byte tmp[] = str.getBytes();
      System.arraycopy(tmp, 0, _array, _end, tmp.length); 
      _end += tmp.length;
      return this;
  };

/**
 * return the byte[] array
 */

  public byte[] array() {
      if(_start == 0 ) {
	  return _array;
      } else {
	  byte copy[] = new byte[_end - _start];
	  System.arraycopy(_array, _start, copy, 0, _end - _start);
	  return copy;
      }
  };

/**
 * byte at an index
 */

  public byte byteAt(int index) throws ArrayIndexOutOfBoundsException {
      return _array[_start + index];
  };


/** 
 * implementation of equals
 */

  public boolean equals(byte array[]) {
      for(int i = 0; i < ( size() < array.length ? size() : array.length); i++)
	  if( _array[_start + i] != array[i])
	      return false;
      return true;
  };

/**
 * size of the array
 */

  public int size() {
      return _end - _start;
  };

/**
 * sets the byte at an index to b
 */

  public void setByteAt(int index, byte b) 
      throws ArrayIndexOutOfBoundsException {
	  _array[_start + index] = b;
	  return;
  };
 
/**
 * remove the first length
 */

  public void removeBeginning(int length) {
      if(length > (_end - _start)) { 
	  _start = _end = 0;      
      }
      else {
	  _start += length;
      };
  };

/**
 * truncate the array to this length
 */

    void truncate(int length) {
	if(_end - _start > length)
	    _end = _start + length;
	return;
    }

/**
 * return the string representation of the array
 */

  public String toString() {
      return new String(_array, _start, _end - _start);
  };

    /**
     * return the string representation of the array to this length
     */

  public String toString(int len) {
      return new String(_array, 
			_start,
			((_end - _start) < len ? _end - _start : len));
  };


  private void selfExtend(int size) {
      byte newArray[] = new byte[ size > STARTSIZE 
				? _array.length + size + STARTSIZE 
				: _array.length + STARTSIZE];
      System.arraycopy(_array, _start, newArray, 0, _end - _start);
      _array = newArray;
  };
  

/**
 * Test harness
 */

  public static void main(String args[]) {
      byte a[] = { 1, 2, 3};
      ByteBuffer b = new ByteBuffer(a, a.length);
      b.removeBeginning(1);
  };
};
