/*
Snmp Library
Copyright (C) 1997 Alex Kowalenko Associates Pty Ltd. All rights reserved.

This software maybe be free distributed, any any form, without fee, 
but may not be modified in any way without express permission of 
the directors of Alex Kowalenko Associates Pty Ltd. 

Alex Kowalenko Associates Pty Ltd makes no representations or
warranties about the suitabililty of the software, not even the
implied warranty of merchantability or fitness for any particular
purpose.    
*/

package aka.snmp;

/**
 * Class encapsulating ASN constants and miscellanous ASN routines.
 * @version     $Id: ASN.java,v 1.3 1997/05/08 13:04:46 alex Exp $
 * @author      Alex Kowalenko
 */

public class ASN {
    
  public static final byte BOOLEAN =      (byte) 0x01;
  public static final byte INTEGER =      (byte) 0x02;
  public static final byte BIT_STR =      (byte) 0x03;
  public static final byte OCTET_STR =    (byte) 0x04;
  public static final byte NULL =         (byte) 0x05;
  public static final byte OBJECT_ID =    (byte) 0x06;
  public static final byte SEQUENCE =     (byte) 0x10;
  public static final byte SET =          (byte) 0x11;
    
  public static final byte UNIVERSAL =     (byte) 0x00;
  public static final byte APPLICATION =   (byte) 0x40;
  public static final byte CONTEXT =       (byte) 0x80;
  public static final byte PRIVATE =       (byte) 0xC0;
    
  public static final byte PRIMITIVE =     (byte) 0x00;
  public static final byte CONSTRUCTOR =   (byte) 0x20;
    
  public static final byte LONG_LEN =      (byte) 0x80;
  public static final byte EXTENSION_ID =  (byte) 0x1F;
  public static final byte BIT8 =          (byte) 0x80;
    
  public static final byte IPADDRESS =    (byte) 0x40;
  public static final byte COUNTER =      (byte) 0x41;
  public static final byte TIMETICKS =    (byte) 0x43;
  public static final byte GAUGE =        (byte) 0x42;
  
/**
 * Returns the ASN encoding of length 
 * @see aka.snmp.ASN#getLength
 */

  public static ByteBuffer buildLength(int length) {
      ByteBuffer result = new ByteBuffer();
      if(length < 0x80) {
	  result.append( (byte) length);
      }
      else if(length <= 0xFF) {
	  result.append( (byte) (0x01 | LONG_LEN));
	  result.append( (byte) length);
      }
      else { // 0xFF < length <= 0xFFFF 
	  result.append( (byte) (0x02 | LONG_LEN));
	  result.append( (byte) ((length >> 8) & 0xFF));
	  result.append( (byte) (length & 0xFF));
      }
      return result;
  };

/**
 * Returns the length from the ASN encoding in buffer
 * @see aka.snmp.ASN#buildLength
 */
  
  public static int getLength(ByteBuffer buffer) {
      byte var = buffer.byteAt(0);
      int result;
      switch(var) {
	case (byte) 0x01 | LONG_LEN:
	  result = buffer.byteAt(1);
	  result = (result < 0) ? result + 256 : result;
	  buffer.removeBeginning(2);
	  return result;
	case (byte) 0x02 | LONG_LEN:
	  result = buffer.byteAt(1);
	  result = (result < 0) ? result + 256 : result;
	  result <<= 8;
	  result = result | ( buffer.byteAt(2) > 0 ? buffer.byteAt(2) :  buffer.byteAt(2) +256);
	  buffer.removeBeginning(3);
	  return result;
	default:
	  result = var;
	  buffer.removeBeginning(1);
      }
      return result;
  };

  private ASN() {};

/**
 * Test Harness
 */

  public static void main(String args[]) {
      StringUtil.Hexdump(buildLength(16));
      StringUtil.Hexdump(buildLength(250));
      StringUtil.Hexdump(buildLength(1000));
      
      System.out.println(getLength(buildLength(16)));
      System.out.println(getLength(buildLength(250)));
      System.out.println(getLength(buildLength(1000)));
  };

};
