/*
 * File: sipp_prism.c 
 *
 * Create a prism from a polygonal cross-section in XY.
 * The points must be given in counter-clockwise order as viewed from
 * the front (+ve Z).
 *
 * Author:  David Jones
 *          djones@awesome.berkeley.edu
 *          11may91 djones
 *
 * Adapted for inclusion into the SIPP package:  Inge Wallin
 */

#include <stdio.h>
#include <math.h>

#include <sipp.h>


Object *
sipp_prism(num_points, points, length, surface, shader)
    int	      num_points;
    Vector  * points;
    double    length;
    void    * surface;
    Shader  * shader;
{
    Object  * prism;
    int	     i;
    int	     j;

    prism = object_create();

    /* The top. */
    for (i = 0; i < num_points ; ++i) {
        vertex_tx_push(points[i].x, points[i].y, length / 2.0, 
                       points[i].x, points[i].y, length / 2.0);
    }
    polygon_push();
    object_add_surface(prism, surface_create(surface, shader) );

    /* The bottom */
    for (i = num_points - 1; i >= 0 ; --i) {
        vertex_tx_push(points[i].x, points[i].y, -length / 2.0, 
                       points[i].x, points[i].y, -length / 2.0);
    }
    polygon_push();
    object_add_surface(prism, surface_create(surface, shader) );

    /* The sides */
    for (i = 0; i < num_points ; ++i) {
        j = i + 1;
        if (j == num_points)
            j=0;
        vertex_tx_push(points[i].x, points[i].y,  length / 2.0, 
                       points[i].x, points[i].y,  length / 2.0);
        vertex_tx_push(points[i].x, points[i].y, -length / 2.0, 
                       points[i].x, points[i].y, -length / 2.0);
        vertex_tx_push(points[j].x, points[j].y, -length / 2.0, 
                       points[j].x, points[j].y, -length / 2.0);
        vertex_tx_push(points[j].x, points[j].y,  length / 2.0, 
                       points[j].x, points[j].y,  length / 2.0);
        polygon_push();
        object_add_surface(prism, surface_create(surface, shader) );
    }
    
    return prism;
}


Object *
sipp_block(xsize, ysize, zsize, surface, shader)
    double    xsize;
    double    ysize;
    double    zsize;
    void    * surface;
    Shader  * shader;
{
    Vector   points[4];
    int      i;

    points[0].x =   xsize / 2.0;
    points[0].y = - ysize / 2.0;
    points[1].x =   xsize / 2.0;
    points[1].y =   ysize / 2.0;
    points[2].x = - xsize / 2.0;
    points[2].y =   ysize / 2.0;
    points[3].x = - xsize / 2.0;
    points[3].y = - ysize / 2.0;
    for (i = 0; i < 4; ++i) {
        points[i].z = 0.0;
    }

    return sipp_prism(4, &points[0], zsize, surface, shader);
}


Object *
sipp_cube(size, surface, shader)
    double   size;
    void     *surface;
    Shader   *shader;
{
    return sipp_block(size, size, size, surface, shader);
}
