UNIT DrawU1;

{ SpeedSoft Sibyl Sample application
  (C) 1996 SpeedSoft, All rights reserved.

  This application demonstrates how to create a multi form applications with drawing
  facilities using the visual environment of SpeedSoft Sibyl.

  The application includes two forms. One MainForm (DrawApp)
  and a About box (AboutDlg). The user may draw different types of figures by using the mouse
  device. Figure types may be toggled by using Switchbuttons integrated within a toolbar.
  The other two forms are displayed according to menu commands from a menubar.
}

INTERFACE

USES Classes,Forms,StdCtrls,StdDlgs,Buttons,DrawU2;
                        
//user defined menu constants. These constanst are used together with the menu entries
//"Fill Interior" and "Color" within the "SendCommand" property
CONST
    cmFillInterior =  cmUser+1;
    cmColors       =  cmUser+2;

//Standard colors of Sibyl for use within the valueset
CONST StdColors:ARRAY[1..18] OF TColor=
        (
         clBlack,clMaroon,clGreen,clOlive,clNavy,clPurple,
         clTeal,clGray,clSilver,clRed,clLime,clYellow,
         clBlue,clFuchsia,clAqua,clLtGray,clDkGray,clWhite
        );

//Figure types constants.
TYPE
    TFigureType=(figLine,figCircle,figRectangle,figBezier,figFreeHand);

//Draw list. All drawing activities are stored within that list. This list is used when
//a part of the window becomes invalid and must be redrawn. In that case the "DrawAppOnPaint"
//event handler is called.
    PDrawList=^TDrawList;
    TDrawList=RECORD
                    FigureType:TFigureType;
                    Color:TColor;
                    FillInterior:BOOLEAN;
                    StartPoint,EndPoint:TPoint;
                    next:PDrawList;
    END;

//The toolbar control. It includes five Switchbuttons to toggle the type of the figure to draw.
//ActiveSwitch identifies the current Switchbutton which is switched on (SwitchButton1 - Lines)
//by default.
    TToolBar1=CLASS(TToolBar)
         SwitchButton1:TSwitchButton;
         SwitchButton2:TSwitchButton;
         SwitchButton3:TSwitchButton;
         SwitchButton4:TSwitchButton;
         SwitchButton5:TSwitchButton;
         ValueSet1:TValueSet;
         ActiveSwitch:TSwitchButton;
         Procedure ValueSet1OnItemActivated(Sender:TComponent;SelColumn:WORD;SelRow:WORD);
         Procedure ToolBarOnSetupShow(Sender:TComponent);
         PROCEDURE FigureChanged(Sender:TComponent;NewState:TSwitchButtonState);
    END;

//The main application form. This type and the global variable are referenced from the main
//program and generated automatically.

    TDrawApp=CLASS(TForm)
         ToolBar1:TToolBar1;
         ColorDialog1:TColorDialog;
       PUBLIC
             StartPoint:TPoint;
             EndPoint:TPoint;
             DrawMode:BOOLEAN;
             FirstPoint:BOOLEAN;
             FigureType:TFigureType;
             FillInterior:BOOLEAN;
             Color:TColor;
             DrawList,LastDrawItem:PDrawList;
       PUBLIC
         PROCEDURE DrawAppOnPaint(Sender:TComponent;rec:TRect);
         PROCEDURE DrawAppOnCommandEvent(Sender:TComponent;VAR Msg:TMessage;Command:TCommand);
         PROCEDURE DrawAppOnMouseMove(Sender:TComponent;VAR Msg:TMessage;VAR pt:TPoint;KeyState:TMouseKeyState);
         PROCEDURE DrawAppOnLMouseUp(Sender:TComponent;VAR Msg:TMessage;VAR pt:TPoint;KeyState:TMouseKeyState);
         PROCEDURE DrawAppOnLMouseDown(Sender:TComponent;VAR Msg:TMessage;VAR pt:TPoint;KeyState:TMouseKeyState);

         PROCEDURE DrawFigure(RubberBand:BOOLEAN;ptEnd:TPoint);
         PROCEDURE DrawElement(X,Y,X1,Y1:LONGINT;RubberBand:BOOLEAN);
         PROCEDURE SetupComponent;OVERRIDE;
    END;

VAR
    DrawApp:TDrawApp;

IMPLEMENTATION

//This event handler is called if the user changed the selection of the valueset
Procedure TToolBar1.ValueSet1OnItemActivated(Sender:TComponent;SelColumn:WORD;SelRow:WORD);
Begin
     DrawApp.Color:=StdColors[SelColumn];
End;

//This event handler is called if the toolbar is about to be shown
Procedure TToolBar1.ToolBarOnSetupShow(Sender:TComponent);
Begin
     //Initialize ValueSet
     ValueSet1.SetRGBData(1,1,StdColors);
End;

//This event handler is called for each of the five Switchbuttons if the state of the button
//has changed. If a Switchbutton is switched on, the previously active Switchbutton will be
//switched off and the type of the figure is changed.
PROCEDURE TToolBar1.FigureChanged(Sender:TComponent;NewState:TSwitchButtonState);
BEGIN
     IF NewState=SwitchOn THEN
     BEGIN
          IF ActiveSwitch<>NIL THEN ActiveSwitch.State:=SwitchOff
          ELSE SwitchButton1.State:=SwitchOff;
          ActiveSwitch:=TSwitchButton(Sender);

          CASE ActiveSwitch OF
              SwitchButton1:DrawApp.FigureType:=figLine;
              SwitchButton2:DrawApp.FigureType:=figRectangle;
              SwitchButton3:DrawApp.FigureType:=figCircle;
              SwitchButton4:DrawApp.FigureType:=figBezier;
              SwitchButton5:DrawApp.FigureType:=figFreeHand;
          END; {case}
     END;
END;

//This event handler is called whenever a part of the window or the whole window has become
//invalid. "rec" identifies the rectangle that needs redraw. To ensure that drawing is made only
//to invalid portions of the window, a clip region is defined. This routine uses the "DrawList"
//variable to draw all figures within that list. This routine uses the Canvas property to
//specify drawing properties. The Canvas of a window provides easy access to manage graphical
//operations on a window.
PROCEDURE TDrawApp.DrawAppOnPaint(Sender:TComponent;rec:TRect);
VAR dummy:PDrawList;
    SaveFillInterior:BOOLEAN;
    SaveFigureType:TFigureType;
BEGIN
     //set the clipping rectangle
     Canvas.ClipRect:=rec;
     Canvas.FgMode:=fgOverpaint;
     SaveFillInterior:=FillInterior;
     SaveFigureType:=FigureType;
     //loop through the list of figures
     dummy:=DrawList;
     WHILE dummy<>NIL DO
     BEGIN
          Canvas.Color:=dummy^.Color;
          FillInterior:=dummy^.FillInterior;
          FigureType:=dummy^.FigureType;
          //Draw a single element
          DrawElement(dummy^.StartPoint.X,dummy^.StartPoint.Y,dummy^.EndPoint.X,dummy^.EndPoint.Y,FALSE);
          dummy:=dummy^.Next;
     END;
     FillInterior:=SaveFillInterior;
     FigureType:=SaveFigureType;
     //delete clip rectangle
     Canvas.DeleteClipRect;
END;

//This event handler is called whenever a control window (such as the menu) has to report a
//command event to its owner. A command event is specified by modifying the control's "SendCommand"
//property. Each time the user selects a menu entry this routine gets called.
PROCEDURE TDrawApp.DrawAppOnCommandEvent(Sender:TComponent;VAR Msg:TMessage;Command:TCommand);
VAR r,c:WORD;
    temp,dummy:PDrawList;
BEGIN
     CASE Command OF                
        cmFillInterior: //fill Interior
        BEGIN
             FillInterior:=not FillInterior;
             //toggle the state of the menu entry
             MainMenu.CheckMenuEntry(cmFillInterior,FillInterior);
        END;
        cmColors: //Color
        BEGIN
             //Create color dialog window with the default color of the current color
             Color:=ColorDialog1.Run(Color);
        END;
        cmAbout:  //About
        BEGIN
             //Create a modal window, display it and destroy it finally.
             AboutDlg.Create(SELF);
             AboutDlg.ShowModal;
             AboutDlg.Destroy;
        END;
        cmNew:
        BEGIN
             //clear the list of figures
             dummy:=DrawList;
             WHILE dummy<>NIL DO
             BEGIN
                  temp:=dummy^.Next;
                  dispose(dummy);
                  dummy:=temp;
             END;
             DrawList:=NIL;
             //Force the window to redraw
             InvalidateWindow;
        END;
        cmOpen,cmSave:ErrorBox('Feature not implemented');
     END;
END;

//This routine performs initialization for the main form. It is called after the "DrawApp"
//object has been created.
PROCEDURE TDrawApp.SetupComponent;
BEGIN
     Inherited SetupComponent;
     FigureType:=figLine;
     Color:=clBlack;
END;

//This routine calculates a value needed by the circle draw commands.
FUNCTION Pythagoras(X,Y,X1,Y1:LONGINT):LONGINT;
VAR temp:Extended;
BEGIN
     temp:=sqr(X1-X)+sqr(Y1-Y);
     result:=round(sqrt(temp));
END;

//This routine draws a single element according to the "FigureType" variable
//It uses the Canvas property to specify drawing properties. The Canvas of a window provides
//easy access to manage graphical operations on windows.
PROCEDURE TDrawApp.DrawElement(X,Y,X1,Y1:LONGINT;RubberBand:BOOLEAN);
VAR aptl:ARRAY[0..2] OF TPoint;
BEGIN
     //look what type of figure to draw
     CASE FigureType OF
        figLine:Canvas.DrawLine(X,Y,X1,Y1);                            //draw lines
        figCircle:                                                     //draw a circle
        BEGIN
             IF ((not RubberBand)AND(FillInterior)) THEN
                  Canvas.DrawFilledCircle(X,Y,Pythagoras(X,Y,X1,Y1))
             ELSE Canvas.DrawCircle(X,Y,Pythagoras(X,Y,X1,Y1));
        END;
        figRectangle:                                                  //draw a rectangle
        BEGIN
             IF ((not RubberBand)AND(FillInterior)) THEN
                  Canvas.DrawBox(Rect(X,Y,X1,Y1))
             ELSE Canvas.DrawRectangle(Rect(X,Y,X1,Y1));
        END;
        figBezier:                                                     //draw a bezier spline
        BEGIN
             IF ((abs(X1-X)<2)OR(abs(Y1-Y)<2)) THEN exit;
             aptl[0].X:=X;
             aptl[0].Y:=Y;
             aptl[1].X:=X1-X;
             aptl[1].Y:=Y1-Y;
             aptl[2].X:=X1;
             aptl[2].Y:=Y1;
             Canvas.DrawBezierSpline(X,Y,aptl);
        END;
        figFreeHand:                                                   //draw freehand
        BEGIN
             Canvas.DrawLine(X,Y,X1,Y1);
        END;
     END; {case}
END;

//This procedure draws a figure. If rubberband is TRUE only the surface
//of the figure is drawed in inverted mode
//this routine calls DrawElement to draw a specific element
//If drawing is complete the figure is added to the draw list.
PROCEDURE TDrawApp.DrawFigure(RubberBand:BOOLEAN;ptEnd:TPoint);
VAR dummy:PDrawList;
BEGIN
     IF ((RubberBand)AND(FigureType<>FigFreeHand)) THEN
     BEGIN
          //draw the element only in inverted state (rubberband)
          Canvas.FgMode:=fgInvert;
          Canvas.Color:=clBlack;
          DrawElement(StartPoint.X,StartPoint.Y,EndPoint.X,EndPoint.Y,TRUE);
          EndPoint:=ptEnd;
          DrawElement(StartPoint.X,StartPoint.Y,EndPoint.X,EndPoint.Y,TRUE);
          Canvas.FgMode:=fgOverPaint;
     END
     ELSE
     BEGIN
          Canvas.FgMode:=fgOverpaint;
          Canvas.Color:=Color;
          EndPoint:=ptEnd;
          DrawElement(StartPoint.X,StartPoint.Y,EndPoint.X,EndPoint.Y,FALSE);

          //update list of elements
          IF DrawList=NIL THEN
          BEGIN
               new(DrawList);
               dummy:=DrawList;
          END
          ELSE
          BEGIN
               New(LastDrawItem^.Next);
               dummy:=LastDrawItem^.Next;
          END;
          dummy^.Next:=NIL;
          IF FigureType=figFreeHand THEN dummy^.FigureType:=figLine
          ELSE dummy^.FigureType:=FigureType;
          dummy^.StartPoint:=StartPoint;
          dummy^.EndPoint:=EndPoint;
          dummy^.Color:=Color;
          dummy^.FillInterior:=FillInterior;
          LastDrawItem:=dummy;
     END;
END;

//This event handler is called whenever the user moves the mouse across the window.
//It will update the window contents by drawing the selected figure. This routine
//will only perform action if the left mouse button is pressed while moving the mouse
//(DrawMode becomes TRUE then)
PROCEDURE TDrawApp.DrawAppOnMouseMove(Sender:TComponent;VAR Msg:TMessage;VAR pt:TPoint;KeyState:TMouseKeyState);
BEGIN
     IF ((DrawMode)AND(not FirstPoint)) THEN
     BEGIN
          DrawFigure(TRUE,pt);
          IF FigureType=figFreeHand THEN StartPoint:=EndPoint;
     END
     ELSE EndPoint:=pt;

     FirstPoint:=FALSE;
END;

//This event handler is called whenever the user releases the left mouse button. It will finally
//paint the figure in its final shape.
PROCEDURE TDrawApp.DrawAppOnLMouseUp(Sender:TComponent;VAR Msg:TMessage;VAR pt:TPoint;KeyState:TMouseKeyState);
BEGIN
     IF DrawMode THEN
     BEGIN
          DrawFigure(FALSE,pt);
          Capture(FALSE);
          DrawMode:=FALSE;
          FirstPoint:=FALSE;
     END;
END;

//This event handler is called whenever the user pressed the left mouse button. It will set
//"DrawMode" to TRUE and initializes the first point for drawing. Subsequent calls to
//"DrawAppOnMouseMove" and "DrawAppOnLMouseUp" will update window contents
PROCEDURE TDrawApp.DrawAppOnLMouseDown(Sender:TComponent;VAR Msg:TMessage;VAR pt:TPoint;KeyState:TMouseKeyState);
BEGIN
     StartPoint:=pt;
     DrawMode:=TRUE;
     FirstPoint:=TRUE;
     Capture(TRUE);
END;

//unit initialization. Register the class types used by this unit.
//Registration becomes important if you add the unit to the Sibyl environment and thus
//extending the components available within the Sibyl component palette.
BEGIN
     RegisterComponents([TDrawApp,TToolBar1,TSwitchButton,
                         TColorDialog,TValueSet]);
END.
