--************************************************************************
--
--  CDROMS.ADB
--
--  A copyright-reserved, free use program.
--  (c)John H. McCoy, 1994, Sam Houston St. Univ., TX 77341-2206
--************************************************************************

with system;
with text_io; use text_io;

package body CDRoms is

--ioctl out
Ioctl_EjectCD        : IoctlOutSubCommand := 0;
Ioctl_LockUnlockCD   : IoctlOutSubCommand := 1;

--ioctl in
Ioctl_GetCDStatus   : IoctlOutSubCommand := 6;

type IoCB_EjectCD is
   record
      IoctlCode      : IoctlOutSubCommand := Ioctl_EjectCD;
   end record;

type IoCB_LockCD is
   record
      IoctlCode      : IoctlOutSubCommand := Ioctl_LockUnlockCD;
      LockCode       : byte := 1;
   end record;

type IoCB_UnlockCD is
   record
      IoctlCode      : IoctlOutSubCommand := Ioctl_LockUnlockCD;
      UnlockCode     : byte := 0;
   end record;

type IoCB_GetCDStatus is
   record
      IoctlCode      : IoctlOutSubCommand := Ioctl_GetCDStatus;
      Status         : DW := long_to_DW(0);
   end record;

type CDDriverEntries is
  record
    Unit        : byte;
    DriverName  : string8;
    Strategy    : system.address;
    Interrupt   : system.address;
  end record;

type CDDriverArray is array (integer range <>) of CDDriverEntries;
type CDDriverArrayAccess is access CDDriverArray;

rh             : rhs;
CB_EjectCD     : IoCB_EjectCD;
CB_LockCD      : IoCB_LockCD;
CB_UnlockCD    : IoCB_UnlockCD;
CB_GetCDStatus : IoCB_GetCDStatus;

task body CDRoms is
  Stop            : boolean := False;
  CDs             : CDDriverArrayAccess;
  MaxCDs          : integer;
  CDIndex         : integer;
  LastCD          : integer := -1;
  DriverHandle    : integer;
  DriverStrategy  : system.address;
  DriverInterrupt : system.address;
  DriverSubUnits  : byte;
  pkt             : pkts;
  dta             : bytesAccess;

  procedure CallCDDriver (rh: in out rhs) is
  begin
    CDIndex             := integer(rh.SubUnit);
    if CDIndex > LastCD then
      -- the following is a kludge
      -- bytes 5 & 6 in rh are device return code(status) for all subcommands
      pkt    := Rhs_to_Pkts(rh);  -- just convert dont't shift
      pkt(5..6) := W(DeviceError OR DeviceDone OR DeviceUnknownUnit);
      rh     := Pkts_to_Rhs(pkt);
    else
      rh.SubUnit          := CDs(CDIndex).Unit;
      pkt                 := Rhs_to_Pkts(rh);
      pkt(3..pkts'last-1) := pkt(4..pkts'last);
      CallDriver (rh              => pkt(1)'address,
                  DeviceStrategy  => CDs(CDIndex).Strategy,
                  DeviceInterrupt => CDs(CDIndex).Interrupt);
      pkt(4..pkts'last)   := pkt(3..pkts'last-1);
      pkt(3)              := 0;
      rh                  := Pkts_to_Rhs(pkt);
      rh.SubUnit          := byte(CDIndex);
    end if;
  end CallCDDriver;

begin
  accept SetUp(MaxDrives:integer) do
    MaxCDs := MaxDrives;
    CDs    := new CDDriverArray (0..MaxCDs-1);
  end SetUp;
loop
  select
    accept InitDriver(DriverName: string8; Units: out integer) do
      begin
        OpenDevice ( DeviceName => DriverName,
                     Handle     => DriverHandle);
        GetDeviceEntryAddresses (Handle           => DriverHandle,
                                 DeviceStrategy   => DriverStrategy,
                                 DeviceInterrupt  => DriverInterrupt,
                                 SubUnits         => DriverSubUnits );
        if (MaxCDs-LastCd-1) < integer(DriverSubUnits) then
          DriverSubUnits := byte(MaxCDs-LastCd-1);
        end if;
        for j in 0..DriverSubUnits-1 loop
          LastCD                := LastCD+1;
          CDs(LastCD).Unit      := j;
          CDs(LastCD).DriverName:= DriverName;
          CDs(LastCD).Strategy  := DriverStrategy;
          CDs(LastCD).Interrupt := DriverInterrupt;
        end loop;
        CloseDevice (Handle => DriverHandle);
        Units := integer(DriverSubUnits);
      exception
        when DEV_Error => Units := 0;
      end;
    end InitDriver;
  or
    accept Call (rh: in out rhs) do
      CallCDDriver(rh);
    end Call;
  or
    accept Lock(Drive:integer) do
      CDIndex := Drive;
      rh := ( Length  => 2+rhXs'size/8,
              SubUnit => byte(Drive),
              rhX     => (DeviceIoctlOutput,
                         (Status        => DeviceDone,
                          reserved      => (others =>0),
                          MediaDesc     => 0,
                          CBPtr         => SA_to_DW(CB_LockCD'address),
                          TransferCount => word_to_W(2),
                          Start         => word_to_W(0),
                          VolIdPtr      => SA_to_DW(0) )) );
      CallCDDriver(rh);
    end Lock;
  or
    accept UnLock(Drive:integer) do
      rh := ( Length  => 2+rhXs'size/8,
              SubUnit => byte(Drive),
              rhX     => (DeviceIoctlOutput,
                         (Status        => DeviceDone,
                          reserved      => (others =>0),
                          MediaDesc     => 0,
                          CBPtr         => SA_to_DW(CB_UnLockCD'address),
                          TransferCount => word_to_W(2),
                          Start         => word_to_W(0),
                          VolIdPtr      => SA_to_DW(0) )) );
      CallCDDriver(rh);
    end UnLock;
  or
    accept GetStatus(Drive       : integer;
                     DeviceStatus: out DEV_ReturnCodes;
                     CDStatus    : out DW;
                     DriverName  : out string8;
                     DriverUnit  : out byte;
                     Label       : out string11) do

      rh := ( Length  => 2+rhXs'size/8,
              SubUnit => byte(Drive),
              rhX     => (DeviceIoctlInput,
                         (Status        => DeviceDone,
                          reserved      => (others =>0),
                          MediaDesc     => 0,
                          CBPtr         => SA_to_DW(CB_GetCDStatus'address),
                          TransferCount => word_to_W(5),
                          Start         => word_to_W(0),
                          VolIdPtr      => SA_to_DW(0) )) );
      CallCDDriver(rh);
      if rh.rhX.IoctlIn.status = DeviceDone then
        CDStatus := CB_GetCDStatus.status;
        if (CB_GetCDStatus.status(1) AND byte(1) ) = 0 then  -- door closed
          dta := new bytes(1..2048);
          rh := (Length  => 2+rhXs'size/8,
                 SubUnit => byte(Drive),
                 rhX     => (DeviceReadLong,
                            (Status          => DeviceDone,
                             reserved        => (others =>0),
                             AddressMode     => 0,
                             DtaPtr          => SA_to_DW(dta.all(1)'address),
                             SectorsToRead   => word_to_W(1),
                             StartSector     => Long_to_DW(long_integer(16)),
                             ReadMode        => 0,
                             InterleaveSize  => 0,
                             InterleaveSkip  => 0,
                             filler          => (0,0))));
          CallCDDriver(rh);
          if rh.rhX.ReadLong.status = DeviceDone then
            if dta(2..6) = string_to_bytes("CD001") then
              Label := bytes_to_string(dta(41..51));
            elsif dta(10..14) = string_to_bytes("CDROM") then
              Label := bytes_to_string(dta(49..59));
            else
              Label := ("           ");
            end if;
          end if;
          ZapBytes(dta);
          DeviceStatus := rh.rhX.ReadLong.status;
        end if;
      else
        DeviceStatus := rh.rhX.IoctlIn.status;
      end if;
      DriverName := CDs(Drive).DriverName;
      DriverUnit := CDs(Drive).Unit;
     end GetStatus;
  or
    accept ShutDown do
      Stop := True;
    end ShutDown;
  end select;
  exit when Stop;
end loop;
end CDRoms;

end CDRoms;