//***************************************************************************
//
//  PROGRAM   : SLDEMO2.CPP
//
//  PROGRAMMER: Steven R Clabaugh
//              SRC Enterprises
//
//  PURPOSE   : SRC Enterprises Slider Custom Control Class for
//              C++ programing.
//
// Note that when a Slider Control is created with the CONTROL statement
// in dialogs, it is not possible to use the Slider Class.  You can use the
// Slider Class when creating the Slider directly in the dialog's
// WM_INITDIALOG message handler.  This is demonstrated in the SLBankDialog6
// function.  Dialogs 1-5 are identical to the dialogs in the SLDEMO1.C
// program.
// 
//***************************************************************************
#include <windows.h>
#include <srcentsl.h>
#include "sldemo2.h"

///**************************************************************************
//***************************************************************************
//
//      Global Variables
//
//***************************************************************************
//***************************************************************************
HINSTANCE hInst;  // current instance

char   MainWndClassName[] = "SLDemo2WClass",
       MainWndMenu[]      = "SLDemo2Menu",
       ProgTitle[]        = "SLDemo2 - Slider Sample Application";

LPSTR cBuff  = "         ";

//***************************************************************************
//
//  FUNCTION: WinMain(HINSTANCE, HINSTANCE, LPSTR, int)
//
//  PURPOSE: calls initialization function, processes message loop
//
//***************************************************************************
int PASCAL WinMain(HINSTANCE hInstance,     // current instance
                   HINSTANCE hPrevInstance, // previous instance
                   LPSTR  lpCmdLine,        // command line
                   int    nCmdShow)         // show-window type
{

   MSG msg;   // message

   //****************************************************************

   if (!hPrevInstance)                  // Other instances of app running?
       if (!InitApplication(hInstance)) // Initialize shared things
           return (FALSE);              // Exits if unable to initialize

   // Perform initializations that apply to a specific instance
   if (!InitInstance(hInstance, nCmdShow))
       return (FALSE);

   // Acquire and dispatch messages until a WM_QUIT message is received.
   while (GetMessage(&msg,       // message structure
           NULL,                 // handle of window receiving the message
           NULL,                 // lowest message to examine
           NULL))                // highest message to examine
       {
       TranslateMessage(&msg);   // Translates virtual key codes
       DispatchMessage(&msg);    // Dispatches message to window
   }

   UnregisterClass((LPSTR)MainWndClassName,hInst);

   return (msg.wParam);          // Returns the value from PostQuitMessage

} // end WinMain

//***************************************************************************
//
//  FUNCTION: InitApplication(HINSTANCE)
//
//  PURPOSE: Initializes window data and registers window class
//
//***************************************************************************
BOOL InitApplication(HINSTANCE hInstance)

{

   WNDCLASS  wc;

   //****************************************************************

   // Fill in window class structure with parameters that describe the
   // main window.

   wc.style = NULL;
   wc.lpfnWndProc = (WNDPROC)MainWndProc;
   wc.cbClsExtra = 0;
   wc.cbWndExtra = 0;
   wc.hInstance = hInstance;
   wc.hIcon = LoadIcon(hInstance,"progicon");
   wc.hCursor = LoadCursor(NULL, IDC_ARROW);
   wc.hbrBackground = GetStockObject(LTGRAY_BRUSH);
   wc.lpszMenuName =  MainWndMenu;
   wc.lpszClassName = MainWndClassName;

   // Register the window class and return success/failure code.
   return (RegisterClass(&wc));

}  // end InitApplication

//***************************************************************************
//
//  FUNCTION:  InitInstance(HINSTANCE, int)
//
//  PURPOSE:  Saves instance handle and creates main window
//
//***************************************************************************
BOOL InitInstance(HINSTANCE hInstance, // Current instance identifier.
                  int       nCmdShow)  // Param for first ShowWindow() call.
{

   HWND hWnd;  // Main window handle

   //****************************************************************

   // Save the instance handle
   hInst = hInstance;

   // Create a main window for this application instance.
   hWnd = CreateWindow(
       MainWndClassName,
       ProgTitle,
       WS_OVERLAPPEDWINDOW,
       10,
       10,
       320,
       220,
       NULL,
       NULL,
       hInstance,
       NULL
   );

   // If window could not be created, return "failure"
   if (!hWnd)
       return (FALSE);

   // Make the window visible; update its client area; and return "success"
   ShowWindow(hWnd, nCmdShow);  // Show the window
   UpdateWindow(hWnd);          // Sends WM_PAINT message
   return (TRUE);               // Returns the value from PostQuitMessage

}  // end InitInstance

//***************************************************************************
//
//  FUNCTION: MainWndProc(HWND, unsigned, WORD, LONG)
//
//  PURPOSE:  Processes messages
//
//***************************************************************************
long FAR PASCAL __export MainWndProc(
                     HWND     hWnd,    // window handle
                     unsigned message, // type of message
                     WORD     wParam,  // additional information
                     LONG     lParam)  // additional information
{

   static HWND hSLEdit1, hSLEdit2, hSLEdit3, hSLEdit4;

   // Declare instances of Slider Class                        
   static CSRCEntSL Slider1, Slider2, Slider3, Slider4;

   //****************************************************************

   switch (message)
   {

      case WM_CREATE:

         // The four Sliders in the main window are created
         // using the Slider Class functions.

         // Create Slider1 on the Main windlow client area.
         // This will be a vertical, NON-reversed slider w/hand cursor
         Slider1.Create(
            NULL,      // Vertical, Nonreversed Slider
            "slface1", // Resource name for the Slider Face
            "slknob1", // Resource name for the Slider Knob
            16,30,     // Slider X,Y position within the main client window
            6,         // Knob slide position
            6,         // Knob upper limit position
            99,        // Knob lower limit position
            0,100,     // range min & max
            0,         // set an initial value
            TRUE,      // Use the hand mouse cursor
            TRUE,      // Enable the mouse
            hWnd,      // Parent window handle
            SLIDER1,   // Slider ID value
            hInst);    // This Program's instance

         // EDIT control to display Slider1's value
         hSLEdit1 = CreateWindow(
            "EDIT",
            "0",
            WS_CHILD | WS_VISIBLE | WS_BORDER | WS_DISABLED,
            12,140,32,22,
            hWnd,
            SLEDIT1,
            hInst,
            NULL
         );
         wsprintf(cBuff,"%d",Slider1.GetVal());
         SetWindowText(hSLEdit1,cBuff);

         // Create Slider2 on the Main windlow client area.
         // This will be a vertical, reversed slider w/arrow cursor
         // Use the same face and knob as Slider 1
         Slider2.Create(
            SL_REVERSED,  // Vertical, reversed Slider
            "slface1",  // Resource name for the Slider Face
            "slknob1",  // Resource name for the Slider Knob
            81,30,      // Slider X,Y position within the main client window
            6,          // Knob slide position
            6,          // Knob upper limit position
            99,         // Knob lower limit position
            0,100,      // range min & max
            0,          // set an initial value
            FALSE,      // Use the arrow mouse cursor
            TRUE,       // Enable the mouse
            hWnd,       // Parent window handle
            SLIDER2,    // Slider ID value
            hInst);     // This Program's instance

         // EDIT control to display Slider2's value
         hSLEdit2 = CreateWindow(
            "EDIT",
            "0",
            WS_CHILD | WS_VISIBLE | WS_BORDER | WS_DISABLED,
            77,140,32,22,
            hWnd,
            SLEDIT2,
            hInst,
            NULL
         );
         wsprintf(cBuff,"%d",Slider2.GetVal());
         SetWindowText(hSLEdit2,cBuff);

         // Create Slider3 on the Main windlow client area.
         // This will be a horizontal, NON-reversed slider w/hand cursor
         Slider3.Create(
            SL_HORZ,   // Horizontal, Nonreversed Slider
            "slface2", // Resource name for the Slider Face
            "slknob2", // Resource name for the Slider Knob
            140,50,    // Slider X,Y position within the main client window
            6,         // Knob slide position
            5,         // Knob upper limit position
            60,        // Knob lower limit position
            0,100,     // range min & max
            0,         // set an initial value
            TRUE,      // Use the hand mouse cursor
            TRUE,      // Enable the mouse
            hWnd,      // Parent window handle
            SLIDER3,   // Slider ID value
            hInst);    // This Program's instance

         // EDIT control to display Slider3's value
         hSLEdit3 = CreateWindow(
            "EDIT",
            "0",
            WS_CHILD | WS_VISIBLE | WS_BORDER | WS_DISABLED,
            210,51,32,22,
            hWnd,
            SLEDIT3,
            hInst,
            NULL
         );
         wsprintf(cBuff,"%d",Slider3.GetVal());
         SetWindowText(hSLEdit3,cBuff);

         // Create Slider4 on the Main windlow client area.
         // This will be a horizontal, NON-reversed slider w/arrow cursor
         // Use the same face and knob as Slider 3
         Slider4.Create(
            SL_HORZ | SL_REVERSED ,   // Horizontal, reversed Slider
            "slface2", // Resource name for the Slider Face
            "slknob2", // Resource name for the Slider Knob
            140,110,   // Slider X,Y position within the main client window
            6,         // Knob slide position
            5,         // Knob upper limit position
            60,        // Knob lower limit position
            0,100,     // range min & max
            0,         // set an initial value
            FALSE,     // Use the hand mouse cursor
            TRUE,      // Enable the mouse
            hWnd,      // Parent window handle
            SLIDER4,   // Slider ID value
            hInst);    // This Program's instance

         // EDIT control to display Slider3's value
         hSLEdit4 = CreateWindow(
            "EDIT",
            "0",
            WS_CHILD | WS_VISIBLE | WS_BORDER | WS_DISABLED,
            210,111,32,22,
            hWnd,
            SLEDIT4,
            hInst,
            NULL
         );
         wsprintf(cBuff,"%d",Slider4.GetVal());
         SetWindowText(hSLEdit4,cBuff);

      break;  // end WM_CREATE

      case WM_PAINT:
      {

         PAINTSTRUCT ps;

         //**********************************************************

         BeginPaint(hWnd,&ps);

         SetBkColor(ps.hdc,RGB(192,192,192));
         TextOut(ps.hdc,5,5,"Slider 1",8);
         TextOut(ps.hdc,70,5,"Slider 2",8);
         TextOut(ps.hdc,150,30,"Slider 3",8);
         TextOut(ps.hdc,150,90,"Slider 4",8);

         EndPaint(hWnd,&ps);

      }
      break;  // end WM_PAINT

      case WM_COMMAND:

         switch (wParam)
         {


            // Here we handle the messages sent from Slider1
            case SLIDER1:
               wsprintf(cBuff,"%d",Slider1.GetVal());
               SetWindowText(hSLEdit1,cBuff);
            break;

            // Here we handle the messages sent from Slider2
            case SLIDER2:
               wsprintf(cBuff,"%d",Slider2.GetVal());
               SetWindowText(hSLEdit2,cBuff);
            break;

            // Here we handle the messages sent from Slider3
            case SLIDER3:
               wsprintf(cBuff,"%d",Slider3.GetVal());
               SetWindowText(hSLEdit3,cBuff);
            break;

            // Here we handle the messages sent from Slider4
            case SLIDER4:
               wsprintf(cBuff,"%d",Slider4.GetVal());
               SetWindowText(hSLEdit4,cBuff);
            break;


            case IDM_ABOUT:
            {

               FARPROC lpProcAbout; // pointer to the "About" function

               //****************************************************

               lpProcAbout = MakeProcInstance((FARPROC)About, hInst);

               DialogBox(hInst,
                         "AboutBox",
                         hWnd,
                         lpProcAbout);

               FreeProcInstance(lpProcAbout);
            }
            break;

            case IDM_VGAUGE:
            {

               FARPROC lpProcAbout; // pointer dialog function

               //****************************************************

               lpProcAbout = MakeProcInstance((FARPROC)VGaugeDialog1, hInst);

               DialogBox(hInst,
                         "VGaugeDLG1",
                         hWnd,
                         lpProcAbout);

               FreeProcInstance(lpProcAbout);
            }
            break;

            case IDM_HGAUGE:
            {

               FARPROC lpProcAbout; // pointer dialog function

               //****************************************************

               lpProcAbout = MakeProcInstance((FARPROC)HGaugeDialog2, hInst);

               DialogBox(hInst,
                         "HGaugeDLG2",
                         hWnd,
                         lpProcAbout);

               FreeProcInstance(lpProcAbout);
            }
            break;

            case IDM_VTHERM:
            {

               FARPROC lpProcAbout; // pointer dialog function

               //****************************************************

               lpProcAbout = MakeProcInstance((FARPROC)VThermDialog3, hInst);

               DialogBox(hInst,
                         "VThermDLG3",
                         hWnd,
                         lpProcAbout);

               FreeProcInstance(lpProcAbout);
            }
            break;

            case IDM_HTHERM:
            {

               FARPROC lpProcAbout; // pointer dialog function

               //****************************************************

               lpProcAbout = MakeProcInstance((FARPROC)HThermDialog4, hInst);

               DialogBox(hInst,
                         "HThermDLG4",
                         hWnd,
                         lpProcAbout);

               FreeProcInstance(lpProcAbout);
            }
            break;

            case IDM_RVTHERM:
            {

               FARPROC lpProcAbout; // pointer dialog function

               //****************************************************

               lpProcAbout = MakeProcInstance((FARPROC)RVThermDialog5, hInst);

               DialogBox(hInst,
                         "RVThermDLG5",
                         hWnd,
                         lpProcAbout);

               FreeProcInstance(lpProcAbout);
            }
            break;

            case IDM_SLBANK:
            {

               FARPROC lpProcAbout; // pointer dialog function

               //****************************************************

               lpProcAbout = MakeProcInstance((FARPROC)SLBankDialog6, hInst);

               DialogBox(hInst,
                         "SLBankDLG6",
                         hWnd,
                         lpProcAbout);

               FreeProcInstance(lpProcAbout);
            }
            break;


         } // end switch(wParam)

      break; // end WM_COMMAND

      case WM_DESTROY:                  // message: window being destroyed
         PostQuitMessage(0);
      break;

      default:   // Passes it on if unproccessed
         return (DefWindowProc(hWnd, message, wParam, lParam));

   }

   return (NULL);

} // end MainWndProc

//***************************************************************************
//
//  FUNCTION: About(HWND, unsigned, WORD, LONG)
//
//  PURPOSE:  "About" dialog box
//
//***************************************************************************
BOOL FAR PASCAL __export About(HWND     hDlg,
                               unsigned message,
                               WORD     wParam,
                               LONG     lParam)
{

   BOOL bRC;

   //****************************************************************

   bRC = TRUE;

   switch (message)
   {

      case WM_INITDIALOG:
      break;

      case WM_COMMAND:
         switch (wParam)
         {

            case IDOK:
            case IDCANCEL:
               EndDialog(hDlg,TRUE);
            break;

         } // end switch(wParam)

      break;

      default:
         bRC = FALSE;  // Didn't process a message

   } // end switch(message)

   return (bRC);

}  // end About

//***************************************************************************
//
//  FUNCTION: VGaugeDialog1(HWND, unsigned, WORD, LONG)
//            Vertical Gauge
//
//***************************************************************************
BOOL FAR PASCAL __export VGaugeDialog1(HWND     hDlg,
                                       unsigned message,
                                       WORD     wParam,
                                       LONG     lParam)
{

   BOOL bRC;

   //****************************************************************

   bRC = TRUE;

   switch (message)
   {

      case WM_INITDIALOG:

         // Initialize the slider
         SRCEntSLInit(
            GetDlgItem(hDlg,D1SLIDER), // The Slider window handle
            "slface3", // Resource name for the Slider Face
            "slknob3", // Resource name for the Slider Knob
            6,         // Knob slide position
            5,         // Knob upper limit position
            60,        // Knob lower limit position
            0,100,     // range min & max
            0,         // set an initial value
            TRUE,      // Use the hand mouse cursor
            TRUE       // Enable the mouse
         );

         // Initialize the slider for use as a guage.
         // Since this slider is used as gauge, there is no need
         // to have an enabled mouse cursor.
         SRCEntSLInit(
            GetDlgItem(hDlg,D1GAUGE), // The Slider window handle
            "gauge1f",  // Resource name for the Slider Face
            "gauge1k",  // Resource name for the Slider Knob
            2,          // Knob slide position
            3,          // Knob upper limit position
            111,        // Knob lower limit position
            0,100,      // range min & max
            0,          // set an initial value
            FALSE,      // To use as a gauge; Use the mouse cursor
            FALSE       // No need for an enabled mouse for gauges
         );

      break; // end WM_INITDIALOG

      case WM_COMMAND:
         switch (wParam)
         {

            case D1SLIDER:  // Handle the Slider Message

               // Use the Slider value to set the Gauge Value
               SRCEntSLSetVal(GetDlgItem(hDlg,D1GAUGE),
                              SRCEntSLGetVal(GetDlgItem(hDlg,D1SLIDER)),
                              TRUE);
            break;

            case IDOK:
            case IDCANCEL:
               EndDialog(hDlg,TRUE);
            break;

         } // end switch(wParam)

      break; // end WM_COMMAND

      case WM_PAINT:
      {
         PAINTSTRUCT ps;
         RECT rect;

         //**********************************************************

         BeginPaint(hDlg,&ps);
         GetClientRect(hDlg,&rect);

         //**** Paint the client area Light Gray *****
         SelectObject(ps.hdc,GetStockObject(LTGRAY_BRUSH));
         PatBlt(ps.hdc,0,0,rect.right,rect.bottom,PATCOPY);

         EndPaint(hDlg,&ps);

      };
      break; // end WM_PAINT

      default:

         bRC = FALSE; // Didn't process a message

   } // end switch(message)

   return (bRC);

}  // end VGaugeDialog1

//***************************************************************************
//
//  FUNCTION: HGaugeDialog2(HWND, unsigned, WORD, LONG)
//            Horizontal Gauge
//
//***************************************************************************
BOOL FAR PASCAL __export HGaugeDialog2(HWND     hDlg,
                                       unsigned message,
                                       WORD     wParam,
                                       LONG     lParam)
{

   BOOL bRC;

   //****************************************************************

   bRC = TRUE;

   switch (message)
   {

      case WM_INITDIALOG:

         // Initialize the slider
         SRCEntSLInit(
            GetDlgItem(hDlg,D2SLIDER), // The Slider window handle
            "slface2", // Resource name for the Slider Face
            "slknob2", // Resource name for the Slider Knob
            6,         // Knob slide position
            5,         // Knob upper limit position
            60,        // Knob lower limit position
            0,100,     // range min & max
            0,         // set an initial value
            TRUE,      // Use the hand mouse cursor
            TRUE       // Enable the mouse
         );

         // Initialize the slider for use as a guage.
         // Since this slider is used as gauge, there is no need
         // to have an enabled mouse cursor.
         SRCEntSLInit(
            GetDlgItem(hDlg,D2GAUGE), // The Slider window handle
            "gauge2f",  // Resource name for the Slider Face
            "gauge2k",  // Resource name for the Slider Knob
            20,         // Knob slide position
            8,          // Knob upper limit position
            109,        // Knob lower limit position
            0,100,      // range min & max
            0,          // set an initial value
            FALSE,      // To use as a gauge; Use the mouse cursor
            FALSE       // No need for an enabled mouse for gauges
         );

      break; // end WM_INITDIALOG

      case WM_COMMAND:
         switch (wParam)
         {

            case D2SLIDER:  // Handle the Slider Message

               // Use the Slider value to set the Gauge Value
               SRCEntSLSetVal(GetDlgItem(hDlg,D2GAUGE),
                              SRCEntSLGetVal(GetDlgItem(hDlg,D2SLIDER)),
                              TRUE);
            break;

            case IDOK:
            case IDCANCEL:
               EndDialog(hDlg,TRUE);
            break;

         } // end switch(wParam)

      break; // end WM_COMMAND

      case WM_PAINT:
      {
         PAINTSTRUCT ps;
         RECT rect;

         //**********************************************************

         BeginPaint(hDlg,&ps);
         GetClientRect(hDlg,&rect);

         //**** Paint the client area Light Gray *****
         SelectObject(ps.hdc,GetStockObject(LTGRAY_BRUSH));
         PatBlt(ps.hdc,0,0,rect.right,rect.bottom,PATCOPY);

         EndPaint(hDlg,&ps);

      };
      break; // end WM_PAINT

      default:
         bRC = FALSE; // Didn't process a message

   } // end switch(message)

   return (bRC);

}  // end HGaugeDialog2

//***************************************************************************
//
//  FUNCTION: VThermDialog3(HWND, unsigned, WORD, LONG)
//            Vertical Thermometer Style Guage
//
//***************************************************************************
BOOL FAR PASCAL __export VThermDialog3(HWND     hDlg,
                                       unsigned message,
                                       WORD     wParam,
                                       LONG     lParam)
{

   BOOL bRC;

   //****************************************************************

   bRC = TRUE;

   switch (message)
   {

      case WM_INITDIALOG:

         // Initialize the slider
         SRCEntSLInit(
            GetDlgItem(hDlg,D3SLIDER), // The Slider window handle
            "slface3", // Resource name for the Slider Face
            "slknob3", // Resource name for the Slider Knob
            6,         // Knob slide position
            5,         // Knob upper limit position
            60,        // Knob lower limit position
            0,100,     // range min & max
            0,         // set an initial value
            TRUE,      // Use the hand mouse cursor
            TRUE       // Enable the mouse
         );

         // Initialize the slider for use as a thermometer guage.
         // Since this slider is used as gauge, there is no need
         // to have an enabled mouse cursor.
         SRCEntSLInit(
            GetDlgItem(hDlg,D3THERM), // The Slider window handle
            "therm5f",  // Resource name for the Slider Face
            "therm5k",  // Resource name for the Slider Knob
            5,          // Knob slide position
            7,          // Knob upper limit position
            107,        // Knob lower limit position
            0,100,      // range min & max
            0,          // set an initial value
            FALSE,      // To use as a gauge; Use the mouse cursor
            FALSE       // No need for an enabled mouse for gauges
         );

      break; // end WM_INITDIALOG

      case WM_COMMAND:
         switch (wParam)
         {

            case D3SLIDER:  // Handle the Slider Message

               // Use the Slider value to set the Gauge Value
               SRCEntSLSetVal(GetDlgItem(hDlg,D3THERM),
                              SRCEntSLGetVal(GetDlgItem(hDlg,D3SLIDER)),
                              TRUE);
            break;

            case IDOK:
            case IDCANCEL:
               EndDialog(hDlg,TRUE);
            break;

         } // end switch(wParam)

      break; // end WM_COMMAND

      case WM_PAINT:
      {
         PAINTSTRUCT ps;
         RECT rect;

         //**********************************************************

         BeginPaint(hDlg,&ps);
         GetClientRect(hDlg,&rect);

         //**** Paint the client area Light Gray *****
         SelectObject(ps.hdc,GetStockObject(LTGRAY_BRUSH));
         PatBlt(ps.hdc,0,0,rect.right,rect.bottom,PATCOPY);

         EndPaint(hDlg,&ps);

      };
      break; // end WM_PAINT

      default:
         bRC = FALSE; // Didn't process a message

   } // end switch(message)

   return (bRC);

}  // end VThermDialog3

//***************************************************************************
//
//  FUNCTION: HThermDialog4(HWND, unsigned, WORD, LONG)
//            Horizontal Thermometer Style Gauge
//
//***************************************************************************
BOOL FAR PASCAL __export HThermDialog4(HWND     hDlg,
                                       unsigned message,
                                       WORD     wParam,
                                       LONG     lParam)
{

   BOOL bRC;

   //****************************************************************

   bRC = TRUE;

   switch (message)
   {

      case WM_INITDIALOG:

         // Initialize the slider
         SRCEntSLInit(
            GetDlgItem(hDlg,D4SLIDER), // The Slider window handle
            "slface2", // Resource name for the Slider Face
            "slknob2", // Resource name for the Slider Knob
            6,         // Knob slide position
            5,         // Knob upper limit position
            60,        // Knob lower limit position
            0,100,     // range min & max
            0,         // set an initial value
            TRUE,      // Use the hand mouse cursor
            TRUE       // Enable the mouse
         );

         // Initialize the slider for use as a thermometer guage.
         // Since this slider is used as gauge, there is no need
         // to have an enabled mouse cursor.
         SRCEntSLInit(
            GetDlgItem(hDlg,D4THERM), // The Slider window handle
            "therm1f",  // Resource name for the Slider Face
            "therm1k",  // Resource name for the Slider Knob
            20,         // Knob slide position
            9,          // Knob upper limit position
            109,        // Knob lower limit position
            0,100,      // range min & max
            0,          // set an initial value
            FALSE,      // To use as a gauge; Use the mouse cursor
            FALSE       // No need for an enabled mouse for gauges
         );

      break; // end WM_INITDIALOG

      case WM_COMMAND:
         switch (wParam)
         {

            case D4SLIDER:  // Handle the Slider Message

               // Use the Slider value to set the Gauge Value
               SRCEntSLSetVal(GetDlgItem(hDlg,D4THERM),
                              SRCEntSLGetVal(GetDlgItem(hDlg,D4SLIDER)),
                              TRUE);
            break;

            case IDOK:
            case IDCANCEL:
               EndDialog(hDlg,TRUE);
            break;

         } // end switch(wParam)

      break; // end WM_COMMAND

      case WM_PAINT:
      {
         PAINTSTRUCT ps;
         RECT rect;

         //**********************************************************

         BeginPaint(hDlg,&ps);
         GetClientRect(hDlg,&rect);

         //**** Paint the client area Light Gray *****
         SelectObject(ps.hdc,GetStockObject(LTGRAY_BRUSH));
         PatBlt(ps.hdc,0,0,rect.right,rect.bottom,PATCOPY);

         EndPaint(hDlg,&ps);

      };
      break; // end WM_PAINT

      default:
         bRC = FALSE; // Didn't process a message

   } // end switch(message)

   return (bRC);

}  // end HThermDialog4

//***************************************************************************
//
//  FUNCTION: RVThermDialog5(HWND, unsigned, WORD, LONG)
//            Reversed Vertical Thermometer Style Guage
//
//***************************************************************************
BOOL FAR PASCAL __export RVThermDialog5(HWND     hDlg,
                                        unsigned message,
                                        WORD     wParam,
                                        LONG     lParam)
{

   BOOL bRC;

   //****************************************************************

   bRC = TRUE;

   switch (message)
   {

      case WM_INITDIALOG:

         // Initialize the slider
         SRCEntSLInit(
            GetDlgItem(hDlg,D5SLIDER), // The Slider window handle
            "slface4", // Resource name for the Slider Face
            "slknob4", // Resource name for the Slider Knob
            6,         // Knob slide position
            5,         // Knob upper limit position
            109,       // Knob lower limit position
            0,100,     // range min & max
            0,         // set an initial value
            TRUE,      // Use the hand mouse cursor
            TRUE       // Enable the mouse
         );

         // Initialize the slider for use as a thermometer guage.
         // Since this slider is used as gauge, there is no need
         // to have an enabled mouse cursor.
         SRCEntSLInit(
            GetDlgItem(hDlg,D5THERM), // The Slider window handle
            "therm2f",  // Resource name for the Slider Face
            "therm2k",  // Resource name for the Slider Knob
            6,          // Knob slide position
            9,          // Knob upper limit position
            109,        // Knob lower limit position
            0,100,      // range min & max
            0,          // set an initial value
            FALSE,      // To use as a gauge; Use the mouse cursor
            FALSE       // No need for an enabled mouse for gauges
         );

      break; // end WM_INITDIALOG

      case WM_COMMAND:
         switch (wParam)
         {

            case D5SLIDER:  // Handle the Slider Message

               // Use the Slider value to set the Gauge Value
               SRCEntSLSetVal(GetDlgItem(hDlg,D5THERM),
                              SRCEntSLGetVal(GetDlgItem(hDlg,D5SLIDER)),
                              TRUE);
            break;

            case IDOK:
            case IDCANCEL:
               EndDialog(hDlg,TRUE);
            break;

         } // end switch(wParam)

      break; // end WM_COMMAND

      case WM_PAINT:
      {
         PAINTSTRUCT ps;
         RECT rect;

         //**********************************************************

         BeginPaint(hDlg,&ps);
         GetClientRect(hDlg,&rect);

         //**** Paint the client area Light Gray *****
         SelectObject(ps.hdc,GetStockObject(GRAY_BRUSH));
         PatBlt(ps.hdc,0,0,rect.right,rect.bottom,PATCOPY);

         EndPaint(hDlg,&ps);

      };
      break; // end WM_PAINT

      default:
         bRC = FALSE; // Didn't process a message

   } // end switch(message)

   return (bRC);

}  // end RVThermDialog5

//***************************************************************************
//
//  FUNCTION: SLBankDialog6(HWND, unsigned, WORD, LONG)
//            Bank of Slider2
//
//  We can use the Slider Class in this dialog since the Sliders are not
//  being created with CONTROL statements in the dialog's resource.
//
//***************************************************************************
BOOL FAR PASCAL __export SLBankDialog6(HWND     hDlg,
                                       unsigned message,
                                       WORD     wParam,
                                       LONG     lParam)

{

   BOOL bRC;
                            
   // Declare instances of Slider Class                        
   static CSRCEntSL Slider1, Slider2, Slider3, Slider4;

   //**********************************************************************

   bRC = TRUE;

   switch (message)
   {

      case WM_INITDIALOG:
      {

         HBITMAP hlface1, hmface1, hrface1;
         BITMAP  BMInfo;

         int rfw, mfw, lfw;    // width face bitmaps

         //***********************************************************

         // Before the Sliders are created, we need to load the face
         // bitmaps and get the their widths
         hlface1  = LoadBitmap(hInst,"lface1");
         hmface1  = LoadBitmap(hInst,"mface1");
         hrface1  = LoadBitmap(hInst,"rface1");

         // Get widths of face bitmaps
         GetObject(hlface1,sizeof(BITMAP),(LPSTR)&BMInfo);
         lfw = BMInfo.bmWidth;
         GetObject(hmface1,sizeof(BITMAP),(LPSTR)&BMInfo);
         mfw = BMInfo.bmWidth;
         GetObject(hrface1,sizeof(BITMAP),(LPSTR)&BMInfo);
         rfw = BMInfo.bmWidth;

         // Now Delete them
         DeleteObject(hlface1);
         DeleteObject(hmface1);
         DeleteObject(hrface1);

         // Now we create the sliders

         // The reason for using the Class Create() function to create
         // the slider windows rather than the CONTROL statement in the
         // resource file is because the CONTROL statement position and
         // size values use dialog box units, thus the sliders cannot be
         // positioned with pixel precision as is necessary here.

         // To make the slider bank appear as they do, there is a bitmap
         // for the left side Slider, the middle Sliders and the right
         // side Slider, then they are positioned to butt up next to
         // eack other.

         Slider1.Create(
            NULL,                   // Slider Style
            "lface1",               // Resource name for the Slider Face
            "knob2",                // Resource name for the Slider Knob
            SLIDERSX,               // Slider X Position
            SLIDERSY,               // Slider Y Position
            7,                      // Knob slide position
            5,                      // Knob upper limit position
            110,                    // Knob lower limit position
            0,100,                  // range min & max
            0,                      // set an initial value
            TRUE,                   // Use the hand mouse cursor
            TRUE,                   // Enable the mouse
            hDlg,                   // Parent window handle
            D6SLIDER1,              // Slider ID value
            hInst);                 // This Program's instance

         Slider2.Create(
            NULL,                   // Slider Style
            "mface1",               // Resource name for the Slider Face
            "knob2",                // Resource name for the Slider Knob
            SLIDERSX+lfw,           // Slider X Position
            SLIDERSY,               // Slider Y Position
            4,                      // Knob slide position
            5,                      // Knob upper limit position
            110,                    // Knob lower limit position
            0,100,                  // range min & max
            0,                      // set an initial value
            TRUE,                   // Use the hand mouse cursor
            TRUE,                   // Enable the mouse
            hDlg,                   // Parent window handle
            D6SLIDER2,              // Slider ID value
            hInst);                 // This Program's instance

         Slider3.Create(
            NULL,                   // Slider Style
            "mface1",               // Resource name for the Slider Face
            "knob2",                // Resource name for the Slider Knob
            SLIDERSX+lfw+mfw,       // Slider X Position
            SLIDERSY,               // Slider Y Position
            4,                      // Knob slide position
            5,                      // Knob upper limit position
            110,                    // Knob lower limit position
            0,100,                  // range min & max
            0,                      // set an initial value
            TRUE,                   // Use the hand mouse cursor
            TRUE,                   // Enable the mouse
            hDlg,                   // Parent window handle
            D6SLIDER3,              // Slider ID value
            hInst);                 // This Program's instance

         Slider4.Create(
            NULL,                   // Slider Style
            "rface1",               // Resource name for the Slider Face
            "knob2",                // Resource name for the Slider Knob
            SLIDERSX+lfw+(2*mfw),   // Slider X Position
            SLIDERSY,               // Slider Y Position
            4,                      // Knob slide position
            5,                      // Knob upper limit position
            110,                    // Knob lower limit position
            0,100,                  // range min & max
            0,                      // set an initial value
            TRUE,                   // Use the hand mouse cursor
            TRUE,                   // Enable the mouse
            hDlg,                   // Parent window handle
            D6SLIDER4,              // Slider ID value
            hInst);                 // This Program's instance


         // Init the EDIT controls that display the Slider values
         wsprintf(cBuff,"%d",Slider1.GetVal());
         SetDlgItemText(hDlg,D6EDIT1,cBuff);

         wsprintf(cBuff,"%d",Slider2.GetVal());
         SetDlgItemText(hDlg,D6EDIT2,cBuff);

         wsprintf(cBuff,"%d",Slider3.GetVal());
         SetDlgItemText(hDlg,D6EDIT3,cBuff);

         wsprintf(cBuff,"%d",Slider4.GetVal());
         SetDlgItemText(hDlg,D6EDIT4,cBuff);

      }
      break; // end WM_INITDIALOG

      case WM_COMMAND:
         switch (wParam)
         {

            case D6SLIDER1:
                 wsprintf(cBuff,"%d",Slider1.GetVal());
                 SetDlgItemText(hDlg,D6EDIT1,cBuff);
            break;

            case D6SLIDER2:
                 wsprintf(cBuff,"%d",Slider2.GetVal());
                 SetDlgItemText(hDlg,D6EDIT2,cBuff);
            break;

            case D6SLIDER3:
                 wsprintf(cBuff,"%d",Slider3.GetVal());
                 SetDlgItemText(hDlg,D6EDIT3,cBuff);
            break;

            case D6SLIDER4:
                 wsprintf(cBuff,"%d",Slider4.GetVal());
                 SetDlgItemText(hDlg,D6EDIT4,cBuff);
            break;

            case IDOK:
            case IDCANCEL:
               EndDialog(hDlg,TRUE);
            break;

         } // end switch(wParam)

      break; // end WM_COMMAND

      case WM_PAINT:
      {
         PAINTSTRUCT ps;
         RECT rect;

         //******************************************************

         BeginPaint(hDlg,&ps);
         GetClientRect(hDlg,&rect);

         //**** Paint the client area Light Gray
         SelectObject(ps.hdc,GetStockObject(LTGRAY_BRUSH));
         PatBlt(ps.hdc,0,0,rect.right,rect.bottom,PATCOPY);


         // Label the Sliders
         SetBkColor(ps.hdc,RGB(192,192,192));
         TextOut(ps.hdc,20,7,"1",1);
         TextOut(ps.hdc,44,7,"2",1);
         TextOut(ps.hdc,68,7,"3",1);
         TextOut(ps.hdc,92,7,"4",1);

         EndPaint(hDlg,&ps);

      };
      break; // end WM_PAINT

      default:
         bRC = FALSE; // Didn't process a message

   } // end switch(message)

   return (bRC);

}  // end SLBankDialog6
